/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.tooling.agent.RuntimeToolingService;

import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Handles a reference to the exploded application from file system.
 *
 * @since 4.0
 */
public class FileSystemArtifactDeployer extends AbstractArtifactDeployer {

  private Function<DeploymentContext, String> deployFunction;
  private Consumer<DeploymentContext> disposeFunction;

  public FileSystemArtifactDeployer(ArtifactResources artifactResources,
                                    Supplier<RuntimeToolingService> runtimeToolingService,
                                    Map<String, String> deploymentProperties) {
    super(artifactResources, runtimeToolingService, deploymentProperties);
    if (artifactResources.getArtifactType().equals(ArtifactType.DOMAIN)) {
      deployFunction =
          (deploymentContext) -> {
            if (deploymentContext.getLocalLocation() != null) {
              return deploymentContext.getRuntimeToolingService().deployDomain(deploymentContext.getArtifactId(),
                                                                               deploymentContext.getLocalLocation(),
                                                                               deploymentContext.getDeploymentProperties());
            }
            return deploymentContext.getRuntimeToolingService().deployDomain(deploymentContext.getArtifactId(),
                                                                             deploymentContext.getContent(),
                                                                             deploymentContext.getDeploymentProperties());
          };
      disposeFunction =
          (deploymentContext) -> deploymentContext.getRuntimeToolingService().disposeDomain(deploymentContext.getArtifactId());
    } else {
      deployFunction = (deploymentContext) -> {
        if (deploymentContext.getDomainId() != null) {
          if (deploymentContext.getLocalLocation() != null) {
            return deploymentContext.getRuntimeToolingService().deployApplication(deploymentContext.getArtifactId(),
                                                                                  deploymentContext.getLocalLocation(),
                                                                                  deploymentContext.getDomainId(),
                                                                                  deploymentContext.getDeploymentProperties());
          }
          return deploymentContext.getRuntimeToolingService().deployApplication(deploymentContext.getArtifactId(),
                                                                                deploymentContext.getContent(),
                                                                                deploymentContext.getDomainId(),
                                                                                deploymentContext.getDeploymentProperties());
        }
        if (deploymentContext.getLocalLocation() != null) {
          return deploymentContext.getRuntimeToolingService().deployApplication(deploymentContext.getArtifactId(),
                                                                                deploymentContext.getLocalLocation(),
                                                                                deploymentContext.getDeploymentProperties());
        }
        return deploymentContext.getRuntimeToolingService().deployApplication(deploymentContext.getArtifactId(),
                                                                              deploymentContext.getContent(),
                                                                              deploymentContext.getDeploymentProperties());
      };
      disposeFunction = (deploymentContext) -> deploymentContext.getRuntimeToolingService()
          .disposeApplication(deploymentContext.getArtifactId());
    }
  }

  @Override
  protected String doInternalDeploy(ArtifactResources artifactResources, RuntimeToolingService runtimeToolingService,
                                    String domainId, Map<String, String> deploymentProperties) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Deploying application from: {}", artifactResources);
    }

    return deployFunction.apply(new DeploymentContext(artifactResources.getArtifactId(), artifactResources.getRootArtifactFile(),
                                                      domainId, deploymentProperties,
                                                      runtimeToolingService));
  }

  @Override
  protected void doInternalDispose(String id, RuntimeToolingService runtimeToolingService) {
    disposeFunction.accept(new DeploymentContext(id, runtimeToolingService));
  }
}
