/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session;

import static java.lang.String.format;

import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.app.declaration.api.ConfigurationElementDeclaration;
import org.mule.runtime.app.declaration.api.ConnectionElementDeclaration;
import org.mule.runtime.app.declaration.api.ElementDeclaration;
import org.mule.runtime.core.internal.locator.ComponentLocator;
import org.mule.tooling.client.api.cache.CacheStorageSerializer;
import org.mule.tooling.client.api.declaration.session.DeclarationSessionCacheService;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.Command;
import org.mule.tooling.client.internal.metadata.ToolingCacheIdGenerator;
import org.mule.tooling.client.internal.serialization.Serializer;

import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ImmutableSet;

public class DefaultDeclarationSessionCacheService implements DeclarationSessionCacheService, Command {

  private final CacheStorageSerializer serializer;
  private final Serializer internalSerializer;

  private final ToolingCacheIdGenerator<ElementDeclaration> cacheIdGenerator;
  private final Map<String, ?> cacheStorage;
  private final ComponentLocator<ElementDeclaration> componentLocator;

  public DefaultDeclarationSessionCacheService(CacheStorageSerializer serializer,
                                               Serializer internalSerializer,
                                               ToolingCacheIdGenerator<ElementDeclaration> cacheIdGenerator,
                                               Map<String, ?> cacheStorage,
                                               ComponentLocator<ElementDeclaration> componentLocator) {
    this.serializer = serializer;
    this.internalSerializer = internalSerializer;
    this.cacheIdGenerator = cacheIdGenerator;
    this.cacheStorage = cacheStorage;
    this.componentLocator = componentLocator;
  }

  @Override
  public CacheStorageSerializer getSerializer() {
    return this.serializer;
  }

  @Override
  public void invalidateDependencies(String globalElementName) {
    if (cacheIsPresent()) {
      Optional<ElementDeclaration> configDeclaration =
          componentLocator.get(Location.builderFromStringRepresentation(globalElementName).build());
      if (!configDeclaration.isPresent()) {
        throw new ToolingException(format("Could not find global element with name: %s", globalElementName));
      }
      Optional<ConnectionElementDeclaration> connectionDeclaration =
          ((ConfigurationElementDeclaration) configDeclaration.get()).getConnection();

      ImmutableSet.Builder<String> parentIds = ImmutableSet.builder();
      cacheIdGenerator.getIdForGlobalMetadata(configDeclaration.get()).ifPresent(key -> parentIds.add(key));
      cacheIdGenerator.getIdForResolvedValuesDependency(configDeclaration.get()).ifPresent(key -> parentIds.add(key));
      connectionDeclaration.ifPresent(
                                      connectionElementDeclaration -> cacheIdGenerator
                                          .getIdForResolvedValuesDependency(connectionElementDeclaration)
                                          .ifPresent(key -> parentIds.add(key)));
      removeRelated(parentIds.build());
    }
    // Do nothing, no cache, nothing to invalidate.
  }

  private void removeRelated(Set<String> parentIds) {
    cacheStorage.keySet().removeIf(k -> parentIds.stream().filter(p -> k.contains(p)).findAny().isPresent());
  }

  private boolean cacheIsPresent() {
    return this.cacheStorage != null;
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "getSerializer": {
        return getSerializer();
      }
      case "invalidateDependencies": {
        validateInput(arguments, classes, String.class);
        invalidateDependencies(internalSerializer.deserialize(arguments[0]));
        return null;
      }
    }
    throw Command.methodNotFound(this.getClass(), methodName);
  }
}
