/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.cache;

import static java.lang.String.format;
import static org.mule.runtime.api.value.ResolvingFailure.Builder.newFailure;
import static org.mule.runtime.api.value.ValueResult.resultFrom;
import static org.mule.tooling.client.internal.utils.FunctionalUtils.executeHandling;
import org.mule.runtime.api.value.ValueResult;
import org.mule.runtime.app.declaration.api.ElementDeclaration;
import org.mule.tooling.client.internal.metadata.ToolingCacheIdGenerator;

import java.util.Map;
import java.util.concurrent.Callable;

public class DefaultDeclarationValueProviderCache implements DeclarationValueProviderCache {

  private final Map<String, ValueResult> storage;
  private final ToolingCacheIdGenerator<ElementDeclaration> cacheIdGenerator;

  public DefaultDeclarationValueProviderCache(ToolingCacheIdGenerator<ElementDeclaration> cacheIdGenerator,
                                              Map<String, ValueResult> storage) {
    this.cacheIdGenerator = cacheIdGenerator;
    this.storage = storage;
  }

  @Override
  public ValueResult getValues(ElementDeclaration elementDeclaration,
                               String parameterName, Callable<ValueResult> callable,
                               boolean forceResolution) {

    return cacheIdGenerator.getIdForResolvedValues(elementDeclaration, parameterName)
        .map(k -> {
          if (forceResolution) {
            storage.keySet().remove(k);
          }
          if (storage.containsKey(k)) {
            try {
              ValueResult result = storage.get(k);
              if (result != null) {
                return result;
              }
            } catch (Exception e) {
              // In case if the cache cannot deserialize the data when retrieved
            }
          }
          ValueResult result = executeHandling(callable, e -> resultFrom(newFailure(e).build()));
          if (result.isSuccess()) {
            storage.put(k, result);
          }
          return result;
        })
        .orElseThrow(() -> new IllegalArgumentException(format("Couldn't create a value provider cache id for component: %s, from extension: %s",
                                                               elementDeclaration.getName(),
                                                               elementDeclaration.getDeclaringExtension())));
  }

  @Override
  public ValueResult getValues(ElementDeclaration elementDeclaration, String parameterName, String targetSelector,
                               Callable<ValueResult> callable, boolean forceResolution) {
    return cacheIdGenerator.getIdForResolvedValues(elementDeclaration, parameterName, targetSelector)
        .map(k -> {
          if (forceResolution) {
            storage.keySet().remove(k);
          }
          if (storage.containsKey(k)) {
            try {
              ValueResult result = storage.get(k);
              if (result != null) {
                return result;
              }
            } catch (Exception e) {
              // In case if the cache cannot deserialize the data when retrieved
            }
          }
          ValueResult result = executeHandling(callable, e -> resultFrom(newFailure(e).build()));
          if (result.isSuccess()) {
            storage.put(k, result);
          }
          return result;
        })
        .orElseThrow(() -> new IllegalArgumentException(format("Couldn't create a field value provider cache id for component: %s, from extension: %s",
                                                               elementDeclaration.getName(),
                                                               elementDeclaration.getDeclaringExtension())));
  }
}
