/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.emptyMap;
import static java.util.function.UnaryOperator.identity;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.codehaus.plexus.util.FileUtils.toFile;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.module.artifact.api.classloader.RegionClassLoader;
import org.mule.runtime.module.deployment.impl.internal.artifact.ExtensionModelDiscoverer;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.junit.Rule;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnit;
import org.mockito.junit.MockitoRule;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature("DataSense")
@Story("ApplicationModelFactory should be able to create the ApplicationModel either from a remote location, from a local file system with jar or expanded content for application")
public class DefaultApplicationModelFactoryTestCase {

  @Rule
  public MockitoRule rule = MockitoJUnit.rule();

  @Mock
  private RegionClassLoader regionClassLoader;

  private final Set<ExtensionModel> runtimeExtensionModels = new ExtensionModelDiscoverer().discoverRuntimeExtensionModels();

  @Test
  @Description("Checks that an application model that uses CE components is created from an expanded folder")
  public void createApplicationModel() throws Exception {
    Pair<ApplicationDescriptor, File> applicationDescriptor = createApplicationDescriptor("applications/deploy-json-descriptor");

    assertApplicationModel(new DefaultApplicationModelFactory()
        .createApplicationModel(applicationDescriptor.getFirst(), runtimeExtensionModels,
                                new URLClassLoader(new URL[] {applicationDescriptor.getSecond().toURI().toURL()},
                                                   regionClassLoader),
                                ast -> identity(), emptyMap()));
  }

  @Test
  @Description("Checks that an application model that references properties can be resolved with the class loader")
  public void createApplicationModelWithPropertiesResolved() throws Exception {
    Pair<ApplicationDescriptor, File> applicationDescriptor = createApplicationDescriptor("applications/deploy-placeholders");
    assertApplicationModel(new DefaultApplicationModelFactory()
        .createApplicationModel(applicationDescriptor.getFirst(), runtimeExtensionModels,
                                new URLClassLoader(new URL[] {applicationDescriptor.getSecond().toURI().toURL()},
                                                   regionClassLoader),

                                ast -> identity(), emptyMap()));
  }

  private Pair<ApplicationDescriptor, File> createApplicationDescriptor(String applicationRelativeLocation) throws IOException {
    File applicationRootFile = toFile(this.getClass().getClassLoader().getResource(applicationRelativeLocation));
    final MuleApplicationModel muleApplicationModel = new MuleApplicationModelJsonSerializer().deserialize(
                                                                                                           readFileToString(new File(new File(new File(applicationRootFile,
                                                                                                                                                       "META-INF"),
                                                                                                                                              "mule-artifact"),
                                                                                                                                     "mule-artifact.json")));
    ApplicationDescriptor applicationDescriptor = mock(ApplicationDescriptor.class);
    when(applicationDescriptor.getConfigResources()).thenReturn(muleApplicationModel.getConfigs());
    return new Pair(applicationDescriptor, applicationRootFile);
  }

  @Test
  @Description("Checks that an application model that has properties that cannot be resolved is created")
  public void createApplicationModelWithPropertiesNotResolved() throws Exception {
    Pair<ApplicationDescriptor, File> applicationDescriptor = createApplicationDescriptor("applications/deploy-placeholders");

    assertApplicationModel(new DefaultApplicationModelFactory()
        .createApplicationModel(applicationDescriptor.getFirst(), runtimeExtensionModels,
                                new URLClassLoader(new URL[] {applicationDescriptor.getSecond().toURI().toURL()},
                                                   regionClassLoader),
                                ast -> identity(), emptyMap()));
  }

  @Test
  @Description("Checks that an application model that uses CE components is created from an expanded folder")
  public void createApplicationModelEE() throws Exception {
    Pair<ApplicationDescriptor, File> applicationDescriptor = createApplicationDescriptor("applications/datasense-static");

    assertApplicationModelEE(new DefaultApplicationModelFactory()
        .createApplicationModel(applicationDescriptor.getFirst(), runtimeExtensionModels,
                                new URLClassLoader(new URL[] {
                                    applicationDescriptor.getSecond().toURI().toURL()},
                                                   regionClassLoader),
                                ast -> identity(), emptyMap()));
  }

  protected void assertApplicationModel(Optional<ArtifactAst> applicationModelOptional) {
    assertThat(applicationModelOptional.isPresent(), is(true));

    ArtifactAst applicationModel = applicationModelOptional.get();
    assertThat(new File(applicationModel.topLevelComponentsStream()
        .filter(cm -> "simpleFlow".equals(cm.getComponentId().orElse(null)))
        .findFirst().get().getMetadata().getFileName().get()).getName(),
               equalTo("myApp.xml"));
    assertThat(new File(applicationModel.topLevelComponentsStream()
        .filter(cm -> "simpleFlowExtended".equals(cm.getComponentId().orElse(null)))
        .findFirst().get().getMetadata().getFileName().get()).getName(),
               equalTo("myApp-extended.xml"));
  }

  protected void assertApplicationModelEE(Optional<ArtifactAst> applicationModelOptional) {
    assertThat(applicationModelOptional.isPresent(), is(true));

    ArtifactAst applicationModel = applicationModelOptional.get();
    List<ComponentAst> componentModels =
        applicationModel.topLevelComponentsStream().filter(cm -> "staticDataSenseFlow".equals(cm.getComponentId().orElse(null)))
            .flatMap(cm -> cm.directChildrenStream())
            .collect(Collectors.toList());
    assertThat(componentModels.get(0).getLocation().getLocation(), is("staticDataSenseFlow/processors/0"));
    assertThat(componentModels.get(1).getLocation().getLocation(), is("staticDataSenseFlow/processors/1"));

    componentModels = applicationModel.topLevelComponentsStream()
        .filter(cm -> "staticDataSenseFlowEE".equals(cm.getComponentId().orElse(null))).flatMap(cm -> cm.directChildrenStream())
        .collect(Collectors.toList());
    assertThat(componentModels.get(0).getLocation().getLocation(), is("staticDataSenseFlowEE/processors/0"));
    assertThat(componentModels.get(1).getLocation().getLocation(), is("staticDataSenseFlowEE/processors/1"));
  }

}
