/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.el.ExpressionLanguageCapabilitiesService;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.api.domain.DomainDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.runtime.module.deployment.impl.internal.application.ToolingApplicationDescriptorFactory;
import org.mule.runtime.module.deployment.impl.internal.domain.DomainDescriptorFactory;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingRequest;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.ArtifactResources;
import org.mule.tooling.client.internal.application.DefaultApplication;
import org.mule.tooling.client.internal.application.Domain;
import org.mule.tooling.client.internal.dsl.DslSyntaxServiceCache;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.service.ServiceRegistry;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.Callable;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
@Feature("ToolingRuntimeClient")
@Story("Tooling Runtime Client configuration")
public class DefaultToolingRuntimeClientTestCase {

  public static final String ARTIFACT_ID = "id";
  @Mock
  private MavenClient mavenClient;
  @Mock
  private Domain domain;
  @Mock
  private Application application;
  @Mock
  private MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
  @Mock
  private MuleArtifactResourcesRegistry muleArtifactResourcesRegistry;
  @Mock
  private ApplicationCache applicationCache;
  @Mock
  private DomainCache domainCache;
  @Mock
  private DslSyntaxServiceCache dslSyntaxServiceCache;
  @Mock
  private ToolingApplicationDescriptorFactory toolingApplicationDescriptorFactory;
  @Mock
  private DomainDescriptorFactory domainDescriptorFactory;
  @Mock
  private ApplicationDescriptor applicationDescriptor;
  @Mock
  private DomainDescriptor domainDescriptor;
  @Mock
  private Serializer serializer;
  @Mock
  private ServiceRegistry serviceRegistry;
  @Mock
  private ExpressionLanguage expressionLanguage;
  @Mock
  private ExpressionLanguageMetadataService expressionLanguageMetadataService;
  @Mock
  private ExpressionLanguageCapabilitiesService expressionLanguageCapabilitiesService;

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();
  @Rule
  public ExpectedException expectedException = none();

  private File appLocation = toFile(this.getClass().getClassLoader().getResource("applications/email"));
  private File appWithDomainLocation = toFile(this.getClass().getClassLoader().getResource("applications/datasense-with-domain"));
  private File domainLocation = toFile(this.getClass().getClassLoader().getResource("domains/datasense-domain"));

  @Before
  public void before() {
    when(serviceRegistry.getExpressionLanguageMetadataService())
        .thenReturn(expressionLanguageMetadataService);
    when(serviceRegistry.getExpressionLanguageCapabilitiesService()).thenReturn(expressionLanguageCapabilitiesService);
  }

  @Test
  @Description("Validates that a new ToolingArtifact is created when not present in cache")
  public void newToolingArtifact() throws IOException {
    when(muleArtifactResourcesRegistry.getApplicationDescriptorFactory()).thenReturn(toolingApplicationDescriptorFactory);
    when(toolingApplicationDescriptorFactory.create(any(File.class), any(Optional.class))).thenReturn(applicationDescriptor);
    when(applicationDescriptor.getDomainDescriptor()).thenReturn(empty());
    when(application.getDomain()).thenReturn(empty());
    when(applicationCache.getApplication(any(String.class), any(Callable.class))).thenReturn(application);

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, serializer, empty(), muleRuntimeExtensionModelProvider,
                                        muleArtifactResourcesRegistry, applicationCache, domainCache,
                                        empty(), empty(), empty(), dslSyntaxServiceCache, serviceRegistry);
    ToolingArtifact toolingArtifact =
        toolingRuntimeClient.newToolingArtifact(appLocation.toURI().toURL(), emptyMap());
    assertThat(toolingArtifact, is(notNullValue()));

    verify(applicationCache).getApplication(any(String.class), any(Callable.class));
  }

  @Test
  @Description("Validates that a ToolingArtifact is invalidated from application cache whenever is disposed")
  public void onDisposeToolingArtifactIsInvalidatedFromApplicationCache() throws IOException {
    when(muleArtifactResourcesRegistry.getApplicationDescriptorFactory()).thenReturn(toolingApplicationDescriptorFactory);
    when(toolingApplicationDescriptorFactory.create(any(File.class), any(Optional.class))).thenReturn(applicationDescriptor);
    when(applicationDescriptor.getDomainDescriptor()).thenReturn(empty());
    when(application.getDomain()).thenReturn(empty());
    when(applicationCache.getApplication(any(String.class), any(Callable.class))).thenReturn(application);

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, serializer, empty(), muleRuntimeExtensionModelProvider,
                                        muleArtifactResourcesRegistry, applicationCache, domainCache,
                                        empty(), empty(), empty(), dslSyntaxServiceCache, serviceRegistry);
    ToolingArtifact toolingArtifact =
        toolingRuntimeClient.newToolingArtifact(appLocation.toURI().toURL(), emptyMap());
    assertThat(toolingArtifact, is(notNullValue()));

    doNothing().when(applicationCache).invalidate(toolingArtifact.getId());
    toolingArtifact.dispose();

    verify(applicationCache).getApplication(any(String.class), any(Callable.class));
    verify(applicationCache).invalidate(toolingArtifact.getId());
  }

  @Test
  @Description("Validates that a ToolingArtifact for an application cannot be created if the parentId doesn't reference to an existing domain in cache")
  public void referenceDomainIdDoesNotExists() throws IOException {
    when(domainCache.getDomain(any(String.class), any(Callable.class))).then(
                                                                             invocationOnMock -> ((Callable) invocationOnMock
                                                                                 .getArguments()[1]).call());

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, serializer, empty(), muleRuntimeExtensionModelProvider,
                                        muleArtifactResourcesRegistry, applicationCache, domainCache,
                                        empty(), empty(), empty(), dslSyntaxServiceCache, serviceRegistry);

    expectedException.expect(ToolingArtifactNotFoundException.class);
    expectedException.expectMessage("ToolingArtifact not found in Domain's cache for id: invalidId");
    toolingRuntimeClient.newToolingArtifact(appLocation.toURI().toURL(), emptyMap(), "invalidId");
  }

  @Test
  @Description("Validates that a new ToolingArtifact for application and the dependent domain")
  public void newToolingArtifactForApplicationDependsOnDomain() throws IOException {
    when(muleArtifactResourcesRegistry.getDomainDescriptorFactory()).thenReturn(domainDescriptorFactory);
    when(domainDescriptorFactory.create(any(File.class), any(Optional.class))).thenReturn(domainDescriptor);
    when(muleArtifactResourcesRegistry.getApplicationDescriptorFactory()).thenReturn(toolingApplicationDescriptorFactory);
    when(toolingApplicationDescriptorFactory.create(any(File.class), any(Optional.class))).thenReturn(applicationDescriptor);
    BundleDescriptor domainBundleDescriptor = new BundleDescriptor.Builder()
        .setGroupId("org.mule.test")
        .setArtifactId("datasense-domain")
        .setVersion("1.0.0")
        .setClassifier("mule-domain")
        .setType("jar")
        .build();
    Optional<BundleDescriptor> optionalDomainDescriptor = of(domainBundleDescriptor);
    when(applicationDescriptor.getDomainDescriptor()).thenReturn(optionalDomainDescriptor);

    when(domainCache.getDomain(any(String.class), any(Callable.class))).thenReturn(domain);
    when(applicationCache.getApplication(any(String.class), any(Callable.class))).thenReturn(application);
    when(application.getDomain()).thenReturn(of(domain));

    BundleDependency domainBundleDependency = mock(BundleDependency.class);
    when(mavenClient.resolveBundleDescriptor(new org.mule.maven.client.api.model.BundleDescriptor.Builder()
        .setGroupId(domainBundleDescriptor.getGroupId())
        .setArtifactId(domainBundleDescriptor.getArtifactId())
        .setVersion(domainBundleDescriptor.getVersion())
        .setClassifier(domainBundleDescriptor.getClassifier().get())
        .setType(domainBundleDescriptor.getType())
        .build())).thenReturn(domainBundleDependency);
    when(domainBundleDependency.getBundleUri()).thenReturn(domainLocation.toURI());

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, serializer, empty(), muleRuntimeExtensionModelProvider,
                                        muleArtifactResourcesRegistry, applicationCache, domainCache,
                                        empty(), empty(), empty(), dslSyntaxServiceCache, serviceRegistry);
    ToolingArtifact toolingArtifact =
        toolingRuntimeClient.newToolingArtifact(appWithDomainLocation.toURI().toURL(), emptyMap());
    assertThat(toolingArtifact, is(notNullValue()));
    assertThat(toolingArtifact.getParent().isPresent(), is(true));

    verify(applicationCache).getApplication(any(String.class), any(Callable.class));
    verify(domainCache).getDomain(any(String.class), any(Callable.class));
  }

  @Test
  @Description("Validates that a ToolingArtifact is invalidated from domain cache whenever is disposed")
  public void onDisposeToolingArtifactIsInvalidatedFromToolingCache() throws MalformedURLException {
    when(muleArtifactResourcesRegistry.getDomainDescriptorFactory()).thenReturn(domainDescriptorFactory);
    when(domainDescriptorFactory.create(any(File.class), any(Optional.class))).thenReturn(domainDescriptor);
    when(domainCache.getDomain(any(String.class), any(Callable.class))).thenReturn(domain);

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, serializer, empty(), muleRuntimeExtensionModelProvider,
                                        muleArtifactResourcesRegistry, applicationCache, domainCache,
                                        empty(), empty(), empty(), dslSyntaxServiceCache, serviceRegistry);
    ToolingArtifact toolingArtifact =
        toolingRuntimeClient.newToolingArtifact(domainLocation.toURI().toURL(), emptyMap());
    assertThat(toolingArtifact, is(notNullValue()));

    doNothing().when(domainCache).invalidate(toolingArtifact.getId());
    toolingArtifact.dispose();

    verify(domainCache).getDomain(any(String.class), any(Callable.class));
    verify(domainCache).invalidate(toolingArtifact.getId());
  }

  @Test
  @Description("Validates that a ToolingArtifact can be fetched and the application context is set")
  public void fetchToolingArtifact() throws IOException {
    String applicationId = UUID.randomUUID().toString();
    String domainId = UUID.randomUUID().toString();

    when(domain.getId()).thenReturn(domainId);
    when(domainCache.getDomain(anyString())).thenReturn(empty());
    when(applicationCache.getApplication(eq(applicationId), any(Callable.class))).thenReturn(application);
    when(application.getDomain()).thenReturn(of(domain));

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, serializer, empty(), muleRuntimeExtensionModelProvider,
                                        muleArtifactResourcesRegistry, applicationCache, domainCache,
                                        empty(), empty(), empty(), dslSyntaxServiceCache, serviceRegistry);
    ToolingArtifact toolingArtifact = toolingRuntimeClient.fetchToolingArtifact(applicationId);
    assertThat(toolingArtifact.getId(), equalTo(applicationId));
    assertThat(toolingArtifact.getParent().isPresent(), is(true));
    assertThat(toolingArtifact.getParent().get().getId(), equalTo(domainId));

    verify(applicationCache).getApplication(any(String.class), any(Callable.class));
    verify(application).setContext(any(ToolingArtifactContext.class));
  }

  @Test
  @Description("Validates that a ToolingArtifact is fetched and no longer in cache an exception is thrown")
  public void fetchToolingArtifactMissedCache() throws IOException {
    String applicationId = UUID.randomUUID().toString();

    when(domainCache.getDomain(anyString())).thenReturn(empty());
    when(applicationCache.getApplication(eq(applicationId), any(Callable.class)))
        .then(invocationOnMock -> ((Callable) invocationOnMock.getArguments()[1]).call());

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, serializer, empty(), muleRuntimeExtensionModelProvider,
                                        muleArtifactResourcesRegistry, applicationCache, domainCache,
                                        empty(), empty(), empty(), dslSyntaxServiceCache, serviceRegistry);

    expectedException.expect(ToolingArtifactNotFoundException.class);
    toolingRuntimeClient.fetchToolingArtifact(applicationId);
  }

  @Test
  @Description("Checks that when tooling artifact has to be deployed due to a connectivity testing request if no configuration for the remote tooling service is set an exception is thrown")
  public void shouldFailWhenTryingToConnectWithoutConfiguration() throws Exception {
    DefaultToolingArtifactContext context = new DefaultToolingArtifactContext(muleArtifactResourcesRegistry);
    context.setAgentConfiguration(empty());

    when(muleArtifactResourcesRegistry.getApplicationDescriptorFactory()).thenReturn(toolingApplicationDescriptorFactory);
    when(toolingApplicationDescriptorFactory.create(any(File.class), any(Optional.class))).thenReturn(applicationDescriptor);
    when(applicationDescriptor.getDomainDescriptor()).thenReturn(empty());

    final DefaultApplication defaultApplication =
        new DefaultApplication(ARTIFACT_ID, new ArtifactResources(ARTIFACT_ID, appLocation.toURI().toURL()),
                               applicationDescriptor, null,
                               context, emptyMap(), false);
    when(applicationCache.getApplication(any(String.class), any(Callable.class))).thenReturn(defaultApplication);

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, serializer, empty(), muleRuntimeExtensionModelProvider,
                                        muleArtifactResourcesRegistry, applicationCache, domainCache,
                                        empty(), empty(), empty(), dslSyntaxServiceCache, serviceRegistry);
    expectedException.expect(MissingToolingConfigurationException.class);
    toolingRuntimeClient.newToolingArtifact(appLocation.toURI().toURL(), emptyMap()).connectivityTestingService()
        .testConnection(new ConnectivityTestingRequest());
  }

}
