/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.join;
import static java.lang.System.clearProperty;
import static java.lang.System.getProperty;
import static java.lang.System.setProperty;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static jersey.repackaged.com.google.common.collect.Sets.newHashSet;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.hasItem;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.withSettings;
import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.mule.runtime.api.meta.model.connection.ConnectionManagementType.NONE;
import static org.mule.runtime.api.meta.model.display.PathModel.Location.EMBEDDED;
import static org.mule.runtime.api.meta.model.display.PathModel.Type.FILE;
import static org.mule.runtime.api.meta.model.operation.ExecutionType.BLOCKING;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.CONTENT;
import static org.mule.runtime.api.meta.model.stereotype.StereotypeModelBuilder.newStereotype;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.CONFIG;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.FLOW;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.OBJECT_STORE;
import static org.mule.tooling.client.api.extension.model.value.ValuesResolverModel.metadataKeyResolverName;
import static org.mule.tooling.client.api.extension.model.value.ValuesResolverModel.valueProviderResolverName;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.FILTER_PARAMETERS_RESERVED_NAMES;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toActingParameterModelsDTO;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.SimpleTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.java.api.JavaTypeLoader;
import org.mule.runtime.api.meta.Category;
import org.mule.runtime.api.meta.ExternalLibraryType;
import org.mule.runtime.api.meta.model.ExternalLibraryModel;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.OutputModel;
import org.mule.runtime.api.meta.model.ParameterDslConfiguration;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionManagementType;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.deprecated.DeprecationModel;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.display.LayoutModel;
import org.mule.runtime.api.meta.model.display.PathModel;
import org.mule.runtime.api.meta.model.error.ImmutableErrorModel;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModelBuilder;
import org.mule.runtime.extension.api.model.ImmutableOutputModel;
import org.mule.runtime.extension.api.model.connection.ImmutableConnectionProviderModel;
import org.mule.runtime.extension.api.model.construct.ImmutableConstructModel;
import org.mule.runtime.extension.api.model.deprecated.ImmutableDeprecationModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedChainModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedComponentModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedRouteModel;
import org.mule.runtime.extension.api.model.operation.ImmutableOperationModel;
import org.mule.runtime.extension.api.model.parameter.ImmutableParameterModel;
import org.mule.runtime.extension.api.model.source.ImmutableSourceModel;
import org.mule.runtime.extension.api.property.MetadataKeyIdModelProperty;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;
import org.mule.runtime.extension.api.property.QNameModelProperty;
import org.mule.runtime.extension.api.property.TypeResolversInformationModelProperty;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.data.sample.SampleDataProviderModel;
import org.mule.tooling.client.api.extension.model.function.FunctionModel;
import org.mule.tooling.client.api.extension.model.nested.NestableElementModel;
import org.mule.tooling.client.api.extension.model.nested.NestedChainModel;
import org.mule.tooling.client.api.extension.model.nested.NestedComponentModel;
import org.mule.tooling.client.api.extension.model.nested.NestedRouteModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.api.extension.model.value.FieldValueProviderModel;
import org.mule.tooling.client.api.extension.model.value.FieldValuesResolverModel;
import org.mule.tooling.client.api.extension.model.value.ValuesResolverModel;
import org.mule.tooling.client.internal.hamcrest.ActingParameterMatcher;
import org.mule.tooling.client.internal.hamcrest.ConnectionProviderMatcher;
import org.mule.tooling.client.internal.hamcrest.ConstructModelMatcher;
import org.mule.tooling.client.internal.hamcrest.NestableElementMatcher;
import org.mule.tooling.client.internal.hamcrest.NestedChainMatcher;
import org.mule.tooling.client.internal.hamcrest.NestedComponentMatcher;
import org.mule.tooling.client.internal.hamcrest.NestedRouteMatcher;
import org.mule.tooling.client.internal.hamcrest.OperationModelMatcher;
import org.mule.tooling.client.internal.hamcrest.ParameterModelMatcher;
import org.mule.tooling.client.internal.hamcrest.SourceModelMatcher;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ImmutableList;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import javax.xml.namespace.QName;
import org.hamcrest.Matchers;
import org.hamcrest.collection.IsCollectionWithSize;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;
import scala.Int;

@Feature("DTOs")
@Story("Extension Model for Tooling API")
@RunWith(MockitoJUnitRunner.class)
public class ExtensionModelFactoryTestCase {

  private static final String EXTENSION_NAME = "name";
  private static final String EXTENSION_DESCRIPTION = "description";
  private static final String EXTENSION_VERSION = "1.0";
  private static final String EXTENSION_VENDOR = "MuleSoft";

  private static final String DISPLAY_NAME = "displayName";
  private static final String DISPLAY_EXAMPLE = "displayExample";
  private static final String DISPLAY_SUMMARY = "displaySummary";
  private static final List<String> FILE_EXTENSIONS = newArrayList("java");

  private static final String PREFIX = "prefix";
  private static final String NAMESPACE = "namespace";
  private static final String SCHEMA_VERSION = "schemaVersion";
  private static final String SCHEMA_LOCATION = "schemaLocation";

  private static final String CONFIGURATION_NAME = "configName";
  private static final String CONFIGURATION_DESCRIPTION = "configDescription";

  private static final String CONNECTION_PROVIDER_NAME = "connectionProviderName";
  private static final String CONNECTION_PROVIDER_DESCRIPTION = "connectionProviderDescription";

  private static final String EXTERNAL_LIB_MODEL_NAME = "externalLibModelName";
  private static final String EXTERNAL_LIB_MODEL_DESCRIPTION = "externalLibModelDescription";
  private static final String EXTERNAL_LIB_MODEL_REGEXP_MATCHER = "a-z";

  private static final String FUNCTION_NAME = "functionName";

  private static final String SIMPLE_PARAMETER_NAME = "parameter";
  private static final String PARAMETER_WITH_REFERENCES_NAME = "withRefs";
  private static final String SIMPLE_PARAMETER_GROUP_NAME = "parameterGroup";

  private static final String ERROR_MODEL_TYPE = "type";
  private static final String PARENT_ERROR_MODEL_TYPE = "parent-type";
  private static final String ERROR_MODEL_NAMESPACE = "namespace";
  private static final String PARENT_ERROR_MODEL_NAMESPACE = "parent-namespace";

  private static final String CONSTRUCT_NAME = "constructName";
  private static final String CONSTRUCT_DESCRIPTION = "constructDescription";

  private org.mule.runtime.api.meta.model.ExtensionModel runtimeExtensionModel;
  private XmlDslModel xmlDslModel;
  private DisplayModel displayModel;
  private ExternalLibraryModel externalLibraryModel;
  private MetadataType stringType;
  private OutputModel outputModel;
  org.mule.runtime.api.meta.model.parameter.ParameterModel simpleRuntimeParameterModel;

  private static final String MIN_MULE_VERSION = "4.0.0";

  @Mock
  org.mule.runtime.api.meta.model.function.FunctionModel runtimeFunctionModel;

  @Mock
  org.mule.runtime.api.meta.model.parameter.ParameterGroupModel simpleRuntimeParameterGroupModel;

  @Before
  public void mockExtensionModelBase() {
    stringType = new JavaTypeLoader(Thread.currentThread().getContextClassLoader()).load(String.class);
    simpleRuntimeParameterModel = mockedParameterModel(SIMPLE_PARAMETER_NAME);

    initialiseExtensionModel();
    initialiseSimpleParameterGroup();
    initialiseFunction(runtimeFunctionModel);

    when(runtimeExtensionModel.getFunctionModels()).thenReturn(singletonList(runtimeFunctionModel));
  }

  @Test
  public void testExtensionModel() {
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertExtensionModelBasicAttributes(dto);
  }

  @Test
  public void testParameterWithConfigReferences() {
    assertParameterAllowedStereotype(CONFIG);
  }

  @Test
  public void testParametersFiltering() {
    org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModelFiltered =
        mockedParameterModel("name");
    when(parameterModelFiltered.isComponentId()).thenReturn(true);

    org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModelNotFiltered =
        mockedParameterModel("notFiltered");
    when(simpleRuntimeParameterGroupModel.getParameterModels())
        .thenReturn(asList(parameterModelFiltered, parameterModelNotFiltered));

    ConfigurationModel configModel = mock(ConfigurationModel.class);
    when(configModel.getName()).thenReturn(CONFIGURATION_NAME);
    when(configModel.getDescription()).thenReturn(CONFIGURATION_DESCRIPTION);
    when(configModel.getDisplayModel()).thenReturn(of(displayModel));
    when(configModel.getParameterGroupModels()).thenReturn(singletonList(simpleRuntimeParameterGroupModel));
    when(configModel.getDeprecationModel())
        .thenReturn(of(new ImmutableDeprecationModel("This config is deprecated", "2.2.0", null)));

    when(runtimeExtensionModel.getConfigurationModels()).thenReturn(newArrayList(configModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    List<ParameterModel> params = dto.getConfigurationModels().get(0).getParameterGroupModels().get(0).getParameterModels();
    assertThat(params, hasSize(1));
  }

  @Test
  public void testParametersNotFiltering() {
    final String previousValue = getProperty(FILTER_PARAMETERS_RESERVED_NAMES);
    setProperty(FILTER_PARAMETERS_RESERVED_NAMES, "false");

    try {
      org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModelFiltered = mockedParameterModel("name");

      org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModelNotFiltered =
          mockedParameterModel("notFiltered");
      when(simpleRuntimeParameterGroupModel.getParameterModels())
          .thenReturn(asList(parameterModelFiltered, parameterModelNotFiltered));

      ConfigurationModel configModel = mock(ConfigurationModel.class);
      when(configModel.getName()).thenReturn(CONFIGURATION_NAME);
      when(configModel.getDescription()).thenReturn(CONFIGURATION_DESCRIPTION);
      when(configModel.getDisplayModel()).thenReturn(of(displayModel));
      when(configModel.getParameterGroupModels()).thenReturn(singletonList(simpleRuntimeParameterGroupModel));
      when(configModel.getDeprecationModel())
          .thenReturn(of(new ImmutableDeprecationModel("This config is deprecated", "2.2.0", null)));

      when(runtimeExtensionModel.getConfigurationModels()).thenReturn(newArrayList(configModel));

      ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
      List<ParameterModel> params = dto.getConfigurationModels().get(0).getParameterGroupModels().get(0).getParameterModels();
      assertThat(params, hasSize(2));
    } finally {
      if (previousValue != null) {
        setProperty(FILTER_PARAMETERS_RESERVED_NAMES, previousValue);
      } else {
        clearProperty(FILTER_PARAMETERS_RESERVED_NAMES);
      }
    }
  }

  @Test
  public void testParameterWithFlowReferences() {
    assertParameterAllowedStereotype(FLOW);
  }

  @Test
  public void testParameterWithObjectStoreReferences() {
    assertParameterAllowedStereotype(OBJECT_STORE);
  }

  private void assertParameterAllowedStereotype(StereotypeModel parent) {
    org.mule.runtime.api.meta.model.parameter.ParameterModel paramWithReferences =
        mockedParameterModel(PARAMETER_WITH_REFERENCES_NAME);

    when(paramWithReferences.getAllowedStereotypes())
        .thenReturn(singletonList(StereotypeModelBuilder.newStereotype("config", "test")
            .withParent(parent)
            .build()));

    when(simpleRuntimeParameterGroupModel.getParameterModels()).thenReturn(singletonList(paramWithReferences));

    ConfigurationModel configModel = mock(ConfigurationModel.class);
    when(configModel.getName()).thenReturn(CONFIGURATION_NAME);
    when(configModel.getDescription()).thenReturn(CONFIGURATION_DESCRIPTION);
    when(configModel.getDisplayModel()).thenReturn(of(displayModel));
    when(configModel.getParameterGroupModels()).thenReturn(singletonList(simpleRuntimeParameterGroupModel));
    when(configModel.getDeprecationModel())
        .thenReturn(of(new ImmutableDeprecationModel("This config is deprecated", "2.2.0", null)));

    when(runtimeExtensionModel.getConfigurationModels()).thenReturn(newArrayList(configModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    List<ParameterModel> params = dto.getConfigurationModels().get(0).getParameterGroupModels().get(0).getParameterModels();
    assertThat(params, hasSize(1));

    org.mule.tooling.client.api.extension.model.StereotypeModel ref = params.get(0).getAllowedStereotypes().get(0);
    assertThat(ref.getParent().get().getType(), is(parent.getType()));
    assertThat(ref.getNamespace(), is("TEST"));
    assertThat(ref.getType(), is("CONFIG"));

    boolean isConfig = parent == CONFIG;
    boolean isFlow = parent == FLOW;
    boolean isObjectStore = parent == OBJECT_STORE;

    assertThat(ref.isModuleConfig(), is(isConfig));
    assertThat(ref.isFlow(), is(isFlow));
    assertThat(ref.isObjectStore(), is(isObjectStore));
  }

  @Test
  public void semanticTermsInNestedComponentModels() {
    ImmutableNestedComponentModel nestedComponentModel = new ImmutableNestedComponentModel(
                                                                                           "name",
                                                                                           "description",
                                                                                           emptyList(),
                                                                                           10,
                                                                                           99,
                                                                                           emptySet(),
                                                                                           emptyList(),
                                                                                           displayModel,
                                                                                           emptySet(),
                                                                                           null,
                                                                                           emptySet(),
                                                                                           null,
                                                                                           singleton("someSemanticTerm"));

    ImmutableConstructModel constructModel = new ImmutableConstructModel(CONSTRUCT_NAME,
                                                                         CONSTRUCT_DESCRIPTION,
                                                                         emptyList(),
                                                                         ImmutableList.of(nestedComponentModel),
                                                                         true,
                                                                         displayModel,
                                                                         emptySet(),
                                                                         null,
                                                                         emptySet(),
                                                                         null);

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);
    NestableElementModel nestedComponentModelDTO = constructModelDTO.getNestedComponents().get(0);

    assertThat(nestedComponentModelDTO, is(NestableElementMatcher.from(nestedComponentModel)));
  }

  @Test
  public void semanticTermsInNestedChainModels() {
    ImmutableNestedChainModel nestedChainModel = new ImmutableNestedChainModel(
                                                                               "name",
                                                                               "description",
                                                                               emptyList(),
                                                                               true,
                                                                               emptySet(),
                                                                               emptyList(),
                                                                               null,
                                                                               emptySet(),
                                                                               null,
                                                                               emptySet(),
                                                                               null,
                                                                               singleton("someSemanticTerm"));

    ImmutableConstructModel constructModel = new ImmutableConstructModel(CONSTRUCT_NAME,
                                                                         CONSTRUCT_DESCRIPTION,
                                                                         emptyList(),
                                                                         ImmutableList.of(nestedChainModel),
                                                                         true,
                                                                         null,
                                                                         emptySet(),
                                                                         null,
                                                                         emptySet(),
                                                                         null);

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);
    NestableElementModel nestedComponentModelDTO = constructModelDTO.getNestedComponents().get(0);

    assertThat(nestedComponentModelDTO, is(NestableElementMatcher.from(nestedChainModel)));
  }

  @Test
  public void semanticTermsInNestedRouteModels() {
    ImmutableNestedRouteModel nestedRouteModel = new ImmutableNestedRouteModel(
                                                                               "name",
                                                                               "description",
                                                                               emptyList(),
                                                                               displayModel,
                                                                               Integer.MIN_VALUE,
                                                                               Integer.MAX_VALUE,
                                                                               emptyList(),
                                                                               null,
                                                                               emptySet(),
                                                                               null,
                                                                               singleton("semanticTerm"));

    ImmutableConstructModel constructModel = new ImmutableConstructModel(CONSTRUCT_NAME,
                                                                         CONSTRUCT_DESCRIPTION,
                                                                         emptyList(),
                                                                         ImmutableList.of(nestedRouteModel),
                                                                         true,
                                                                         null,
                                                                         emptySet(),
                                                                         null,
                                                                         emptySet(),
                                                                         null);

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);
    NestableElementModel nestedRouteModelDTO = constructModelDTO.getNestedComponents().get(0);

    assertThat(nestedRouteModelDTO, is(NestableElementMatcher.from(nestedRouteModel)));
  }

  @Test
  public void semanticTermsInParameterModel() {
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel =
        new ImmutableParameterModel("name",
                                    "description",
                                    BaseTypeBuilder.create(JAVA).stringType().build(),
                                    false,
                                    true,
                                    false,
                                    false,
                                    SUPPORTED,
                                    "defaultValue",
                                    BEHAVIOUR,
                                    ParameterDslConfiguration.getDefaultInstance(),
                                    displayModel,
                                    LayoutModel.builder().build(),
                                    null,
                                    emptyList(),
                                    emptySet(),
                                    null,
                                    singleton("semanticTerm"),
                                    emptyList());

    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel = mockedParameterGroupModel("group");
    when(parameterGroupModel.getParameterModels()).thenReturn(singletonList(parameterModel));
    final List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels =
        singletonList(parameterGroupModel);

    org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel("operation");
    when(operationModel.getParameterGroupModels()).thenReturn(parameterGroupModels);

    when(runtimeExtensionModel.getOperationModels()).thenReturn(singletonList(operationModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<OperationModel> operationModelsDTO = dto.getOperationModels();
    assertThat(operationModelsDTO, hasSize(1));

    ParameterModel parameterModelDTO = operationModelsDTO.get(0).getAllParameterModels().get(0);

    assertThat(parameterModelDTO, is(ParameterModelMatcher.from(parameterModel)));
  }

  @Test
  public void semanticTermsInConnectionProvider() {
    ConnectionProviderModel connectionProviderModel =
        new ImmutableConnectionProviderModel("name", "description", emptyList(), NONE, true, emptySet(), displayModel, null,
                                             emptySet(), null, singleton("semanticTerm"));

    when(runtimeExtensionModel.getConnectionProviders()).thenReturn(singletonList(connectionProviderModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel> connectionProviderModelsDTO =
        dto.getConnectionProviders();
    assertThat(connectionProviderModelsDTO, hasSize(1));

    org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel connectionProviderDTO =
        connectionProviderModelsDTO.get(0);

    assertThat(connectionProviderDTO, is(ConnectionProviderMatcher.from(connectionProviderModel)));
  }

  @Test
  public void semanticTermsInOperationModel() {
    org.mule.runtime.api.meta.model.operation.OperationModel operationModel = new ImmutableOperationModel(
                                                                                                          "name",
                                                                                                          "description",
                                                                                                          emptyList(),
                                                                                                          emptyList(),
                                                                                                          outputModel,
                                                                                                          outputModel,
                                                                                                          false,
                                                                                                          BLOCKING,
                                                                                                          false,
                                                                                                          false,
                                                                                                          false,
                                                                                                          displayModel,
                                                                                                          emptySet(),
                                                                                                          null,
                                                                                                          emptySet(),
                                                                                                          emptySet(),
                                                                                                          null,
                                                                                                          null,
                                                                                                          singleton("semanticTerm"));

    when(runtimeExtensionModel.getOperationModels()).thenReturn(singletonList(operationModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<OperationModel> operationModelsDTO = dto.getOperationModels();
    assertThat(operationModelsDTO, hasSize(1));

    OperationModel operationModelDTO = operationModelsDTO.get(0);

    assertThat(operationModelDTO, is(OperationModelMatcher.from(operationModel)));
  }

  @Test
  public void semanticTermsInSourceModel() {
    org.mule.runtime.api.meta.model.source.SourceModel sourceModel = new ImmutableSourceModel(
                                                                                              "name",
                                                                                              "description",
                                                                                              false,
                                                                                              false,
                                                                                              emptyList(),
                                                                                              emptyList(),
                                                                                              outputModel,
                                                                                              outputModel,
                                                                                              empty(),
                                                                                              empty(),
                                                                                              empty(),
                                                                                              false,
                                                                                              false,
                                                                                              false,
                                                                                              displayModel,
                                                                                              null,
                                                                                              emptySet(),
                                                                                              emptySet(),
                                                                                              emptySet(),
                                                                                              null,
                                                                                              null,
                                                                                              singleton("semanticTerm"));

    when(runtimeExtensionModel.getSourceModels()).thenReturn(singletonList(sourceModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<SourceModel> sourceModelsDTO = dto.getSourceModels();
    assertThat(sourceModelsDTO, hasSize(1));

    SourceModel sourceModelDTO = sourceModelsDTO.get(0);

    assertThat(sourceModelDTO, is(SourceModelMatcher.from(sourceModel)));
  }

  @Test
  public void semanticTermsInConstructModel() {
    org.mule.runtime.api.meta.model.construct.ConstructModel constructModel = new ImmutableConstructModel(
                                                                                                          CONSTRUCT_NAME,
                                                                                                          CONSTRUCT_DESCRIPTION,
                                                                                                          emptyList(),
                                                                                                          emptyList(),
                                                                                                          true,
                                                                                                          displayModel,
                                                                                                          emptySet(),
                                                                                                          null,
                                                                                                          emptySet(),
                                                                                                          null,
                                                                                                          singleton("semanticTerm"));

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);

    assertThat(constructModelDTO, is(ConstructModelMatcher.from(constructModel)));
  }

  @Test
  public void testErrorModel() {
    org.mule.runtime.api.meta.model.error.ErrorModel parentErrorModel =
        new ImmutableErrorModel(PARENT_ERROR_MODEL_TYPE, PARENT_ERROR_MODEL_NAMESPACE, false, null);
    org.mule.runtime.api.meta.model.error.ErrorModel errorModel =
        new ImmutableErrorModel(ERROR_MODEL_TYPE, ERROR_MODEL_NAMESPACE, true, parentErrorModel);

    when(runtimeExtensionModel.getErrorModels()).thenReturn(singleton(errorModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    Set<ErrorModel> errorModelsDTO = dto.getErrorModels();
    assertThat(errorModelsDTO, hasSize(1));

    ErrorModel errorModelDTO = errorModelsDTO.iterator().next();

    assertThat(errorModelDTO.getType(), is(ERROR_MODEL_TYPE));
    assertThat(errorModelDTO.getNamespace(), is(ERROR_MODEL_NAMESPACE));
    assertThat(errorModelDTO.getHandleable().get(), is(true));
    assertThat(errorModelDTO.getParent(), is(not(empty())));
    assertThat(errorModelDTO.getParent().get().getType(), is(PARENT_ERROR_MODEL_TYPE));
    assertThat(errorModelDTO.getParent().get().getNamespace(), is(PARENT_ERROR_MODEL_NAMESPACE));
    assertThat(errorModelDTO.getParent().get().getParent(), is(empty()));
    assertThat(errorModelDTO.getParent().get().getHandleable().get(), is(false));
  }

  @Test
  public void testConstructModels() {
    final StereotypeModel stereotypeModel = StereotypeModelBuilder.newStereotype("config", "test").build();

    org.mule.runtime.api.meta.model.error.ErrorModel errorModel =
        new ImmutableErrorModel(ERROR_MODEL_TYPE, ERROR_MODEL_NAMESPACE, true, null);

    ImmutableNestedComponentModel nestedComponentModel = new ImmutableNestedComponentModel(
                                                                                           "name", "description", null, true,
                                                                                           newHashSet(stereotypeModel),
                                                                                           emptySet());
    ImmutableNestedChainModel nestedChainModel = new ImmutableNestedChainModel(
                                                                               "name", "description", null, true,
                                                                               newHashSet(stereotypeModel), emptySet());
    ImmutableNestedRouteModel nestedRouteModel = new ImmutableNestedRouteModel(
                                                                               "name", "description", emptyList(), null,
                                                                               Int.MinValue(),
                                                                               Int.MaxValue(), ImmutableList
                                                                                   .of(nestedComponentModel, nestedChainModel),
                                                                               emptySet());

    ImmutableConstructModel constructModel = new ImmutableConstructModel(CONSTRUCT_NAME,
                                                                         CONSTRUCT_DESCRIPTION,
                                                                         emptyList(),
                                                                         ImmutableList.of(nestedComponentModel, nestedChainModel,
                                                                                          nestedRouteModel),
                                                                         true,
                                                                         null,
                                                                         newHashSet(errorModel),
                                                                         stereotypeModel,
                                                                         emptySet(),
                                                                         new ImmutableDeprecationModel("Deprecated construct",
                                                                                                       "1.4.0", null));

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);

    assertThat(constructModelDTO.getName(), is(CONSTRUCT_NAME));
    assertThat(constructModelDTO.getDescription(), is(CONSTRUCT_DESCRIPTION));
    assertThat(constructModelDTO.getDisplayModel(), is(empty()));
    assertThat(constructModelDTO.getNestedComponents(), IsCollectionWithSize.hasSize(3));
    assertThat(constructModelDTO.getStereotype().getType(), is(stereotypeModel.getType()));
    assertThat(constructModelDTO.getStereotype().getNamespace(), is(stereotypeModel.getNamespace()));
    assertThat(constructModelDTO.getStereotype().getParent(), is(empty()));
    assertTrue(constructModelDTO.getDeprecationModel().isEnabled());
    assertThat(constructModelDTO.getDeprecationModel().get().getMessage(), is("Deprecated construct"));
    assertThat(constructModelDTO.getDeprecationModel().get().getDeprecatedSince(), is("1.4.0"));
    assertFalse(constructModelDTO.getDeprecationModel().get().getToRemoveIn().isPresent());

    assertThat(constructModelDTO.getErrorModels(), hasSize(1));
    ErrorModel errorModelDTO = constructModelDTO.getErrorModels().iterator().next();

    assertThat(errorModelDTO.getType(), is(ERROR_MODEL_TYPE));
    assertThat(errorModelDTO.getNamespace(), is(ERROR_MODEL_NAMESPACE));
    assertThat(errorModelDTO.getParent(), is(empty()));

    assertThat(constructModelDTO.getNestedComponents(), IsCollectionWithSize.hasSize(3));

    NestedComponentModel nestedComponentModelDTO = (NestedComponentModel) constructModelDTO.getNestedComponents().get(0);
    assertThat(nestedComponentModelDTO.getName(), is(nestedComponentModel.getName()));
    assertThat(nestedComponentModelDTO.getDescription(), is(nestedComponentModel.getDescription()));
    assertThat(nestedComponentModelDTO.getDisplayModel(), is(nullValue()));
    assertThat(nestedComponentModelDTO.isRequired(), is(true));
    assertThat(nestedComponentModelDTO.getAllowedStereotypes(), IsCollectionWithSize.hasSize(1));

    assertThat(nestedComponentModelDTO, is(NestedComponentMatcher.from(nestedComponentModel)));

    NestedChainModel nestedChainModelDTO = (NestedChainModel) constructModelDTO.getNestedComponents().get(1);
    assertThat(nestedChainModelDTO.getName(), is(nestedComponentModel.getName()));
    assertThat(nestedChainModelDTO.getDescription(), is(nestedComponentModel.getDescription()));
    assertThat(nestedChainModelDTO.getDisplayModel(), is(nullValue()));
    assertThat(nestedChainModelDTO.isRequired(), is(true));
    assertThat(nestedChainModelDTO.getAllowedStereotypes(), IsCollectionWithSize.hasSize(1));

    assertThat(nestedChainModelDTO, is(NestedChainMatcher.from(nestedChainModel)));

    NestedRouteModel nestedRouteModelDTO = (NestedRouteModel) constructModelDTO.getNestedComponents().get(2);
    assertThat(nestedRouteModelDTO.getName(), is(nestedComponentModel.getName()));
    assertThat(nestedRouteModelDTO.getDescription(), is(nestedComponentModel.getDescription()));
    assertThat(nestedRouteModelDTO.getDisplayModel(), is(nullValue()));
    assertThat(nestedRouteModelDTO.isRequired(), is(false));
    assertThat(nestedRouteModelDTO.getMinOccurs(), is(nestedRouteModel.getMinOccurs()));
    assertThat(nestedRouteModelDTO.getMaxOccurs(), is(nestedRouteModel.getMaxOccurs()));
    assertThat(nestedRouteModelDTO.getChildComponents(), IsCollectionWithSize.hasSize(2));


    assertThat(nestedRouteModelDTO.getChildComponents().get(0), is(nestedComponentModelDTO));
    assertThat(nestedRouteModelDTO.getChildComponents().get(1), is(nestedChainModelDTO));

    assertThat(nestedRouteModelDTO, is(NestedRouteMatcher.from(nestedRouteModel)));
  }

  @Test
  public void testMinAndMaxOccurs() {
    ImmutableNestedComponentModel nestedComponentModel = new ImmutableNestedComponentModel(
                                                                                           "name",
                                                                                           "description",
                                                                                           emptyList(),
                                                                                           10,
                                                                                           20,
                                                                                           emptySet(),
                                                                                           emptyList(),
                                                                                           displayModel,
                                                                                           emptySet(),
                                                                                           null,
                                                                                           emptySet(),
                                                                                           null,
                                                                                           emptySet());

    ImmutableNestedChainModel nestedChainModel = new ImmutableNestedChainModel(
                                                                               "name",
                                                                               "description",
                                                                               emptyList(),
                                                                               true,
                                                                               emptySet(),
                                                                               emptyList(),
                                                                               displayModel,
                                                                               emptySet(),
                                                                               null,
                                                                               emptySet(),
                                                                               null,
                                                                               emptySet());

    ImmutableNestedRouteModel nestedRouteModel = new ImmutableNestedRouteModel(
                                                                               "name",
                                                                               "description",
                                                                               emptyList(),
                                                                               displayModel,
                                                                               10,
                                                                               20,
                                                                               emptyList(),
                                                                               emptySet());

    ImmutableConstructModel constructModel = new ImmutableConstructModel(CONSTRUCT_NAME,
                                                                         CONSTRUCT_DESCRIPTION,
                                                                         emptyList(),
                                                                         ImmutableList.of(nestedComponentModel, nestedChainModel,
                                                                                          nestedRouteModel),
                                                                         true,
                                                                         displayModel,
                                                                         emptySet(),
                                                                         null,
                                                                         emptySet(),
                                                                         null);

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);
    assertThat(constructModelDTO, is(ConstructModelMatcher.from(constructModel)));
  }

  @Test
  public void testNestedComponentModel() {
    final StereotypeModel stereotypeModel = StereotypeModelBuilder.newStereotype("config", "test").build();
    final org.mule.runtime.api.meta.model.error.ErrorModel errorModel =
        new ImmutableErrorModel(ERROR_MODEL_TYPE, ERROR_MODEL_NAMESPACE, true, null);
    final DeprecationModel deprecationModel = new ImmutableDeprecationModel("message", "0.0.0", "99.99.99");
    final ModelProperty modelProperty = new QNameModelProperty(new QName("ns", "v"));
    final ValueProviderModel valueProviderModel =
        new ValueProviderModel(emptyList(), false, false, false, 1, "providerName", "providerId");
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel = mockedParameterModel("parameter");
    when(parameterModel.getValueProviderModel()).thenReturn(of(valueProviderModel));
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel = mockedParameterGroupModel("group");
    when(parameterGroupModel.getParameterModels()).thenReturn(singletonList(parameterModel));
    final List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels =
        singletonList(parameterGroupModel);
    final ImmutableNestedChainModel nestedChainModel = new ImmutableNestedChainModel(
                                                                                     "name", "description", null, true,
                                                                                     newHashSet(stereotypeModel), emptySet());

    ImmutableNestedComponentModel nestedComponentModel = new ImmutableNestedComponentModel(
                                                                                           "name",
                                                                                           "description",
                                                                                           parameterGroupModels,
                                                                                           10,
                                                                                           99,
                                                                                           singleton(stereotypeModel),
                                                                                           ImmutableList.of(nestedChainModel,
                                                                                                            nestedChainModel),
                                                                                           null,
                                                                                           singleton(errorModel),
                                                                                           stereotypeModel,
                                                                                           singleton(modelProperty),
                                                                                           deprecationModel,
                                                                                           singleton("someSemanticTerm"));

    ImmutableConstructModel constructModel = new ImmutableConstructModel(CONSTRUCT_NAME,
                                                                         CONSTRUCT_DESCRIPTION,
                                                                         emptyList(),
                                                                         ImmutableList.of(nestedComponentModel),
                                                                         true,
                                                                         null,
                                                                         newHashSet(errorModel),
                                                                         stereotypeModel,
                                                                         emptySet(),
                                                                         null);

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);
    NestableElementModel nestedComponentModelDTO = constructModelDTO.getNestedComponents().get(0);

    assertThat(nestedComponentModelDTO, is(NestableElementMatcher.from(nestedComponentModel)));
  }

  @Test
  public void testNestedChainModel() {
    final StereotypeModel stereotypeModel = StereotypeModelBuilder.newStereotype("config", "test").build();
    final org.mule.runtime.api.meta.model.error.ErrorModel errorModel =
        new ImmutableErrorModel(ERROR_MODEL_TYPE, ERROR_MODEL_NAMESPACE, true, null);
    final DeprecationModel deprecationModel = new ImmutableDeprecationModel("message", "0.0.0", "99.99.99");
    final ModelProperty modelProperty = new QNameModelProperty(new QName("ns", "v"));
    final ValueProviderModel valueProviderModel =
        new ValueProviderModel(emptyList(), false, false, false, 1, "providerName", "providerId");
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel = mockedParameterModel("parameter");
    when(parameterModel.getValueProviderModel()).thenReturn(of(valueProviderModel));
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel = mockedParameterGroupModel("group");
    when(parameterGroupModel.getParameterModels()).thenReturn(singletonList(parameterModel));
    final List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels =
        singletonList(parameterGroupModel);
    final ImmutableNestedComponentModel nestedComponentModel = new ImmutableNestedComponentModel(
                                                                                                 "name",
                                                                                                 "description",
                                                                                                 parameterGroupModels,
                                                                                                 10,
                                                                                                 99,
                                                                                                 singleton(stereotypeModel),
                                                                                                 emptyList(),
                                                                                                 null,
                                                                                                 singleton(errorModel),
                                                                                                 stereotypeModel,
                                                                                                 singleton(modelProperty),
                                                                                                 deprecationModel,
                                                                                                 singleton("someSemanticTerm"));

    final ImmutableNestedChainModel nestedChainModel = new ImmutableNestedChainModel(
                                                                                     "name",
                                                                                     "description",
                                                                                     parameterGroupModels,
                                                                                     true,
                                                                                     singleton(stereotypeModel),
                                                                                     singletonList(nestedComponentModel),
                                                                                     displayModel,
                                                                                     singleton(errorModel),
                                                                                     stereotypeModel,
                                                                                     singleton(modelProperty),
                                                                                     deprecationModel,
                                                                                     null);

    ImmutableConstructModel constructModel = new ImmutableConstructModel(CONSTRUCT_NAME,
                                                                         CONSTRUCT_DESCRIPTION,
                                                                         emptyList(),
                                                                         ImmutableList.of(nestedChainModel),
                                                                         true,
                                                                         null,
                                                                         newHashSet(errorModel),
                                                                         stereotypeModel,
                                                                         emptySet(),
                                                                         null);

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);
    NestableElementModel nestedChainModelDTO = constructModelDTO.getNestedComponents().get(0);

    assertThat(nestedChainModelDTO, is(NestableElementMatcher.from(nestedChainModel)));
  }

  @Test
  public void testNestedRouteModel() {
    final StereotypeModel stereotypeModel = StereotypeModelBuilder.newStereotype("config", "test").build();
    final org.mule.runtime.api.meta.model.error.ErrorModel errorModel =
        new ImmutableErrorModel(ERROR_MODEL_TYPE, ERROR_MODEL_NAMESPACE, true, null);
    final DeprecationModel deprecationModel = new ImmutableDeprecationModel("message", "0.0.0", "99.99.99");
    final ModelProperty modelProperty = new QNameModelProperty(new QName("ns", "v"));
    final ValueProviderModel valueProviderModel =
        new ValueProviderModel(emptyList(), false, false, false, 1, "providerName", "providerId");
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel = mockedParameterModel("parameter");
    when(parameterModel.getValueProviderModel()).thenReturn(of(valueProviderModel));
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel = mockedParameterGroupModel("group");
    when(parameterGroupModel.getParameterModels()).thenReturn(singletonList(parameterModel));
    final List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels =
        singletonList(parameterGroupModel);
    final ImmutableNestedComponentModel nestedComponentModel = new ImmutableNestedComponentModel(
                                                                                                 "name",
                                                                                                 "description",
                                                                                                 parameterGroupModels,
                                                                                                 10,
                                                                                                 99,
                                                                                                 singleton(stereotypeModel),
                                                                                                 emptyList(),
                                                                                                 null,
                                                                                                 singleton(errorModel),
                                                                                                 stereotypeModel,
                                                                                                 singleton(modelProperty),
                                                                                                 deprecationModel,
                                                                                                 singleton("someSemanticTerm"));

    final ImmutableNestedRouteModel nestedRouteModel = new ImmutableNestedRouteModel(
                                                                                     "name",
                                                                                     "description",
                                                                                     parameterGroupModels,
                                                                                     displayModel,
                                                                                     10,
                                                                                     99,
                                                                                     singletonList(nestedComponentModel),
                                                                                     stereotypeModel,
                                                                                     singleton(modelProperty),
                                                                                     null,
                                                                                     emptySet());

    ImmutableConstructModel constructModel = new ImmutableConstructModel(CONSTRUCT_NAME,
                                                                         CONSTRUCT_DESCRIPTION,
                                                                         emptyList(),
                                                                         ImmutableList.of(nestedRouteModel),
                                                                         true,
                                                                         null,
                                                                         newHashSet(errorModel),
                                                                         stereotypeModel,
                                                                         emptySet(),
                                                                         null);

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);
    NestableElementModel nestedRouteModelDTO = constructModelDTO.getNestedComponents().get(0);

    assertThat(nestedRouteModelDTO, is(NestableElementMatcher.from(nestedRouteModel)));
  }

  @Test
  public void testFunctionModel() {
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertThat(dto.getFunctionModels(), hasSize(1));
    FunctionModel functionDto = dto.getFunctionModels().get(0);
    assertThat(functionDto.getName(), is(FUNCTION_NAME));
    assertThat(functionDto.getDescription(), is(EXTENSION_DESCRIPTION));
    assertThat(functionDto.getDisplayModel().get().getDisplayName(), is(DISPLAY_NAME));
    assertThat(functionDto.getOutput().getType(), is(stringType));
    assertThat(functionDto.getParameterGroupModels(), hasSize(1));
    assertTrue(functionDto.getDeprecationModel().isEnabled());
    assertThat(functionDto.getDeprecationModel().get().getMessage(), is("This is deprecated"));
    assertThat(functionDto.getDeprecationModel().get().getDeprecatedSince(), is("1.2.0"));
    assertTrue(functionDto.getDeprecationModel().get().getToRemoveIn().isPresent());
    assertThat(functionDto.getDeprecationModel().get().getToRemoveIn().get(), is("2.0.0"));
    ParameterGroupModel parameterGroupModel = functionDto.getParameterGroupModels().get(0);
    assertThat(parameterGroupModel.getName(), is(SIMPLE_PARAMETER_GROUP_NAME));
    assertThat(parameterGroupModel.getParameterModels(), hasSize(1));
    assertThat(parameterGroupModel.getParameterModels().get(0).getName(), is(SIMPLE_PARAMETER_NAME));
  }

  @Test
  public void sampleDataProviderModelInOperation() {
    String operationModelName = "op1";
    org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationModelName);
    org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel runtimeSampleDataProviderModel =
        new org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel(asActingParameterModels(asList("a", "b")),
                                                                                "providerId", true, false);
    when(operationModel.getSampleDataProviderModel()).thenReturn(of(runtimeSampleDataProviderModel));

    when(runtimeExtensionModel.getOperationModels()).thenReturn(asList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertThat(dto.getOperationModel(operationModelName), not(empty()));
    org.mule.tooling.client.api.extension.model.operation.OperationModel toolingOperationModel =
        dto.getOperationModel(operationModelName).get();
    org.mule.tooling.client.api.feature.Feature<org.mule.tooling.client.api.extension.model.data.sample.SampleDataProviderModel> toolingSampleDataProviderModelFeature =
        toolingOperationModel.getSampleDataProviderModel();
    assertThat(toolingSampleDataProviderModelFeature.isEnabled(), is(true));
    assertSampleDataProviderModel(runtimeSampleDataProviderModel, toolingSampleDataProviderModelFeature.get());
  }

  @Test
  public void sampleDataProviderModelInSource() {
    org.mule.runtime.api.meta.model.source.SourceModel sourceModel =
        mock(org.mule.runtime.api.meta.model.source.SourceModel.class);
    String sourceModelName = "op1";
    when(sourceModel.getName()).thenReturn(sourceModelName);
    org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel runtimeSampleDataProviderModel =
        new org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel(asActingParameterModels(asList("a", "b")),
                                                                                "providerId", true, false);
    when(sourceModel.getSampleDataProviderModel()).thenReturn(of(runtimeSampleDataProviderModel));

    OutputModel outputModel = mock(OutputModel.class);
    when(sourceModel.getOutput()).thenReturn(outputModel);
    when(sourceModel.getOutputAttributes()).thenReturn(outputModel);

    when(runtimeExtensionModel.getSourceModels()).thenReturn(asList(sourceModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertThat(dto.getSourceModel(sourceModelName), not(empty()));
    org.mule.tooling.client.api.extension.model.source.SourceModel toolingSourceModel =
        dto.getSourceModel(sourceModelName).get();
    org.mule.tooling.client.api.feature.Feature<org.mule.tooling.client.api.extension.model.data.sample.SampleDataProviderModel> toolingSampleDataProviderModelFeature =
        toolingSourceModel.getSampleDataProviderModel();
    assertThat(toolingSampleDataProviderModelFeature.isEnabled(), is(true));
    assertSampleDataProviderModel(runtimeSampleDataProviderModel, toolingSampleDataProviderModelFeature.get());
  }

  private void assertSampleDataProviderModel(org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel runtimeSampleDataProviderModel,
                                             org.mule.tooling.client.api.extension.model.data.sample.SampleDataProviderModel sampleDataProviderModel) {
    assertThat(sampleDataProviderModel.getParameters(),
               equalTo(toActingParameterModelsDTO(runtimeSampleDataProviderModel.getParameters())));
    assertThat(sampleDataProviderModel.getProviderId(), equalTo(runtimeSampleDataProviderModel.getProviderId()));
    assertThat(sampleDataProviderModel.requiresConfiguration(), equalTo(runtimeSampleDataProviderModel.requiresConfiguration()));
    assertThat(sampleDataProviderModel.requiresConnection(), equalTo(runtimeSampleDataProviderModel.requiresConnection()));
  }

  @Test
  public void testValuesResolverModelForValueProviderMultiLevel() {
    doTestValuesResolverModelForValueProviderMultiLevel(true);
  }

  @Test
  public void testValuesResolverModelForValueProvider() {
    doTestValuesResolverModelForValueProviderMultiLevel(false);
  }

  private void doTestValuesResolverModelForValueProviderMultiLevel(boolean multiLevel) {
    String operationModelName = "op1";
    org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationModelName);

    when(operationModel.getModelProperty(TypeResolversInformationModelProperty.class)).thenReturn(empty());
    List<org.mule.runtime.api.meta.model.parameter.ParameterModel> parameterModels = new ArrayList<>();

    List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels = new ArrayList<>();
    org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.class);
    String parameterGroupName = "Wsdl";
    when(parameterGroupModel.getName()).thenReturn(parameterGroupName);

    org.mule.runtime.api.meta.model.parameter.ParameterModel serviceParameter =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(serviceParameter);
    when(serviceParameter.getExpressionSupport()).thenReturn(SUPPORTED);
    when(serviceParameter.getRole()).thenReturn(CONTENT);
    when(serviceParameter.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String serviceParameterName = "service";
    when(serviceParameter.getName()).thenReturn(serviceParameterName);
    String providerId = "wsdlProviderId";
    when(serviceParameter.getValueProviderModel())
        .thenReturn(of(new ValueProviderModel(asActingParameterModels(asList("wsdlLocation")), false, true, true, 1,
                                              parameterGroupName,
                                              providerId)));

    String portParameterName = "port";
    String addressParameterName = "address";
    if (multiLevel) {
      org.mule.runtime.api.meta.model.parameter.ParameterModel portParameter =
          mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
      parameterModels.add(portParameter);
      when(portParameter.getExpressionSupport()).thenReturn(SUPPORTED);
      when(portParameter.getRole()).thenReturn(CONTENT);
      when(portParameter.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
      when(portParameter.getName()).thenReturn(portParameterName);
      when(portParameter.getValueProviderModel())
          .thenReturn(of(new ValueProviderModel(asActingParameterModels(asList("wsdlLocation")), false, true, true, 2,
                                                parameterGroupName,
                                                providerId)));

      org.mule.runtime.api.meta.model.parameter.ParameterModel addressParameter =
          mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
      parameterModels.add(addressParameter);
      when(addressParameter.getExpressionSupport()).thenReturn(SUPPORTED);
      when(addressParameter.getRole()).thenReturn(CONTENT);
      when(addressParameter.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
      when(addressParameter.getName()).thenReturn(addressParameterName);
      when(addressParameter.getValueProviderModel())
          .thenReturn(of(new ValueProviderModel(asActingParameterModels(asList("wsdlLocation")), false, true, true, 3,
                                                parameterGroupName,
                                                providerId)));
    }

    org.mule.runtime.api.meta.model.parameter.ParameterModel configParameter =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(configParameter);
    when(configParameter.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(configParameter.getRole()).thenReturn(CONTENT);
    when(configParameter.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    when(configParameter.getName()).thenReturn("configuration-reference");
    when(configParameter.getAllowedStereotypes())
        .thenReturn(asList(newStereotype("testConfigStereotype", "test").withParent(CONFIG).build()));

    when(parameterGroupModel.getParameterModels()).thenReturn(parameterModels);
    parameterGroupModels.add(parameterGroupModel);

    when(operationModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(operationModel.getAllParameterModels()).thenReturn(parameterModels);

    when(runtimeExtensionModel.getOperationModels()).thenReturn(asList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertThat(dto.getOperationModel(operationModelName), not(empty()));
    org.mule.tooling.client.api.extension.model.operation.OperationModel toolingOperationModel =
        dto.getOperationModel(operationModelName).get();
    assertThat(toolingOperationModel.getParameterGroupModel(parameterGroupName), not(empty()));

    assertValuesResolverModel(serviceParameterName, toolingOperationModel.getParameterGroupModel(parameterGroupName).get(),
                              asActingParameterModels(asList("wsdlLocation")), true, false, true,
                              valueProviderResolverName(providerId));
    if (multiLevel) {
      assertValuesResolverModel(portParameterName, toolingOperationModel.getParameterGroupModel(parameterGroupName).get(),
                                asActingParameterModels(asList("wsdlLocation", "service")), true, false, true,
                                valueProviderResolverName(providerId));
      assertValuesResolverModel(addressParameterName, toolingOperationModel.getParameterGroupModel(parameterGroupName).get(),
                                asActingParameterModels(asList("wsdlLocation", "port", "service")), true, false, true,
                                valueProviderResolverName(providerId));
    }
  }

  @Test
  public void testDataValueProviderModelOnConstructModels() {
    final StereotypeModel stereotypeModel = StereotypeModelBuilder.newStereotype("config", "test").build();

    org.mule.runtime.api.meta.model.error.ErrorModel errorModel =
        new ImmutableErrorModel(ERROR_MODEL_TYPE, ERROR_MODEL_NAMESPACE, true, null);

    org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.class);
    String parameterGroupModelName = "parameterGroupModel";
    when(parameterGroupModel.getName()).thenReturn(parameterGroupModelName);
    org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    when(parameterModel.getExpressionSupport()).thenReturn(SUPPORTED);
    when(parameterModel.getRole()).thenReturn(CONTENT);
    when(parameterModel.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String parameterName = "parameter";
    when(parameterModel.getName()).thenReturn(parameterName);
    String providerName = "service";
    String providerId = "serviceProviderId";
    when(parameterModel.getValueProviderModel())
        .thenReturn(of(new ValueProviderModel(asActingParameterModels(emptyList()), false, true, false, 1, providerName,
                                              providerId)));
    when(parameterGroupModel.getParameterModels()).thenReturn(asList(parameterModel));
    ImmutableConstructModel constructModel = new ImmutableConstructModel(CONSTRUCT_NAME,
                                                                         CONSTRUCT_DESCRIPTION,
                                                                         asList(parameterGroupModel),
                                                                         emptyList(),
                                                                         true,
                                                                         null,
                                                                         newHashSet(errorModel),
                                                                         stereotypeModel,
                                                                         emptySet(),
                                                                         new ImmutableDeprecationModel("Deprecated construct",
                                                                                                       "1.4.0", null));

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);
    Optional<ParameterGroupModel> parameterGroupModelDTO = constructModelDTO.getParameterGroupModel(parameterGroupModelName);
    assertThat(parameterGroupModelDTO, not(empty()));

    assertValuesResolverModel(parameterName, parameterGroupModelDTO.get(), emptyList(), false, false, true,
                              valueProviderResolverName(providerId));
  }

  @Test
  public void testValuesResolverModelForMetadataKeyPartsWithRequiresConfigurationWithConnection() {
    doTestValuesResolverModelForMetadataKeyParts(true, true);
  }

  @Test
  public void testValuesResolverModelForMetadataKeyPartsWithRequiresConfigurationWithoutConnection() {
    doTestValuesResolverModelForMetadataKeyParts(true, false);
  }

  @Test
  public void tesValuesResolverModelForMetadataKeyPartsWithoutRequiresConfigurationWithConnection() {
    doTestValuesResolverModelForMetadataKeyParts(false, true);
  }

  @Test
  public void tesValuesResolverModelForMetadataKeyPartsWithoutRequiresConfigurationWithoutConnection() {
    doTestValuesResolverModelForMetadataKeyParts(false, false);
  }

  @Test
  public void testValuesResolverModelForMetadataKeyPartsFirstLevelNotProvidedByKeyResolver() {
    String operationModelName = "op1";
    org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationModelName);

    String category = "category";
    MetadataKeyIdModelProperty metadataKeyIdModelProperty = new MetadataKeyIdModelProperty(stringType, "pojo", category);
    when(operationModel.getModelProperty(MetadataKeyIdModelProperty.class)).thenReturn(of(metadataKeyIdModelProperty));

    String keysResolver = "keysResolver";
    TypeResolversInformationModelProperty typeResolverModelProperty = new TypeResolversInformationModelProperty(category,
                                                                                                                emptyMap(),
                                                                                                                "outputResolver",
                                                                                                                "attributesResolver",
                                                                                                                keysResolver,
                                                                                                                true,
                                                                                                                false);
    when(operationModel.getModelProperty(TypeResolversInformationModelProperty.class)).thenReturn(of(typeResolverModelProperty));
    List<org.mule.runtime.api.meta.model.parameter.ParameterModel> parameterModels = new ArrayList<>();

    org.mule.runtime.api.meta.model.parameter.ParameterModel partOrder1Param =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(partOrder1Param);
    when(partOrder1Param.getExpressionSupport()).thenReturn(SUPPORTED);
    when(partOrder1Param.getRole()).thenReturn(CONTENT);
    when(partOrder1Param.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String parameterPart1Name = "part1";
    when(partOrder1Param.getName()).thenReturn(parameterPart1Name);
    MetadataKeyPartModelProperty part1MetadataKeyPartModelProperty = new MetadataKeyPartModelProperty(1, false);
    when(partOrder1Param.getModelProperty(MetadataKeyPartModelProperty.class)).thenReturn(of(part1MetadataKeyPartModelProperty));

    org.mule.runtime.api.meta.model.parameter.ParameterModel partOrder2Param =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(partOrder2Param);
    when(partOrder2Param.getExpressionSupport()).thenReturn(SUPPORTED);
    when(partOrder2Param.getRole()).thenReturn(CONTENT);
    when(partOrder2Param.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String parameterPart2Name = "part2";
    when(partOrder2Param.getName()).thenReturn(parameterPart2Name);
    MetadataKeyPartModelProperty part2MetadataKeyPartModelProperty = new MetadataKeyPartModelProperty(2, true);
    when(partOrder2Param.getModelProperty(MetadataKeyPartModelProperty.class)).thenReturn(of(part2MetadataKeyPartModelProperty));

    org.mule.runtime.api.meta.model.parameter.ParameterModel partOrder3Param =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(partOrder3Param);
    when(partOrder3Param.getExpressionSupport()).thenReturn(SUPPORTED);
    when(partOrder3Param.getRole()).thenReturn(CONTENT);
    when(partOrder3Param.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String parameterPart3Name = "part3";
    when(partOrder3Param.getName()).thenReturn(parameterPart3Name);
    MetadataKeyPartModelProperty part3MetadataKeyPartModelProperty = new MetadataKeyPartModelProperty(3, true);
    when(partOrder3Param.getModelProperty(MetadataKeyPartModelProperty.class)).thenReturn(of(part3MetadataKeyPartModelProperty));

    List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels = new ArrayList<>();
    org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.class);
    String parameterGroupName = "GeneralGroup";
    when(parameterGroupModel.getName()).thenReturn(parameterGroupName);
    when(parameterGroupModel.getParameterModels()).thenReturn(parameterModels);
    parameterGroupModels.add(parameterGroupModel);

    when(operationModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(operationModel.getAllParameterModels()).thenReturn(parameterModels);

    when(runtimeExtensionModel.getOperationModels()).thenReturn(asList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertThat(dto.getOperationModel(operationModelName), not(empty()));
    org.mule.tooling.client.api.extension.model.operation.OperationModel toolingOperationModel =
        dto.getOperationModel(operationModelName).get();
    assertThat(toolingOperationModel.getParameterGroupModel(parameterGroupName), not(empty()));

    assertThat(toolingOperationModel
        .getParameterGroupModel(parameterGroupName).get()
        .getParameterModel(parameterPart1Name).get()
        .getValuesResolverModel().get(), is(equalTo(null)));
    assertValuesResolverModel(parameterPart2Name, toolingOperationModel.getParameterGroupModel(parameterGroupName).get(),
                              asActingParameterModels(asList(parameterPart1Name)), false, false, true,
                              metadataKeyResolverName(category, keysResolver));
    assertValuesResolverModel(parameterPart3Name, toolingOperationModel.getParameterGroupModel(parameterGroupName).get(),
                              asActingParameterModels(asList(parameterPart2Name, parameterPart1Name)), false, false, true,
                              metadataKeyResolverName(category, keysResolver));
  }

  @Test
  public void testValuesResolverModelForMetadataKeyPartsPreviousLevelsNotProvidedByKeyResolver() {
    String operationModelName = "op1";
    org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationModelName);

    String category = "category";
    MetadataKeyIdModelProperty metadataKeyIdModelProperty = new MetadataKeyIdModelProperty(stringType, "pojo", category);
    when(operationModel.getModelProperty(MetadataKeyIdModelProperty.class)).thenReturn(of(metadataKeyIdModelProperty));

    String keysResolver = "keysResolver";
    TypeResolversInformationModelProperty typeResolverModelProperty = new TypeResolversInformationModelProperty(category,
                                                                                                                emptyMap(),
                                                                                                                "outputResolver",
                                                                                                                "attributesResolver",
                                                                                                                keysResolver,
                                                                                                                true,
                                                                                                                false);
    when(operationModel.getModelProperty(TypeResolversInformationModelProperty.class)).thenReturn(of(typeResolverModelProperty));
    List<org.mule.runtime.api.meta.model.parameter.ParameterModel> parameterModels = new ArrayList<>();

    org.mule.runtime.api.meta.model.parameter.ParameterModel partOrder1Param =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(partOrder1Param);
    when(partOrder1Param.getExpressionSupport()).thenReturn(SUPPORTED);
    when(partOrder1Param.getRole()).thenReturn(CONTENT);
    when(partOrder1Param.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String parameterPart1Name = "part1";
    when(partOrder1Param.getName()).thenReturn(parameterPart1Name);
    MetadataKeyPartModelProperty part1MetadataKeyPartModelProperty = new MetadataKeyPartModelProperty(1, false);
    when(partOrder1Param.getModelProperty(MetadataKeyPartModelProperty.class)).thenReturn(of(part1MetadataKeyPartModelProperty));

    org.mule.runtime.api.meta.model.parameter.ParameterModel partOrder2Param =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(partOrder2Param);
    when(partOrder2Param.getExpressionSupport()).thenReturn(SUPPORTED);
    when(partOrder2Param.getRole()).thenReturn(CONTENT);
    when(partOrder2Param.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String parameterPart2Name = "part2";
    when(partOrder2Param.getName()).thenReturn(parameterPart2Name);
    MetadataKeyPartModelProperty part2MetadataKeyPartModelProperty = new MetadataKeyPartModelProperty(2, false);
    when(partOrder2Param.getModelProperty(MetadataKeyPartModelProperty.class)).thenReturn(of(part2MetadataKeyPartModelProperty));

    org.mule.runtime.api.meta.model.parameter.ParameterModel partOrder3Param =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(partOrder3Param);
    when(partOrder3Param.getExpressionSupport()).thenReturn(SUPPORTED);
    when(partOrder3Param.getRole()).thenReturn(CONTENT);
    when(partOrder3Param.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String parameterPart3Name = "part3";
    when(partOrder3Param.getName()).thenReturn(parameterPart3Name);
    MetadataKeyPartModelProperty part3MetadataKeyPartModelProperty = new MetadataKeyPartModelProperty(3, true);
    when(partOrder3Param.getModelProperty(MetadataKeyPartModelProperty.class)).thenReturn(of(part3MetadataKeyPartModelProperty));

    List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels = new ArrayList<>();
    org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.class);
    String parameterGroupName = "GeneralGroup";
    when(parameterGroupModel.getName()).thenReturn(parameterGroupName);
    when(parameterGroupModel.getParameterModels()).thenReturn(parameterModels);
    parameterGroupModels.add(parameterGroupModel);

    when(operationModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(operationModel.getAllParameterModels()).thenReturn(parameterModels);

    when(runtimeExtensionModel.getOperationModels()).thenReturn(asList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertThat(dto.getOperationModel(operationModelName), not(empty()));
    org.mule.tooling.client.api.extension.model.operation.OperationModel toolingOperationModel =
        dto.getOperationModel(operationModelName).get();
    assertThat(toolingOperationModel.getParameterGroupModel(parameterGroupName), not(empty()));

    assertThat(toolingOperationModel
        .getParameterGroupModel(parameterGroupName).get()
        .getParameterModel(parameterPart1Name).get()
        .getValuesResolverModel().get(), is(equalTo(null)));
    assertThat(toolingOperationModel
        .getParameterGroupModel(parameterGroupName).get()
        .getParameterModel(parameterPart2Name).get()
        .getValuesResolverModel().get(), is(equalTo(null)));
    assertValuesResolverModel(parameterPart3Name, toolingOperationModel.getParameterGroupModel(parameterGroupName).get(),
                              asActingParameterModels(asList(parameterPart2Name, parameterPart1Name)), false, false, true,
                              metadataKeyResolverName(category, keysResolver));
  }

  private void doTestValuesResolverModelForMetadataKeyParts(boolean requiresConfiguration, boolean requiresConnection) {
    String operationModelName = "op1";
    org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationModelName);

    String category = "category";
    MetadataKeyIdModelProperty metadataKeyIdModelProperty = new MetadataKeyIdModelProperty(stringType, "pojo", category);
    when(operationModel.getModelProperty(MetadataKeyIdModelProperty.class)).thenReturn(of(metadataKeyIdModelProperty));

    String keysResolver = "keysResolver";
    TypeResolversInformationModelProperty typeResolverModelProperty = new TypeResolversInformationModelProperty(category,
                                                                                                                emptyMap(),
                                                                                                                "outputResolver",
                                                                                                                "attributesResolver",
                                                                                                                keysResolver,
                                                                                                                requiresConnection,
                                                                                                                requiresConfiguration);
    when(operationModel.getModelProperty(TypeResolversInformationModelProperty.class)).thenReturn(of(typeResolverModelProperty));
    List<org.mule.runtime.api.meta.model.parameter.ParameterModel> parameterModels = new ArrayList<>();

    org.mule.runtime.api.meta.model.parameter.ParameterModel partOrder1Param =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(partOrder1Param);
    when(partOrder1Param.getExpressionSupport()).thenReturn(SUPPORTED);
    when(partOrder1Param.getRole()).thenReturn(CONTENT);
    when(partOrder1Param.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String parameterPart1Name = "part1";
    when(partOrder1Param.getName()).thenReturn(parameterPart1Name);
    MetadataKeyPartModelProperty part1MetadataKeyPartModelProperty = new MetadataKeyPartModelProperty(1, true);
    when(partOrder1Param.getModelProperty(MetadataKeyPartModelProperty.class)).thenReturn(of(part1MetadataKeyPartModelProperty));

    org.mule.runtime.api.meta.model.parameter.ParameterModel partOrder2Param =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(partOrder2Param);
    when(partOrder2Param.getExpressionSupport()).thenReturn(SUPPORTED);
    when(partOrder2Param.getRole()).thenReturn(CONTENT);
    when(partOrder2Param.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String parameterPart2Name = "part2";
    when(partOrder2Param.getName()).thenReturn(parameterPart2Name);
    MetadataKeyPartModelProperty part2MetadataKeyPartModelProperty = new MetadataKeyPartModelProperty(2, true);
    when(partOrder2Param.getModelProperty(MetadataKeyPartModelProperty.class)).thenReturn(of(part2MetadataKeyPartModelProperty));

    org.mule.runtime.api.meta.model.parameter.ParameterModel partOrder3Param =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(partOrder3Param);
    when(partOrder3Param.getExpressionSupport()).thenReturn(SUPPORTED);
    when(partOrder3Param.getRole()).thenReturn(CONTENT);
    when(partOrder3Param.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String parameterPart3Name = "part3";
    when(partOrder3Param.getName()).thenReturn(parameterPart3Name);
    MetadataKeyPartModelProperty part3MetadataKeyPartModelProperty = new MetadataKeyPartModelProperty(3, true);
    when(partOrder3Param.getModelProperty(MetadataKeyPartModelProperty.class)).thenReturn(of(part3MetadataKeyPartModelProperty));

    org.mule.runtime.api.meta.model.parameter.ParameterModel configParameter =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    parameterModels.add(configParameter);
    when(configParameter.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(configParameter.getRole()).thenReturn(CONTENT);
    when(configParameter.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    when(configParameter.getName()).thenReturn("configuration-reference");
    if (requiresConfiguration) {
      when(configParameter.getAllowedStereotypes())
          .thenReturn(asList(newStereotype("testConfigStereotype", "test").withParent(CONFIG).build()));
    } else {
      when(configParameter.getAllowedStereotypes()).thenReturn(emptyList());
    }

    List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels = new ArrayList<>();
    org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.class);
    String parameterGroupName = "GeneralGroup";
    when(parameterGroupModel.getName()).thenReturn(parameterGroupName);
    when(parameterGroupModel.getParameterModels()).thenReturn(parameterModels);
    parameterGroupModels.add(parameterGroupModel);

    when(operationModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(operationModel.getAllParameterModels()).thenReturn(parameterModels);

    when(runtimeExtensionModel.getOperationModels()).thenReturn(asList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertThat(dto.getOperationModel(operationModelName), not(empty()));
    org.mule.tooling.client.api.extension.model.operation.OperationModel toolingOperationModel =
        dto.getOperationModel(operationModelName).get();
    assertThat(toolingOperationModel.getParameterGroupModel(parameterGroupName), not(empty()));

    assertValuesResolverModel(parameterPart1Name, toolingOperationModel.getParameterGroupModel(parameterGroupName).get(),
                              emptyList(), false, requiresConfiguration, requiresConnection,
                              metadataKeyResolverName(category, keysResolver));
    assertValuesResolverModel(parameterPart2Name, toolingOperationModel.getParameterGroupModel(parameterGroupName).get(),
                              asActingParameterModels(asList(parameterPart1Name)), false, requiresConfiguration,
                              requiresConnection,
                              metadataKeyResolverName(category, keysResolver));
    assertValuesResolverModel(parameterPart3Name, toolingOperationModel.getParameterGroupModel(parameterGroupName).get(),
                              asActingParameterModels(asList(parameterPart2Name, parameterPart1Name)), false,
                              requiresConfiguration,
                              requiresConnection,
                              metadataKeyResolverName(category, keysResolver));
  }

  private void assertValuesResolverModel(String parameterName, ParameterGroupModel parameterGroupModel,
                                         List<org.mule.runtime.api.meta.model.parameter.ActingParameterModel> parameters,
                                         boolean isOpen,
                                         boolean requiresConfiguration,
                                         boolean requiresConnection,
                                         String resolverName) {
    assertThat(parameterGroupModel.getParameterModel(parameterName), not(empty()));
    ParameterModel parameterModel = parameterGroupModel.getParameterModel(parameterName).get();
    org.mule.tooling.client.api.feature.Feature<ValuesResolverModel> dataValueProviderModelOptional =
        parameterModel.getValuesResolverModel();
    assertThat(dataValueProviderModelOptional, not(empty()));
    ValuesResolverModel dataValueProviderModel = dataValueProviderModelOptional.get();
    assertThat(dataValueProviderModel, not(nullValue()));
    if (parameters.isEmpty()) {
      assertThat(dataValueProviderModel.getParameters(), is(Matchers.empty()));
    } else {
      assertThat(dataValueProviderModel.getParameters(),
                 contains(ActingParameterMatcher.sFrom(parameters)));
    }
    assertThat(dataValueProviderModel.isOpen(), is(isOpen));
    assertThat(dataValueProviderModel.requiresConfiguration(), is(requiresConfiguration));
    assertThat(dataValueProviderModel.requiresConnection(), is(requiresConnection));
    assertThat(dataValueProviderModel.getResolverName(), is(resolverName));
  }

  @Test
  public void testConfigModel() {
    ConfigurationModel configModel = mock(ConfigurationModel.class, withSettings().lenient());
    when(configModel.getName()).thenReturn(CONFIGURATION_NAME);
    when(configModel.getDescription()).thenReturn(CONFIGURATION_DESCRIPTION);
    when(configModel.getDisplayModel()).thenReturn(of(displayModel));
    when(configModel.getDeprecationModel()).thenReturn(empty());

    org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class, withSettings().lenient());
    when(parameterModel.getName()).thenReturn("parameter");
    when(parameterModel.getExpressionSupport()).thenReturn(SUPPORTED);
    when(parameterModel.getRole()).thenReturn(CONTENT);
    when(parameterModel.getDslConfiguration()).thenReturn(ParameterDslConfiguration.builder().build());
    String providerName = "parameter";
    String providerId = "parameterProviderId";
    ValueProviderModel valueProviderModel =
        new ValueProviderModel(asActingParameterModels(emptyList()), false, true, true, 1, providerName, providerId);
    when(parameterModel.getValueProviderModel()).thenReturn(of(valueProviderModel));
    org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroup =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.class, withSettings().lenient());
    when(parameterGroup.getName()).thenReturn("parameterGroup");
    when(parameterGroup.getParameterModels()).thenReturn(asList(parameterModel));

    when(configModel.getParameterGroupModels()).thenReturn(asList(parameterGroup));

    ConnectionProviderModel connectionProvider = mock(ConnectionProviderModel.class);
    when(connectionProvider.getName()).thenReturn(CONNECTION_PROVIDER_NAME);
    when(connectionProvider.getDescription()).thenReturn(CONNECTION_PROVIDER_DESCRIPTION);
    when(connectionProvider.getDisplayModel()).thenReturn(of(displayModel));
    when(connectionProvider.getConnectionManagementType()).thenReturn(ConnectionManagementType.POOLING);
    when(connectionProvider.getDeprecationModel())
        .thenReturn(of(new ImmutableDeprecationModel("This is a deprecated connection provider", "2.2.0", "3.0.0")));
    when(connectionProvider.getParameterGroupModels()).thenReturn(asList(parameterGroup));

    externalLibraryModel = ExternalLibraryModel.builder()
        .withName(EXTERNAL_LIB_MODEL_NAME)
        .withDescription(EXTERNAL_LIB_MODEL_DESCRIPTION)
        .withRegexpMatcher(EXTERNAL_LIB_MODEL_REGEXP_MATCHER)
        .withType(ExternalLibraryType.JAR)
        .withRequiredClassName(String.class.getName())
        .build();
    when(connectionProvider.getExternalLibraryModels()).thenReturn(newHashSet(externalLibraryModel));

    when(configModel.getConnectionProviders()).thenReturn(newArrayList(connectionProvider));

    when(runtimeExtensionModel.getConfigurationModels()).thenReturn(newArrayList(configModel));
    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertExtensionModelBasicAttributes(dto);

    assertThat(dto.getConfigurationModels(), hasSize(1));
    final org.mule.tooling.client.api.extension.model.config.ConfigurationModel configurationModel =
        dto.getConfigurationModels().get(0);
    assertThat(configurationModel.getName(), is(CONFIGURATION_NAME));
    assertThat(configurationModel.getDescription(), is(CONFIGURATION_DESCRIPTION));
    assertDisplayModel(configurationModel.getDisplayModel());
    ParameterGroupModel parameterGroupModel = configurationModel.getParameterGroupModel("parameterGroup")
        .orElseThrow(() -> new AssertionError("ParameterGroup not found"));
    assertValuesResolverModel("parameter", parameterGroupModel, emptyList(), true, false,
                              true, valueProviderResolverName(providerId));

    assertThat(configurationModel.getConnectionProviders(), hasSize(1));
    final org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel connectionProviderModel =
        configurationModel.getConnectionProviders().get(0);
    assertThat(connectionProviderModel.getName(), is(CONNECTION_PROVIDER_NAME));
    assertThat(connectionProviderModel.getDescription(), is(CONNECTION_PROVIDER_DESCRIPTION));
    assertDisplayModel(connectionProviderModel.getDisplayModel());
    assertThat(connectionProviderModel.getConnectionManagementType().isPooling(), is(true));
    assertThat(connectionProviderModel.getConnectionManagementType().isNone(), is(false));
    assertThat(connectionProviderModel.getConnectionManagementType().isCached(), is(false));
    assertThat(connectionProviderModel.getConnectionManagementType().isUnknown(), is(false));
    parameterGroupModel = connectionProviderModel.getParameterGroupModel("parameterGroup")
        .orElseThrow(() -> new AssertionError("ParameterGroup not found"));
    assertValuesResolverModel("parameter", parameterGroupModel, emptyList(), true, false,
                              true, valueProviderResolverName(providerId));

    assertThat(connectionProviderModel.getExternalLibraryModels(), hasSize(1));
    final org.mule.tooling.client.api.extension.model.ExternalLibraryModel externalLibraryModel =
        connectionProviderModel.getExternalLibraryModels().iterator().next();
    assertThat(externalLibraryModel.getName(), is(EXTERNAL_LIB_MODEL_NAME));
    assertThat(externalLibraryModel.getDescription(), is(EXTERNAL_LIB_MODEL_DESCRIPTION));
    assertThat(externalLibraryModel.getRegexMatcher().get(), is(EXTERNAL_LIB_MODEL_REGEXP_MATCHER));
    assertThat(externalLibraryModel.getRequiredClassName().get(), is(String.class.getName()));
    assertThat(externalLibraryModel.getType().isJar(), is(true));
    assertThat(externalLibraryModel.getType().isDependency(), is(false));
    assertThat(externalLibraryModel.getType().isNativeLibrary(), is(false));
    assertThat(externalLibraryModel.getType().isUnknown(), is(false));
  }

  @Test
  public void simpleFieldValueResolverModelFromFieldValueProvider() {
    final String fieldProviderName = "fieldProvider";
    final String targetSelector = "some.target.field";

    final String actingParameterName = "actingParameter";
    final org.mule.runtime.api.meta.model.parameter.ActingParameterModel actingParameterModel =
        mockedActingParameterModel(actingParameterName);
    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel fieldValueProviderModel =
        mockedFieldValueProviderModel(fieldProviderName, targetSelector);
    when(fieldValueProviderModel.getParameters()).thenReturn(singletonList(actingParameterModel));
    final org.mule.runtime.api.meta.model.parameter.ParameterModel actingParameter = mockedParameterModel(actingParameterName);

    final String parameterName = "parameter";
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel = mockedParameterModel(parameterName);
    when(parameterModel.getFieldValueProviderModels()).thenReturn(singletonList(fieldValueProviderModel));

    final String parameterGroupName = "parameterGroup";
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mockedParameterGroupModel(parameterGroupName);

    when(parameterGroupModel.getParameterModels()).thenReturn(asList(parameterModel, actingParameter));

    final String operationName = "op1";
    final org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationName);
    when(operationModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    when(runtimeExtensionModel.getOperationModels()).thenReturn(singletonList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    OperationModel opModelDto = dto.getOperationModel(operationName).get();
    ParameterGroupModel parameterGroupModelDto = opModelDto.getParameterGroupModel(parameterGroupName).get();
    ParameterModel parameterModelDto = parameterGroupModelDto.getParameterModel(parameterName).get();

    List<FieldValuesResolverModel> fieldValuesResolverModels = parameterModelDto.getFieldValuesResolverModels();

    assertThat(fieldValuesResolverModels, hasSize(1));
    FieldValuesResolverModel fieldValuesResolverModel = fieldValuesResolverModels.get(0);
    assertThat(fieldValuesResolverModel.getResolverName(), is(valueProviderResolverName(fieldProviderName)));
    assertThat(fieldValuesResolverModel.getTargetSelector(), is(targetSelector));
    assertThat(fieldValuesResolverModel.getParameters(), hasSize(1));
    assertThat(fieldValuesResolverModel.getParameters(), hasItem(ActingParameterMatcher.from(actingParameterModel)));
    assertThat(fieldValuesResolverModel.requiresConfiguration(), is(false));
    assertThat(fieldValuesResolverModel.requiresConnection(), is(false));
    assertThat(fieldValuesResolverModel.isOpen(), is(false));
  }

  @Test
  public void multipleValueResolverModelFromFieldValueProviderModel() {
    final String fieldProviderName1 = "fieldProvider1";
    final String fieldProviderName2 = "fieldProvider2";
    final String targetSelector1 = "some.target.field";
    final String targetSelector2 = "some.other.target.field";


    final String parameterName = "parameter";
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel = mockedParameterModel(parameterName);

    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel fieldValueProviderModel1 =
        mockedFieldValueProviderModel(fieldProviderName1, targetSelector1);
    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel fieldValueProviderModel2 =
        mockedFieldValueProviderModel(fieldProviderName2, targetSelector2);
    when(parameterModel.getFieldValueProviderModels()).thenReturn(asList(fieldValueProviderModel1, fieldValueProviderModel2));

    final String parameterGroupName = "parameterGroup";
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mockedParameterGroupModel(parameterGroupName);

    when(parameterGroupModel.getParameterModels()).thenReturn(singletonList(parameterModel));
    final String operationName = "op1";
    final org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationName);
    when(operationModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    when(runtimeExtensionModel.getOperationModels()).thenReturn(singletonList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    OperationModel opModelDto = dto.getOperationModel(operationName).get();
    ParameterGroupModel parameterGroupModelDto = opModelDto.getParameterGroupModel(parameterGroupName).get();
    ParameterModel parameterModelDto = parameterGroupModelDto.getParameterModel(parameterName).get();

    List<FieldValuesResolverModel> fieldValuesResolverModels = parameterModelDto.getFieldValuesResolverModels();

    assertThat(fieldValuesResolverModels, hasSize(2));
    FieldValuesResolverModel fieldValuesResolverModel1 = fieldValuesResolverModels.get(0);
    assertThat(fieldValuesResolverModel1.getResolverName(), is(valueProviderResolverName(fieldProviderName1)));
    assertThat(fieldValuesResolverModel1.getTargetSelector(), is(targetSelector1));
    assertThat(fieldValuesResolverModel1.getParameters(), is(emptyList()));
    assertThat(fieldValuesResolverModel1.requiresConfiguration(), is(false));
    assertThat(fieldValuesResolverModel1.requiresConnection(), is(false));
    assertThat(fieldValuesResolverModel1.isOpen(), is(false));

    FieldValuesResolverModel fieldValuesResolverModel2 = fieldValuesResolverModels.get(1);
    assertThat(fieldValuesResolverModel2.getResolverName(), is(valueProviderResolverName(fieldProviderName2)));
    assertThat(fieldValuesResolverModel2.getTargetSelector(), is(targetSelector2));
    assertThat(fieldValuesResolverModel2.getParameters(), is(emptyList()));
    assertThat(fieldValuesResolverModel2.requiresConfiguration(), is(false));
    assertThat(fieldValuesResolverModel2.requiresConnection(), is(false));
    assertThat(fieldValuesResolverModel2.isOpen(), is(false));
  }

  @Test
  public void multiLevelFieldValuesResolverModelFromFieldValueProviderModel() {
    final String providerName = "Provider";

    final String part0TargetSelector = "some.target.field.part0";
    final String part1TargetSelector = "some.target.field.part1";
    final String part2TargetSelector = "some.target.field.part2";

    final String actingParameterName = "actingParameter";
    final org.mule.runtime.api.meta.model.parameter.ActingParameterModel actingParameterModel =
        mockedActingParameterModel(actingParameterName);
    final org.mule.runtime.api.meta.model.parameter.ParameterModel actingParameter = mockedParameterModel(actingParameterName);

    final String parameterName = "parameter";
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel = mockedParameterModel(parameterName);

    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel fieldValueProviderModel0 =
        mockedFieldValueProviderModel(providerName, part0TargetSelector);
    when(fieldValueProviderModel0.getPartOrder()).thenReturn(1);
    when(fieldValueProviderModel0.getParameters()).thenReturn(singletonList(actingParameterModel));
    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel fieldValueProviderModel1 =
        mockedFieldValueProviderModel(providerName, part1TargetSelector);
    when(fieldValueProviderModel1.getPartOrder()).thenReturn(2);
    when(fieldValueProviderModel1.getParameters()).thenReturn(singletonList(actingParameterModel));
    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel fieldValueProviderModel2 =
        mockedFieldValueProviderModel(providerName, part2TargetSelector);
    when(fieldValueProviderModel2.getPartOrder()).thenReturn(3);
    when(fieldValueProviderModel2.getParameters()).thenReturn(singletonList(actingParameterModel));

    when(parameterModel.getFieldValueProviderModels())
        .thenReturn(asList(fieldValueProviderModel0, fieldValueProviderModel1, fieldValueProviderModel2));

    final String parameterGroupName = "parameterGroup";
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mockedParameterGroupModel(parameterGroupName);

    when(parameterGroupModel.getParameterModels()).thenReturn(asList(parameterModel, actingParameter));
    final String operationName = "op1";
    final org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationName);
    when(operationModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    when(runtimeExtensionModel.getOperationModels()).thenReturn(singletonList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    OperationModel opModelDto = dto.getOperationModel(operationName).get();
    ParameterGroupModel parameterGroupModelDto = opModelDto.getParameterGroupModel(parameterGroupName).get();
    ParameterModel parameterModelDto = parameterGroupModelDto.getParameterModel(parameterName).get();

    List<FieldValuesResolverModel> fieldValuesResolverModels = parameterModelDto.getFieldValuesResolverModels();

    assertThat(fieldValuesResolverModels, hasSize(3));
    FieldValuesResolverModel fieldValuesResolverModel0 = fieldValuesResolverModels.get(0);
    assertThat(fieldValuesResolverModel0.getResolverName(), is(valueProviderResolverName(providerName)));
    assertThat(fieldValuesResolverModel0.getTargetSelector(), is(part0TargetSelector));
    assertThat(fieldValuesResolverModel0.getParameters(), hasSize(1));
    assertThat(fieldValuesResolverModel0.getParameters(), hasItem(ActingParameterMatcher.from(actingParameterModel)));
    assertThat(fieldValuesResolverModel0.requiresConfiguration(), is(false));
    assertThat(fieldValuesResolverModel0.requiresConnection(), is(false));
    assertThat(fieldValuesResolverModel0.isOpen(), is(false));

    org.mule.runtime.api.meta.model.parameter.ActingParameterModel expectedPart1ActingParameter =
        mockedActingParameterModel(part0TargetSelector);
    when(expectedPart1ActingParameter.getExtractionExpression()).thenReturn(join(".", parameterName, part0TargetSelector));
    FieldValuesResolverModel fieldValuesResolverModel1 = fieldValuesResolverModels.get(1);
    assertThat(fieldValuesResolverModel1.getResolverName(), is(valueProviderResolverName(providerName)));
    assertThat(fieldValuesResolverModel1.getTargetSelector(), is(part1TargetSelector));
    assertThat(fieldValuesResolverModel1.getParameters(), hasSize(2));
    assertThat(fieldValuesResolverModel1.getParameters(), contains(
                                                                   ActingParameterMatcher.from(actingParameterModel),
                                                                   ActingParameterMatcher.from(expectedPart1ActingParameter)));
    assertThat(fieldValuesResolverModel1.requiresConfiguration(), is(false));
    assertThat(fieldValuesResolverModel1.requiresConnection(), is(false));
    assertThat(fieldValuesResolverModel1.isOpen(), is(false));

    org.mule.runtime.api.meta.model.parameter.ActingParameterModel expectedPart2ActingParameter =
        mockedActingParameterModel(part1TargetSelector);
    when(expectedPart2ActingParameter.getExtractionExpression()).thenReturn(join(".", parameterName, part1TargetSelector));
    FieldValuesResolverModel fieldValuesResolverModel2 = fieldValuesResolverModels.get(2);
    assertThat(fieldValuesResolverModel2.getResolverName(), is(valueProviderResolverName(providerName)));
    assertThat(fieldValuesResolverModel2.getTargetSelector(), is(part2TargetSelector));
    assertThat(fieldValuesResolverModel2.getParameters(), hasSize(3));
    assertThat(fieldValuesResolverModel2.getParameters(), contains(
                                                                   ActingParameterMatcher.from(actingParameterModel),
                                                                   ActingParameterMatcher.from(expectedPart2ActingParameter),
                                                                   ActingParameterMatcher.from(expectedPart1ActingParameter)));
    assertThat(fieldValuesResolverModel2.requiresConfiguration(), is(false));
    assertThat(fieldValuesResolverModel2.requiresConnection(), is(false));
    assertThat(fieldValuesResolverModel2.isOpen(), is(false));
  }

  @Test
  public void simpleFieldValueProviderModel() {
    final String fieldProviderName = "fieldProvider";
    final String targetSelector = "some.target.field";

    final String actingParameterName = "actingParameter";
    final org.mule.runtime.api.meta.model.parameter.ActingParameterModel actingParameterModel =
        mockedActingParameterModel(actingParameterName);
    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel runtimeFieldValueProviderModel =
        mockedFieldValueProviderModel(fieldProviderName, targetSelector);
    when(runtimeFieldValueProviderModel.getParameters()).thenReturn(singletonList(actingParameterModel));
    final org.mule.runtime.api.meta.model.parameter.ParameterModel actingParameter = mockedParameterModel(actingParameterName);

    final String parameterName = "parameter";
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel = mockedParameterModel(parameterName);
    when(parameterModel.getFieldValueProviderModels()).thenReturn(singletonList(runtimeFieldValueProviderModel));

    final String parameterGroupName = "parameterGroup";
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mockedParameterGroupModel(parameterGroupName);

    when(parameterGroupModel.getParameterModels()).thenReturn(asList(parameterModel, actingParameter));

    final String operationName = "op1";
    final org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationName);
    when(operationModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    when(runtimeExtensionModel.getOperationModels()).thenReturn(singletonList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    OperationModel opModelDto = dto.getOperationModel(operationName).get();
    ParameterGroupModel parameterGroupModelDto = opModelDto.getParameterGroupModel(parameterGroupName).get();
    ParameterModel parameterModelDto = parameterGroupModelDto.getParameterModel(parameterName).get();

    List<FieldValueProviderModel> fieldValueProviderModels = parameterModelDto.getFieldValueProviderModels();

    assertThat(fieldValueProviderModels, hasSize(1));
    FieldValueProviderModel fieldValueProviderModel = fieldValueProviderModels.get(0);
    assertThat(fieldValueProviderModel.getProviderName(), is(fieldProviderName));
    assertThat(fieldValueProviderModel.getProviderId().isEnabled(), is(true));
    assertThat(fieldValueProviderModel.getProviderId().get(), is(fieldProviderName));
    assertThat(fieldValueProviderModel.getTargetSelector(), is(targetSelector));
    assertThat(fieldValueProviderModel.getParameters().isEnabled(), is(true));
    assertThat(fieldValueProviderModel.getParameters().get(), hasSize(1));
    assertThat(fieldValueProviderModel.getParameters().get(), hasItem(ActingParameterMatcher.from(actingParameterModel)));
    assertThat(fieldValueProviderModel.requiresConfiguration(), is(false));
    assertThat(fieldValueProviderModel.requiresConnection(), is(false));
    assertThat(fieldValueProviderModel.isOpen(), is(false));
    assertThat(fieldValueProviderModel.getPartOrder(), is(1));
  }

  @Test
  public void multipleFieldValueProviderModel() {
    final String fieldProviderName1 = valueProviderResolverName("fieldProvider1");
    final String fieldProviderName2 = valueProviderResolverName("fieldProvider2");
    final String targetSelector1 = "some.target.field";
    final String targetSelector2 = "some.other.target.field";


    final String parameterName = "parameter";
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel = mockedParameterModel(parameterName);

    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel runtimeFieldValueProviderModel1 =
        mockedFieldValueProviderModel(fieldProviderName1, targetSelector1);
    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel runtimeFieldValueProviderModel2 =
        mockedFieldValueProviderModel(fieldProviderName2, targetSelector2);
    when(parameterModel.getFieldValueProviderModels())
        .thenReturn(asList(runtimeFieldValueProviderModel1, runtimeFieldValueProviderModel2));

    final String parameterGroupName = "parameterGroup";
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mockedParameterGroupModel(parameterGroupName);

    when(parameterGroupModel.getParameterModels()).thenReturn(singletonList(parameterModel));
    final String operationName = "op1";
    final org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationName);
    when(operationModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    when(runtimeExtensionModel.getOperationModels()).thenReturn(singletonList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    OperationModel opModelDto = dto.getOperationModel(operationName).get();
    ParameterGroupModel parameterGroupModelDto = opModelDto.getParameterGroupModel(parameterGroupName).get();
    ParameterModel parameterModelDto = parameterGroupModelDto.getParameterModel(parameterName).get();

    List<FieldValueProviderModel> fieldValueProviderModels = parameterModelDto.getFieldValueProviderModels();

    assertThat(fieldValueProviderModels, hasSize(2));
    FieldValueProviderModel fieldValueProviderModel1 = fieldValueProviderModels.get(0);
    assertThat(fieldValueProviderModel1.getProviderName(), is(fieldProviderName1));
    assertThat(fieldValueProviderModel1.getProviderId().isEnabled(), is(true));
    assertThat(fieldValueProviderModel1.getProviderId().get(), is(fieldProviderName1));
    assertThat(fieldValueProviderModel1.getTargetSelector(), is(targetSelector1));
    assertThat(fieldValueProviderModel1.getParameters().isEnabled(), is(true));
    assertThat(fieldValueProviderModel1.getParameters().get(), is(emptyList()));
    assertThat(fieldValueProviderModel1.requiresConfiguration(), is(false));
    assertThat(fieldValueProviderModel1.requiresConnection(), is(false));
    assertThat(fieldValueProviderModel1.isOpen(), is(false));
    assertThat(fieldValueProviderModel1.getPartOrder(), is(1));

    FieldValueProviderModel fieldValueProviderModel2 = fieldValueProviderModels.get(1);
    assertThat(fieldValueProviderModel2.getProviderName(), is(fieldProviderName2));
    assertThat(fieldValueProviderModel2.getProviderId().isEnabled(), is(true));
    assertThat(fieldValueProviderModel2.getProviderId().get(), is(fieldProviderName2));
    assertThat(fieldValueProviderModel2.getTargetSelector(), is(targetSelector2));
    assertThat(fieldValueProviderModel2.getParameters().isEnabled(), is(true));
    assertThat(fieldValueProviderModel2.getParameters().get(), is(emptyList()));
    assertThat(fieldValueProviderModel2.requiresConfiguration(), is(false));
    assertThat(fieldValueProviderModel2.requiresConnection(), is(false));
    assertThat(fieldValueProviderModel2.isOpen(), is(false));
    assertThat(fieldValueProviderModel2.getPartOrder(), is(1));
  }

  @Test
  public void fieldValueProviderModelWithActingParameters() {
    final String providerName = "Provider";
    final String targetSelector = "some.target.field.part0";
    final String actingParameterName = "actingParameter";
    final org.mule.runtime.api.meta.model.parameter.ActingParameterModel actingParameterModel =
        mockedActingParameterModel(actingParameterName);
    final org.mule.runtime.api.meta.model.parameter.ParameterModel actingParameter = mockedParameterModel(actingParameterName);

    final String parameterName = "parameter";
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel = mockedParameterModel(parameterName);

    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel fieldValueProviderModel0 =
        mockedFieldValueProviderModel(providerName, targetSelector);
    when(fieldValueProviderModel0.getPartOrder()).thenReturn(1);
    when(fieldValueProviderModel0.getParameters()).thenReturn(singletonList(actingParameterModel));

    when(parameterModel.getFieldValueProviderModels()).thenReturn(asList(fieldValueProviderModel0));

    final String parameterGroupName = "parameterGroup";
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mockedParameterGroupModel(parameterGroupName);

    when(parameterGroupModel.getParameterModels()).thenReturn(asList(parameterModel, actingParameter));
    final String operationName = "op1";
    final org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationName);
    when(operationModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    when(runtimeExtensionModel.getOperationModels()).thenReturn(singletonList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    OperationModel opModelDto = dto.getOperationModel(operationName).get();
    ParameterGroupModel parameterGroupModelDto = opModelDto.getParameterGroupModel(parameterGroupName).get();
    ParameterModel parameterModelDto = parameterGroupModelDto.getParameterModel(parameterName).get();

    List<FieldValueProviderModel> fieldValueProviderModels = parameterModelDto.getFieldValueProviderModels();

    assertThat(fieldValueProviderModels, hasSize(1));
    FieldValueProviderModel fieldValueProviderModel = fieldValueProviderModels.get(0);
    assertThat(fieldValueProviderModel.getProviderName(), is(providerName));
    assertThat(fieldValueProviderModel.getProviderId().isEnabled(), is(true));
    assertThat(fieldValueProviderModel.getProviderId().get(), is(providerName));
    assertThat(fieldValueProviderModel.getTargetSelector(), is(targetSelector));
    assertThat(fieldValueProviderModel.getParameters().isEnabled(), is(true));
    assertThat(fieldValueProviderModel.getParameters().get(), hasSize(1));
    assertThat(fieldValueProviderModel.getParameters().get(), hasItem(ActingParameterMatcher.from(actingParameterModel)));
    assertThat(fieldValueProviderModel.requiresConfiguration(), is(false));
    assertThat(fieldValueProviderModel.requiresConnection(), is(false));
    assertThat(fieldValueProviderModel.isOpen(), is(false));
    assertThat(fieldValueProviderModel.getPartOrder(), is(1));
  }

  @Test
  public void sampleDataProviderModel() {
    final String providerId = "Provider";
    final String targetSelector = "some.target.field.part0";
    final String actingParameterName = "actingParameter";
    final org.mule.runtime.api.meta.model.parameter.ActingParameterModel actingParameterModel =
        mockedActingParameterModel(actingParameterName);
    when(actingParameterModel.getExtractionExpression()).thenReturn(targetSelector);
    final org.mule.runtime.api.meta.model.parameter.ParameterModel actingParameter = mockedParameterModel(actingParameterName);

    final org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel sampleDataProviderModel =
        mockedSampleDataProviderModel(providerId);
    when(sampleDataProviderModel.getParameters()).thenReturn(singletonList(actingParameterModel));

    final String parameterGroupName = "parameterGroup";
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mockedParameterGroupModel(parameterGroupName);

    when(parameterGroupModel.getParameterModels()).thenReturn(singletonList(actingParameter));
    final String operationName = "op1";
    final org.mule.runtime.api.meta.model.operation.OperationModel operationModel = mockedOperationModel(operationName);
    when(operationModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));
    when(operationModel.getSampleDataProviderModel()).thenReturn(of(sampleDataProviderModel));

    when(runtimeExtensionModel.getOperationModels()).thenReturn(singletonList(operationModel));

    ExtensionModel dto = new ExtensionModelFactory(empty()).createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    OperationModel opModelDto = dto.getOperationModel(operationName).get();
    SampleDataProviderModel sampleDataModelDto = opModelDto.getSampleDataProviderModel().get();

    assertThat(sampleDataModelDto.getProviderId(), is(providerId));
    assertThat(sampleDataModelDto.requiresConfiguration(), is(false));
    assertThat(sampleDataModelDto.requiresConnection(), is(false));
    assertThat(sampleDataModelDto.getParameters(), hasSize(1));
    assertThat(sampleDataModelDto.getParameters(), hasItem(ActingParameterMatcher.from(actingParameterModel)));
  }


  private void initialiseSimpleParameterGroup() {
    when(simpleRuntimeParameterGroupModel.getName()).thenReturn(SIMPLE_PARAMETER_GROUP_NAME);
    when(simpleRuntimeParameterGroupModel.isShowInDsl()).thenReturn(false);
    when(simpleRuntimeParameterGroupModel.getExclusiveParametersModels()).thenReturn(emptyList());
    when(simpleRuntimeParameterGroupModel.getParameterModels()).thenReturn(singletonList(simpleRuntimeParameterModel));
    when(simpleRuntimeParameterGroupModel.getDisplayModel()).thenReturn(empty());
    when(simpleRuntimeParameterGroupModel.getLayoutModel()).thenReturn(empty());
  }

  private void initialiseExtensionModel() {
    runtimeExtensionModel =
        mock(org.mule.runtime.api.meta.model.ExtensionModel.class);
    when(runtimeExtensionModel.getName()).thenReturn(EXTENSION_NAME);
    when(runtimeExtensionModel.getDescription()).thenReturn(EXTENSION_DESCRIPTION);
    when(runtimeExtensionModel.getVersion()).thenReturn(EXTENSION_VERSION);
    when(runtimeExtensionModel.getVendor()).thenReturn(EXTENSION_VENDOR);
    when(runtimeExtensionModel.getCategory()).thenReturn(Category.CERTIFIED);
    when(runtimeExtensionModel.getDeprecationModel()).thenReturn(empty());

    xmlDslModel = XmlDslModel.builder()
        .setPrefix(PREFIX)
        .setNamespace(NAMESPACE)
        .setSchemaVersion(SCHEMA_VERSION)
        .setSchemaLocation(SCHEMA_LOCATION)
        .build();
    when(runtimeExtensionModel.getXmlDslModel()).thenReturn(xmlDslModel);

    PathModel pathModel = new PathModel(FILE, true, EMBEDDED, FILE_EXTENSIONS.toArray(new String[0]));
    displayModel = DisplayModel.builder()
        .displayName(DISPLAY_NAME)
        .example(DISPLAY_EXAMPLE)
        .summary(DISPLAY_SUMMARY)
        .path(pathModel)
        .build();
    when(runtimeExtensionModel.getDisplayModel()).thenReturn(of(displayModel));

    outputModel =
        new ImmutableOutputModel("outputModelDescription", BaseTypeBuilder.create(JAVA).stringType().build(), false, emptySet());
  }

  private void initialiseFunction(org.mule.runtime.api.meta.model.function.FunctionModel runtimeFunctionModel) {

    org.mule.runtime.api.meta.model.OutputModel runtimeOutput = mock(org.mule.runtime.api.meta.model.OutputModel.class);
    when(runtimeOutput.getType()).thenReturn(stringType);
    when(runtimeOutput.getDescription()).thenReturn(EXTENSION_DESCRIPTION);

    when(runtimeFunctionModel.getName()).thenReturn(FUNCTION_NAME);
    when(runtimeFunctionModel.getDescription()).thenReturn(EXTENSION_DESCRIPTION);
    when(runtimeFunctionModel.getOutput()).thenReturn(runtimeOutput);
    when(runtimeFunctionModel.getDisplayModel()).thenReturn(Optional.of(displayModel));
    when(runtimeFunctionModel.getParameterGroupModels()).thenReturn(singletonList(simpleRuntimeParameterGroupModel));
    when(runtimeFunctionModel.getDeprecationModel())
        .thenReturn(of(new ImmutableDeprecationModel("This is deprecated", "1.2.0", "2.0.0")));
  }

  private void assertExtensionModelBasicAttributes(ExtensionModel dto) {
    assertThat(dto.getName(), is(EXTENSION_NAME));
    assertThat(dto.getDescription(), is(EXTENSION_DESCRIPTION));
    assertThat(dto.getVendor(), is(EXTENSION_VENDOR));
    assertThat(dto.getMinMuleVersion(), is(MIN_MULE_VERSION));
    assertThat(dto.getVersion(), is(EXTENSION_VERSION));

    final org.mule.tooling.client.api.extension.model.Category category = dto.getCategory();
    assertThat(category.isCertified(), is(true));
    assertThat(category.isCommunity(), is(false));
    assertThat(category.isPremium(), is(false));
    assertThat(category.isSelect(), is(false));
    assertThat(category.isUnknown(), is(false));
    assertThat(category.getValue(), is(Category.CERTIFIED.name()));

    assertDisplayModel(dto.getDisplayModel());
    assertXmlDslModel(dto.getXmlDslModel());
  }

  private void assertXmlDslModel(org.mule.tooling.client.api.extension.model.XmlDslModel xmlDslModel) {
    assertThat(xmlDslModel.getPrefix(), is(PREFIX));
    assertThat(xmlDslModel.getNamespace(), is(NAMESPACE));
    assertThat(xmlDslModel.getSchemaLocation(), is(SCHEMA_LOCATION));
    assertThat(xmlDslModel.getSchemaVersion(), is(SCHEMA_VERSION));
  }

  private void assertDisplayModel(Optional<org.mule.tooling.client.api.extension.model.DisplayModel> displayModelOptional) {
    assertThat(displayModelOptional.isPresent(), is(true));
    final org.mule.tooling.client.api.extension.model.DisplayModel displayModel = displayModelOptional.get();
    assertThat(displayModel.getDisplayName(), is(DISPLAY_NAME));
    assertThat(displayModel.getSummary(), is(DISPLAY_SUMMARY));
    assertThat(displayModel.getExample(), is(DISPLAY_EXAMPLE));
    assertThat(displayModel.getPathModel().isPresent(), is(true));
    final org.mule.tooling.client.api.extension.model.PathModel pathModel = displayModel.getPathModel().get();
    assertThat(pathModel.getFileExtensions(), hasItem(FILE_EXTENSIONS.get(0)));
    assertThat(pathModel.getType().isFile(), is(true));
    assertThat(pathModel.getType().isDirectory(), is(false));
    assertThat(pathModel.getType().isAny(), is(false));
    assertThat(pathModel.getType().isUnknown(), is(false));
    assertThat(pathModel.getLocation().isEmbedded(), is(true));
    assertThat(pathModel.getLocation().isAny(), is(false));
    assertThat(pathModel.getLocation().isExternal(), is(false));
  }

  private List<org.mule.runtime.api.meta.model.parameter.ActingParameterModel> asActingParameterModels(List<String> actingParameters) {
    return actingParameters.stream()
        .map(name -> new org.mule.runtime.extension.api.model.parameter.ImmutableActingParameterModel(name, true))
        .collect(toList());
  }

  private org.mule.runtime.api.meta.model.parameter.ActingParameterModel mockedActingParameterModel(String name) {
    final org.mule.runtime.api.meta.model.parameter.ActingParameterModel actingParameterModel =
        mock(org.mule.runtime.api.meta.model.parameter.ActingParameterModel.class);
    when(actingParameterModel.getName()).thenReturn(name);
    when(actingParameterModel.getExtractionExpression()).thenReturn(name);
    when(actingParameterModel.isRequired()).thenReturn(true);
    return actingParameterModel;
  }

  private org.mule.runtime.api.meta.model.operation.OperationModel mockedOperationModel(String name) {
    final org.mule.runtime.api.meta.model.operation.OperationModel operationModel =
        mock(org.mule.runtime.api.meta.model.operation.OperationModel.class);
    when(operationModel.getName()).thenReturn(name);

    final OutputModel outputModel = mock(OutputModel.class);
    when(operationModel.getOutput()).thenReturn(outputModel);
    when(operationModel.getOutputAttributes()).thenReturn(outputModel);
    return operationModel;
  }

  private org.mule.runtime.api.meta.model.parameter.ParameterModel mockedParameterModel(String name) {
    final org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    when(parameterModel.getName()).thenReturn(name);
    when(parameterModel.getExpressionSupport()).thenReturn(SUPPORTED);
    when(parameterModel.getModelProperty(any())).thenReturn(empty());
    when(parameterModel.getDslConfiguration()).thenReturn(ParameterDslConfiguration.getDefaultInstance());
    when(parameterModel.getLayoutModel()).thenReturn(empty());
    when(parameterModel.getRole()).thenReturn(BEHAVIOUR);
    when(parameterModel.getDisplayModel()).thenReturn(empty());
    when(parameterModel.getValueProviderModel()).thenReturn(empty());
    when(parameterModel.getAllowedStereotypes()).thenReturn(emptyList());
    when(parameterModel.getDeprecationModel()).thenReturn(empty());
    return parameterModel;
  }

  private org.mule.runtime.api.meta.model.parameter.ParameterGroupModel mockedParameterGroupModel(String name) {
    final org.mule.runtime.api.meta.model.parameter.ParameterGroupModel parameterGroupModel =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.class);
    when(parameterGroupModel.getName()).thenReturn(name);
    return parameterGroupModel;
  }

  private org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel mockedFieldValueProviderModel(String id,
                                                                                                          String targetSelector) {
    final org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel fieldValueProviderModel = mock(
                                                                                                           org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel.class);
    when(fieldValueProviderModel.getParameters()).thenReturn(emptyList());
    when(fieldValueProviderModel.getProviderId()).thenReturn(id);
    when(fieldValueProviderModel.getProviderName()).thenReturn(id);
    when(fieldValueProviderModel.getTargetSelector()).thenReturn(targetSelector);
    when(fieldValueProviderModel.requiresConfiguration()).thenReturn(false);
    when(fieldValueProviderModel.requiresConnection()).thenReturn(false);
    when(fieldValueProviderModel.getPartOrder()).thenReturn(1);
    when(fieldValueProviderModel.isOpen()).thenReturn(false);
    return fieldValueProviderModel;
  }

  private org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel mockedSampleDataProviderModel(String id) {
    final org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel sampleDataProviderModel = mock(
                                                                                                             org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel.class);
    when(sampleDataProviderModel.getParameters()).thenReturn(emptyList());
    when(sampleDataProviderModel.getProviderId()).thenReturn(id);
    when(sampleDataProviderModel.requiresConfiguration()).thenReturn(false);
    when(sampleDataProviderModel.requiresConnection()).thenReturn(false);
    return sampleDataProviderModel;
  }
}
