/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.hamcrest;

import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.hamcrest.core.IsNull.nullValue;
import static org.mule.tooling.client.internal.hamcrest.HamcrestUtils.validateThat;

import org.mule.runtime.extension.api.property.TypeResolversInformationModelProperty;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

public class ConstructModelMatcher extends TypeSafeDiagnosingMatcher<ConstructModel> {

  public static Matcher<ConstructModel> from(org.mule.runtime.api.meta.model.construct.ConstructModel runtimeConstructModel) {
    if (runtimeConstructModel == null) {
      return nullValue(ConstructModel.class);
    }
    return new ConstructModelMatcher(runtimeConstructModel);
  }

  private final org.mule.runtime.api.meta.model.construct.ConstructModel runtimeConstructModel;

  private ConstructModelMatcher(org.mule.runtime.api.meta.model.construct.ConstructModel runtimeConstructModel) {
    this.runtimeConstructModel = runtimeConstructModel;
  }

  @Override
  protected boolean matchesSafely(ConstructModel item, Description description) {
    return validateThat("name", item.getName(), equalTo(runtimeConstructModel.getName()), description) &&
        validateThat("description", item.getDescription(), equalTo(runtimeConstructModel.getDescription()), description) &&


        (runtimeConstructModel.getParameterGroupModels().isEmpty()
            ? validateThat("parameterGroupModels", item.getParameterGroupModels(), hasSize(0), description)
            : validateThat("parameterGroupModels", item.getParameterGroupModels(),
                           contains(ParameterGroupMatcher.sFrom(runtimeConstructModel.getParameterGroupModels())), description))

        &&

        runtimeConstructModel
            .getDisplayModel()
            .map(rdm -> item.getDisplayModel()
                .map(dm -> validateThat("displayModel", dm, DisplayModelMatcher.from(rdm), description))
                .orElse(false))
            .orElseGet(() -> validateThat("displayModel", item.getDisplayModel(), nullValue(), description))

        &&

        (runtimeConstructModel.getStereotype() == null
            ? validateThat("stereotype", item.getStereotype(), nullValue(), description)
            : validateThat("stereotype", item.getStereotype(), StereotypeMatcher.from(runtimeConstructModel.getStereotype()),
                           description))

        &&

        (runtimeConstructModel.getNestedComponents().isEmpty()
            ? validateThat("nestedComponents", item.getNestedComponents(), hasSize(0), description)
            : validateThat("nestedComponents", item.getNestedComponents(),
                           contains(NestableElementMatcher.sFrom(runtimeConstructModel.getNestedComponents())), description))

        &&

        validateThat("allowsTopLevelDeclaration", item.allowsTopLevelDeclaration(),
                     equalTo(runtimeConstructModel.allowsTopLevelDeclaration()), description)

        &&

        (runtimeConstructModel.getErrorModels().isEmpty()
            ? validateThat("errorModels", item.getErrorModels(), hasSize(0), description)
            : validateThat("errorModels", item.getErrorModels(),
                           contains(ErrorModelMatcher.sFrom(runtimeConstructModel.getErrorModels())), description))

        &&

        (!item.getDeprecationModel().isEnabled() ||
            runtimeConstructModel
                .getDeprecationModel()
                .map(rdp -> validateThat("deprecationModel", item.getDeprecationModel().get(), DeprecationModelMatcher.from(rdp),
                                         description))
                .orElseGet(() -> validateThat("deprecationModel", item.getDeprecationModel().get(), nullValue(), description)))
        &&

        (!item.getTypeResolversInformationModel().isEnabled() ||
            runtimeConstructModel
                .getModelProperty(TypeResolversInformationModelProperty.class)
                .map(rmki -> validateThat("typeResolvers", item.getTypeResolversInformationModel().get(),
                                          TypeResolverInformationModelMatcher.from(rmki), description))
                .orElseGet(() -> validateThat("typeResolvers", item.getTypeResolversInformationModel().get(), nullValue(),
                                              description)))

        &&

        validateThat("semanticTerms", item.getSemanticTerms(), equalTo(runtimeConstructModel.getSemanticTerms()), description);


  }

  @Override
  public void describeTo(Description description) {
    description.appendText("ConstructModel: ").appendValue(runtimeConstructModel);
  }
}
