/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.hamcrest;

import static java.util.Optional.empty;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.hamcrest.core.IsNull.nullValue;
import static org.mule.tooling.client.internal.hamcrest.HamcrestUtils.validateThat;

import org.mule.tooling.client.api.extension.model.nested.NestedRouteModel;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

public class NestedRouteMatcher extends TypeSafeDiagnosingMatcher<NestedRouteModel> {

  public static Matcher<NestedRouteModel> from(org.mule.runtime.api.meta.model.nested.NestedRouteModel runtimeNestedComponentModel) {
    if (runtimeNestedComponentModel == null) {
      return nullValue(NestedRouteModel.class);
    }
    return new NestedRouteMatcher(runtimeNestedComponentModel);
  }

  private final org.mule.runtime.api.meta.model.nested.NestedRouteModel runtimeNestedRouteModel;

  private NestedRouteMatcher(org.mule.runtime.api.meta.model.nested.NestedRouteModel runtimeNestedRouteModel) {
    this.runtimeNestedRouteModel = runtimeNestedRouteModel;
  }

  @Override
  protected boolean matchesSafely(NestedRouteModel nestedRouteModel, Description description) {
    return validateThat("isRequired", nestedRouteModel.isRequired(), equalTo(runtimeNestedRouteModel.isRequired()), description)
        &&

        validateThat("name", nestedRouteModel.getName(), equalTo(runtimeNestedRouteModel.getName()), description) &&

        validateThat("description", nestedRouteModel.getDescription(), equalTo(runtimeNestedRouteModel.getDescription()),
                     description)
        &&

        runtimeNestedRouteModel
            .getDisplayModel()
            .map(rdp -> validateThat("displayModel", nestedRouteModel.getDisplayModel(),
                                     DisplayModelMatcher.from(rdp),
                                     description))
            .orElseGet(() -> validateThat("displayModel", nestedRouteModel.getDisplayModel(), nullValue(), description))
        &&

        (runtimeNestedRouteModel.getParameterGroupModels().isEmpty()
            ? validateThat("parameterGroupModels", nestedRouteModel.getParameterGroupModels(), hasSize(0), description)
            : validateThat("parameterGroupModels", nestedRouteModel.getParameterGroupModels(),
                           contains(ParameterGroupMatcher.sFrom(runtimeNestedRouteModel.getParameterGroupModels())),
                           description))
        &&

        validateThat("minOccurs", nestedRouteModel.getMinOccurs(), equalTo(runtimeNestedRouteModel.getMinOccurs()), description)
        &&

        runtimeNestedRouteModel
            .getMaxOccurs()
            .map(rmo -> nestedRouteModel.getMaxOccurs()
                .map(mo -> validateThat("maxOccurs", mo, equalTo(rmo), description))
                .orElse(false))
            .orElseGet(() -> validateThat("maxOccurs", nestedRouteModel.getMaxOccurs(), equalTo(empty()), description))
        &&

        (runtimeNestedRouteModel.getNestedComponents().isEmpty()
            ? validateThat("nestedComponents", nestedRouteModel.getNestedComponents(), hasSize(0), description)
            : validateThat("nestedComponents", nestedRouteModel.getNestedComponents(),
                           contains(NestableElementMatcher.sFrom(runtimeNestedRouteModel.getNestedComponents())),
                           description))

        &&

        (!nestedRouteModel.getStereotype().isEnabled() ||
            (runtimeNestedRouteModel.getStereotype() == null
                ? validateThat("stereotype", nestedRouteModel.getStereotype().get(), nullValue(), description)
                : validateThat("stereotype", nestedRouteModel.getStereotype().get(),
                               StereotypeMatcher.from(runtimeNestedRouteModel.getStereotype()),
                               description)))
        &&

        (!nestedRouteModel.getDeprecationModel().isEnabled() ||
            runtimeNestedRouteModel
                .getDeprecationModel()
                .map(rdp -> validateThat("deprecationModel", nestedRouteModel.getDeprecationModel().get(),
                                         DeprecationModelMatcher.from(rdp), description))
                .orElseGet(() -> validateThat("deprecationModel", nestedRouteModel.getDeprecationModel().get(), nullValue(),
                                              description)))
        &&

        (!nestedRouteModel.getErrorModels().isEnabled() ||
            (runtimeNestedRouteModel.getErrorModels().isEmpty()
                ? validateThat("errorModels", nestedRouteModel.getErrorModels().get(), hasSize(0), description)
                : validateThat("errorModels", nestedRouteModel.getErrorModels().get(),
                               contains(ErrorModelMatcher.sFrom(runtimeNestedRouteModel.getErrorModels())),
                               description)))

        && validateThat("semanticTerms", nestedRouteModel.getSemanticTerms(), equalTo(runtimeNestedRouteModel.getSemanticTerms()),
                        description);

  }

  @Override
  public void describeTo(Description description) {
    description.appendText("NestedRouteModel: ").appendValue(runtimeNestedRouteModel);
  }
}
