/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.hamcrest;

import static java.util.Optional.empty;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.hamcrest.core.IsNull.nullValue;
import static org.mule.tooling.client.internal.hamcrest.HamcrestUtils.validateThat;

import org.mule.runtime.extension.api.property.MetadataKeyIdModelProperty;
import org.mule.runtime.extension.api.property.TypeResolversInformationModelProperty;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

public class OperationModelMatcher extends TypeSafeDiagnosingMatcher<OperationModel> {

  public static Matcher<OperationModel> from(org.mule.runtime.api.meta.model.operation.OperationModel runtimeOperationModel) {
    if (runtimeOperationModel == null) {
      return nullValue(OperationModel.class);
    }
    return new OperationModelMatcher(runtimeOperationModel);
  }

  private final org.mule.runtime.api.meta.model.operation.OperationModel runtimeOperationModel;

  private OperationModelMatcher(org.mule.runtime.api.meta.model.operation.OperationModel runtimeOperationModel) {
    this.runtimeOperationModel = runtimeOperationModel;
  }

  @Override
  protected boolean matchesSafely(OperationModel item, Description description) {
    return validateThat("name", item.getName(), equalTo(runtimeOperationModel.getName()), description) &&
        validateThat("description", item.getDescription(), equalTo(runtimeOperationModel.getDescription()), description) &&


        (runtimeOperationModel.getParameterGroupModels().isEmpty()
            ? validateThat("parameterGroupModels", item.getParameterGroupModels(), hasSize(0), description)
            : validateThat("parameterGroupModels", item.getParameterGroupModels(),
                           contains(ParameterGroupMatcher.sFrom(runtimeOperationModel.getParameterGroupModels())), description))

        &&

        runtimeOperationModel
            .getDisplayModel()
            .map(rdm -> item.getDisplayModel()
                .map(dm -> validateThat("displayModel", dm, DisplayModelMatcher.from(rdm), description))
                .orElse(false))
            .orElseGet(() -> validateThat("displayModel", item.getDisplayModel(), nullValue(), description))

        &&

        validateThat("isBlocking", item.isBlocking(), equalTo(runtimeOperationModel.isBlocking()), description)

        &&

        (runtimeOperationModel.getErrorModels().isEmpty()
            ? validateThat("errorModels", item.getErrorModels(), hasSize(0), description)
            : validateThat("errorModels", item.getErrorModels(),
                           contains(ErrorModelMatcher.sFrom(runtimeOperationModel.getErrorModels())), description))

        &&

        validateThat("output", item.getOutput(), OutputModelMatcher.from(runtimeOperationModel.getOutput()), description)

        &&

        validateThat("outputAttributes", item.getOutputAttributes(),
                     OutputModelMatcher.from(runtimeOperationModel.getOutputAttributes()), description)

        &&

        validateThat("isTransactional", item.isTransactional(), equalTo(runtimeOperationModel.isTransactional()), description)

        &&

        validateThat("requiresConnection", item.requiresConnection(), equalTo(runtimeOperationModel.requiresConnection()),
                     description)

        &&

        validateThat("supportsStreaming", item.supportsStreaming(), equalTo(runtimeOperationModel.supportsStreaming()),
                     description)

        &&


        (runtimeOperationModel.getStereotype() == null
            ? validateThat("stereotype", item.getStereotype(), nullValue(), description)
            : validateThat("stereotype", item.getStereotype(), StereotypeMatcher.from(runtimeOperationModel.getStereotype()),
                           description))

        &&


        (runtimeOperationModel.getNestedComponents().isEmpty()
            ? validateThat("nestedComponents", item.getNestedComponents(), hasSize(0), description)
            : validateThat("nestedComponents", item.getNestedComponents(),
                           contains(NestableElementMatcher.sFrom(runtimeOperationModel.getNestedComponents())), description))

        &&

        (!item.getDeprecationModel().isEnabled() ||
            runtimeOperationModel
                .getDeprecationModel()
                .map(rdp -> validateThat("deprecationModel", item.getDeprecationModel().get(),
                                         DeprecationModelMatcher.from(rdp), description))
                .orElseGet(() -> validateThat("deprecationModel", item.getDeprecationModel().get(), nullValue(),
                                              description)))
        &&


        (!item.getSampleDataProviderModel().isEnabled() ||
            runtimeOperationModel
                .getSampleDataProviderModel()
                .map(rsdm -> validateThat("sampleDataProvider", item.getSampleDataProviderModel().get(),
                                          SampleDataMatcher.from(rsdm), description))
                .orElseGet(() -> validateThat("sampleDataProvider", item.getSampleDataProviderModel().get(), nullValue(),
                                              description)))


        &&

        validateThat("semanticTerms", item.getSemanticTerms(), equalTo(runtimeOperationModel.getSemanticTerms()), description)

        &&

        runtimeOperationModel
            .getModelProperty(MetadataKeyIdModelProperty.class)
            .map(rmki -> item.getMetadataKeyIdModel()
                .map(mki -> validateThat("metadataKeyIdModel", mki, MetadataKeyIdMatcher.from(rmki), description))
                .orElse(false))
            .orElseGet(() -> validateThat("metadataKeyIdModel", item.getMetadataKeyIdModel(), equalTo(empty()), description))
        &&

        (!item.getTypeResolversInformationModel().isEnabled() ||
            runtimeOperationModel
                .getModelProperty(TypeResolversInformationModelProperty.class)
                .map(rmki -> validateThat("typeResolvers", item.getTypeResolversInformationModel().get(),
                                          TypeResolverInformationModelMatcher.from(rmki), description))
                .orElseGet(() -> validateThat("typeResolvers", item.getTypeResolversInformationModel().get(), nullValue(),
                                              description)));

  }

  @Override
  public void describeTo(Description description) {
    description.appendText("OperationModel: ").appendValue(runtimeOperationModel);
  }
}
