/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session;

import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.CoreMatchers.hasItems;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.metadata.MetadataKeyBuilder.newKey;
import static org.mule.runtime.api.metadata.resolving.FailureCode.INVALID_METADATA_KEY;
import static org.mule.runtime.api.metadata.resolving.MetadataFailure.Builder.newFailure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.success;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ComponentModelVisitor;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.OutputModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.MetadataKeysContainerBuilder;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataTypesDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataFailure;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.app.declaration.api.ElementDeclaration;
import org.mule.runtime.app.declaration.api.OperationElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterGroupElementDeclaration;
import org.mule.runtime.app.declaration.api.fluent.ParameterSimpleValue;
import org.mule.runtime.extension.api.property.MetadataKeyIdModelProperty;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;
import org.mule.tooling.client.internal.metadata.ToolingCacheIdGenerator;
import org.mule.tooling.client.internal.session.cache.DeclarationMetadataCache;
import org.mule.tooling.client.internal.session.cache.DefaultDeclarationMetadataCache;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import org.junit.Before;
import org.junit.Test;
import org.mockito.stubbing.Answer;

public class DefaultDeclarationMetadataCacheTestCase {

  private ToolingCacheIdGenerator<ElementDeclaration> cacheIdGenerator;
  private Map<String, MetadataResult<MetadataKeysContainer>> keysStorage;
  private Map<String, MetadataType> typesStorage;
  private OperationElementDeclaration elementDeclaration;
  private ExtensionModelProvider extensionModelProvider;
  private MetadataResult<MetadataKeysContainer> metadataKeysResult;
  private MetadataResult<ComponentMetadataTypesDescriptor> metadataTypesResult;
  private MetadataType metadataType;
  private DeclarationMetadataCache metadataCache;

  @Before
  public void setUp() {
    cacheIdGenerator = mock(ToolingCacheIdGenerator.class);
    metadataKeysResult = mock(MetadataResult.class);
    metadataTypesResult = mock(MetadataResult.class);
    metadataType = mock(MetadataType.class);
    typesStorage = new HashMap<>();
    keysStorage = new HashMap<>();

    String opName = "op";
    OutputModel outputModel = mock(OutputModel.class);
    when(outputModel.hasDynamicType()).thenReturn(true);
    when(outputModel.getType()).thenReturn(metadataType);
    OutputModel attributesModel = mock(OutputModel.class);
    when(attributesModel.hasDynamicType()).thenReturn(false);
    when(attributesModel.getType()).thenReturn(metadataType);
    OperationModel operationModel = mock(OperationModel.class);
    when(operationModel.getName()).thenReturn(opName);
    doAnswer(i -> {
      ((ComponentModelVisitor) i.getArgument(0)).visit(operationModel);
      return null;
    }).when(operationModel).accept(any());
    when(operationModel.getOutput()).thenReturn(outputModel);
    when(operationModel.getOutputAttributes()).thenReturn(attributesModel);
    String extensionName = "extension";
    ExtensionModel extensionModel = mock(ExtensionModel.class);
    when(extensionModel.getName()).thenReturn(extensionName);
    when(extensionModel.getOperationModels()).thenReturn(singletonList(operationModel));

    extensionModelProvider = mock(ExtensionModelProvider.class);
    when(extensionModelProvider.getAll()).thenReturn(singleton(extensionModel));
    when(extensionModelProvider.get(any())).thenReturn(of(extensionModel));
    when(extensionModelProvider.getAllNames()).thenReturn(singleton(extensionName));

    elementDeclaration = mock(OperationElementDeclaration.class);
    when(elementDeclaration.getName()).thenReturn(opName);
    when(elementDeclaration.getDeclaringExtension()).thenReturn(extensionName);

    metadataCache = new DefaultDeclarationMetadataCache(
                                                        cacheIdGenerator,
                                                        typesStorage,
                                                        keysStorage,
                                                        extensionModelProvider);
  }

  @Test
  public void exceptionWhenGettingValueFromCache() {
    String key = "key";
    when(cacheIdGenerator.getIdForMetadataKeys(eq(elementDeclaration))).thenReturn(of(key));
    when(metadataKeysResult.isSuccess()).thenReturn(true);
    AtomicInteger calls = new AtomicInteger();

    Map<String, MetadataResult<MetadataKeysContainer>> mockStorage = mock(Map.class);
    when(mockStorage.containsKey(key)).thenReturn(true);
    when(mockStorage.get(key)).thenThrow(new RuntimeException("Expected exception"));
    DeclarationMetadataCache metadataCache = new DefaultDeclarationMetadataCache(
                                                                                 cacheIdGenerator,
                                                                                 typesStorage,
                                                                                 mockStorage,
                                                                                 extensionModelProvider);

    metadataCache.getMetadataKeys(elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataKeysResult;
    },
                                  false);
    assertThat(calls.get(), is(1));

    verify(mockStorage).containsKey(key);
    verify(mockStorage).get(key);
    verify(mockStorage).put(eq(key), any());
  }

  @Test
  public void nullValue() {
    String key = "key";
    when(cacheIdGenerator.getIdForMetadataKeys(eq(elementDeclaration))).thenReturn(of(key));
    when(metadataKeysResult.isSuccess()).thenReturn(true);
    AtomicInteger calls = new AtomicInteger();

    Map<String, MetadataResult<MetadataKeysContainer>> mockStorage = mock(Map.class);
    when(mockStorage.containsKey(key)).thenReturn(true);
    when(mockStorage.get(key)).thenReturn(null);
    DeclarationMetadataCache metadataCache = new DefaultDeclarationMetadataCache(
                                                                                 cacheIdGenerator,
                                                                                 typesStorage,
                                                                                 mockStorage,
                                                                                 extensionModelProvider);

    metadataCache.getMetadataKeys(elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataKeysResult;
    },
                                  false);
    assertThat(calls.get(), is(1));

    verify(mockStorage).containsKey(key);
    verify(mockStorage).get(key);
    verify(mockStorage).put(eq(key), any());
  }

  @Test
  public void keysStoredIfSuccess() {
    String key = "key";
    when(cacheIdGenerator.getIdForMetadataKeys(eq(elementDeclaration))).thenReturn(of(key));
    when(metadataKeysResult.isSuccess()).thenReturn(true);
    AtomicInteger calls = new AtomicInteger();
    metadataCache.getMetadataKeys(elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataKeysResult;
    },
                                  false);
    metadataCache.getMetadataKeys(elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataKeysResult;
    },
                                  false);
    assertThat(calls.get(), is(1));
    assertThat(keysStorage.entrySet(), hasSize(1));
  }

  @Test
  public void keysNotStoredIfFailure() {
    String key = "key";
    when(cacheIdGenerator.getIdForMetadataKeys(eq(elementDeclaration))).thenReturn(of(key));
    when(metadataKeysResult.isSuccess()).thenReturn(false);
    AtomicInteger calls = new AtomicInteger();
    metadataCache.getMetadataKeys(elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataKeysResult;
    },
                                  false);
    metadataCache.getMetadataKeys(elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataKeysResult;
    },
                                  false);
    assertThat(calls.get(), is(2));
    assertThat(keysStorage.entrySet(), hasSize(0));
  }

  @Test
  public void onPartialKeysFailureUpperLevelsShouldBeRemoved() {
    ComponentModel parameterizedModel = mock(ComponentModel.class);
    BaseTypeBuilder baseTypeBuilder = new BaseTypeBuilder(MetadataFormat.JAVA);
    ObjectType objectType = baseTypeBuilder.objectType().build();

    String multiLevelParameterGroupName = "MultiLevelParameter";
    String categoryName = "category1";
    when(parameterizedModel.getModelProperty(MetadataKeyIdModelProperty.class))
        .thenReturn(of(new MetadataKeyIdModelProperty(objectType, multiLevelParameterGroupName, categoryName)));

    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.getName()).thenReturn(multiLevelParameterGroupName);

    ParameterModel rootParameterModel = mock(ParameterModel.class);
    String rootLevel = "rootLevel";
    when(rootParameterModel.getName()).thenReturn(rootLevel);
    when(rootParameterModel.getModelProperty(MetadataKeyPartModelProperty.class))
        .thenReturn(of(new MetadataKeyPartModelProperty(1, true)));
    when(parameterGroupModel.getParameter(rootLevel)).thenReturn(of(rootParameterModel));

    ParameterModel firstLevelParameterModel = mock(ParameterModel.class);
    String firstLevel = "firstLevel";
    when(firstLevelParameterModel.getName()).thenReturn(firstLevel);
    when(firstLevelParameterModel.getModelProperty(MetadataKeyPartModelProperty.class))
        .thenReturn(of(new MetadataKeyPartModelProperty(2, true)));
    when(parameterGroupModel.getParameter(firstLevel)).thenReturn(of(firstLevelParameterModel));

    ParameterModel secondLevelParameterModel = mock(ParameterModel.class);
    String secondLevel = "secondLevel";
    when(secondLevelParameterModel.getName()).thenReturn(secondLevel);
    when(secondLevelParameterModel.getModelProperty(MetadataKeyPartModelProperty.class))
        .thenReturn(of(new MetadataKeyPartModelProperty(3, true)));
    when(parameterGroupModel.getParameter(secondLevel)).thenReturn(of(secondLevelParameterModel));

    List<ParameterModel> parameterModels = new ArrayList<>();
    parameterModels.add(rootParameterModel);
    parameterModels.add(firstLevelParameterModel);
    parameterModels.add(secondLevelParameterModel);
    when(parameterGroupModel.getParameterModels()).thenReturn(parameterModels);
    when(parameterizedModel.getAllParameterModels()).thenReturn(parameterModels);

    List<ParameterGroupModel> parameterGroupModels = new ArrayList<>();
    parameterGroupModels.add(parameterGroupModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(parameterGroupModels);

    // Cannot use Mockito as it will modify the internal collection
    ParameterGroupElementDeclaration parameterGroupElementDeclaration =
        new ParameterGroupElementDeclaration(multiLevelParameterGroupName);
    String extension = "extension";
    parameterGroupElementDeclaration.setDeclaringExtension(extension);

    ParameterElementDeclaration rootLevelParameterElementDeclaration = new ParameterElementDeclaration(rootLevel);
    rootLevelParameterElementDeclaration.setDeclaringExtension(extension);
    rootLevelParameterElementDeclaration.setValue(ParameterSimpleValue.of("A1"));
    parameterGroupElementDeclaration.addParameter(rootLevelParameterElementDeclaration);

    ParameterElementDeclaration firstLevelParameterElementDeclaration = new ParameterElementDeclaration(firstLevel);
    firstLevelParameterElementDeclaration.setDeclaringExtension(extension);
    firstLevelParameterElementDeclaration.setValue(ParameterSimpleValue.of("A1:B11"));
    parameterGroupElementDeclaration.addParameter(firstLevelParameterElementDeclaration);

    when(elementDeclaration.getParameterGroup(multiLevelParameterGroupName)).thenReturn(of(parameterGroupElementDeclaration));

    String rootLevelKey = categoryName;
    String firstLevelKeyForA1 = categoryName + ":A1";
    String secondLevelKeyForA1B11 = categoryName + ":A1:B11";
    String secondLevelKeyForA1B12 = categoryName + ":A1:B12";

    String firstLevelKeyForA2 = categoryName + ":A2";
    String secondLevelKeyForA2B21 = categoryName + ":A2:B21";
    String secondLevelKeyForA2B22 = categoryName + ":A2:B22";

    when(cacheIdGenerator.getIdForMetadataKeys(any(ElementDeclaration.class)))
        .then((Answer<Optional<String>>) invocationOnMock -> {
          OperationElementDeclaration operationElementDeclaration = invocationOnMock.getArgument(0);
          ParameterGroupElementDeclaration groupElementDeclaration =
              operationElementDeclaration.getParameterGroup(multiLevelParameterGroupName).get();
          if (groupElementDeclaration.getParameter(firstLevel).isPresent()) {
            return of(secondLevelKeyForA1B11);
          } else if (groupElementDeclaration.getParameter(rootLevel).isPresent()) {
            return of(firstLevelKeyForA1);
          } else {
            return of(rootLevelKey);
          }
        });

    when(metadataKeysResult.isSuccess()).thenReturn(false);
    List<MetadataFailure> failures = new ArrayList<>();
    failures.add(newFailure().withFailureCode(INVALID_METADATA_KEY).onKeys());
    when(metadataKeysResult.getFailures()).thenReturn(failures);
    AtomicInteger calls = new AtomicInteger();

    keysStorage.put(rootLevelKey, success(MetadataKeysContainerBuilder.getInstance()
        .add(categoryName, ImmutableSet.of(newKey("A1").build(), newKey("A2").build())).build()));
    keysStorage.put(firstLevelKeyForA1, success(MetadataKeysContainerBuilder.getInstance()
        .add(categoryName, ImmutableSet.of(newKey("B11").build(), newKey("B12").build())).build()));
    keysStorage.put(secondLevelKeyForA1B11, success(MetadataKeysContainerBuilder.getInstance()
        .add(categoryName, ImmutableSet.of(newKey("C111").build(), newKey("C112").build())).build()));
    keysStorage.put(secondLevelKeyForA1B12, success(MetadataKeysContainerBuilder.getInstance()
        .add(categoryName, ImmutableSet.of(newKey("C121").build(), newKey("C122").build())).build()));

    keysStorage.put(firstLevelKeyForA2, success(MetadataKeysContainerBuilder.getInstance()
        .add(categoryName, ImmutableSet.of(newKey("B21").build(), newKey("B22").build())).build()));
    keysStorage.put(secondLevelKeyForA2B21, success(MetadataKeysContainerBuilder.getInstance()
        .add(categoryName, ImmutableSet.of(newKey("C211").build(), newKey("C212").build())).build()));
    keysStorage.put(secondLevelKeyForA2B22, success(MetadataKeysContainerBuilder.getInstance()
        .add(categoryName, ImmutableSet.of(newKey("C221").build(), newKey("C222").build())).build()));

    metadataCache.getMetadataKeysPartialFetch(parameterizedModel, elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataKeysResult;
    },
                                              true);
    assertThat(calls.get(), is(1));
    assertThat(keysStorage.entrySet(), hasSize(4));
    assertThat(keysStorage.entrySet().stream().map(e -> e.getKey()).collect(Collectors.toList()),
               allOf(hasItems(firstLevelKeyForA2, secondLevelKeyForA2B21, secondLevelKeyForA2B22, secondLevelKeyForA1B12),
                     not(hasItems(secondLevelKeyForA1B11, firstLevelKeyForA1, rootLevelKey))));
  }

  @Test
  public void typesStoredIfSuccess() {
    String key = "key";
    when(cacheIdGenerator.getIdForComponentOutputMetadata(eq(elementDeclaration))).thenReturn(of(key));
    when(metadataTypesResult.isSuccess()).thenReturn(true);
    ComponentMetadataTypesDescriptor descriptor = mock(ComponentMetadataTypesDescriptor.class);
    when(descriptor.getOutputMetadata()).thenReturn(of(metadataType));
    when(descriptor.getOutputAttributesMetadata()).thenReturn(empty());
    when(descriptor.getInputMetadata()).thenReturn(emptyMap());
    when(descriptor.getInputMetadata(any())).thenReturn(empty());
    when(metadataTypesResult.get()).thenReturn(descriptor);

    AtomicInteger calls = new AtomicInteger();
    metadataCache.getComponentMetadata(elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataTypesResult;
    },
                                       false);
    metadataCache.getComponentMetadata(elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataTypesResult;
    },
                                       false);

    assertThat(calls.get(), is(1));
    assertThat(typesStorage.entrySet(), hasSize(1));
  }

  @Test
  public void typesNotStoredIfFailure() {
    String key = "key";
    when(cacheIdGenerator.getIdForComponentOutputMetadata(eq(elementDeclaration))).thenReturn(of(key));
    when(metadataTypesResult.isSuccess()).thenReturn(false);

    AtomicInteger calls = new AtomicInteger();
    metadataCache.getComponentMetadata(elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataTypesResult;
    },
                                       false);
    metadataCache.getComponentMetadata(elementDeclaration, () -> {
      calls.incrementAndGet();
      return metadataTypesResult;
    },
                                       false);

    assertThat(calls.get(), is(2));
    assertThat(typesStorage.entrySet(), hasSize(0));
  }

}
