/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static com.google.common.collect.Sets.newHashSet;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.ExtensionModelService;

import java.io.File;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * {@link ExtensionModelService} that use delegates to {@link InternalExtensionModelService} to resolve {@link ExtensionModel}s
 */
public class DefaultMuleRuntimeExtensionModelProvider extends AbstractMuleRuntimeExtensionModelProvider {

  /**
   * Creates a new instance
   *
   * @param extensionModelService the service to use to load {@link ExtensionModel}s in case of a cache miss.
   */
  public DefaultMuleRuntimeExtensionModelProvider(MavenClient mavenClient, InternalExtensionModelService extensionModelService) {
    super(mavenClient, extensionModelService);
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(ArtifactDescriptor pluginDescriptor) {
    return extensionModelService.loadExtensionData(toBundleDescriptor(pluginDescriptor))
        .map(loadedExtensionInformation -> loadedExtensionInformation.getExtensionModel());
  }

  @Override
  public Optional<String> getMinMuleVersion(ArtifactDescriptor pluginDescriptor) {
    return extensionModelService.loadExtensionData(toBundleDescriptor(pluginDescriptor))
        .map(loadedExtensionInformation -> loadedExtensionInformation.getMinMuleVersion());
  }

  @Override
  public Optional<LoadedExtensionInformation> getExtensionModel(File pluginFile) {
    return extensionModelService.loadExtensionData(pluginFile);
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(BundleDependency bundleDependency) {
    return extensionModelService.loadExtensionData(bundleDependency.getDescriptor())
        .map(loadedExtensionInformation -> loadedExtensionInformation.getExtensionModel());
  }

  @Override
  public Set<ExtensionModel> loadExtensionModels(List<ArtifactClassLoader> artifactPluginClassLoaders) {
    Set<ExtensionModel> extensionModels = newHashSet();
    for (ArtifactClassLoader artifactClassLoader : artifactPluginClassLoaders) {
      Optional<LoadedExtensionInformation> loadedExtensionInformation =
          extensionModelService.loadExtensionData(artifactClassLoader, artifactPluginClassLoaders);
      loadedExtensionInformation.ifPresent(modelInformation -> extensionModels.add(modelInformation.getExtensionModel()));
    }
    return extensionModels;
  }

  @Override
  public Optional<String> getMinMuleVersion(File pluginFile) {
    return extensionModelService.loadExtensionData(pluginFile)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getMinMuleVersion());
  }

  @Override
  public Optional<String> getExtensionSchema(File pluginFile) {
    return extensionModelService.loadExtensionData(pluginFile)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getSchema().get());
  }

  @Override
  public Optional<String> getExtensionSchema(ArtifactDescriptor pluginDescriptor) {
    return extensionModelService.loadExtensionData(toBundleDescriptor(pluginDescriptor))
        .map(loadedExtensionInformation -> loadedExtensionInformation.getSchema().get());
  }

  @Override
  public List<ExtensionModel> getRuntimeExtensionModels() {
    return extensionModelService.loadRuntimeExtensionModels();
  }

}
