/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.api.plugin.resolver.PluginDependenciesResolver;
import org.mule.runtime.module.artifact.api.descriptor.AbstractArtifactDescriptorFactory;
import org.mule.runtime.module.deployment.impl.internal.domain.DomainDescriptorFactory;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.datasense.MetadataCacheFactory;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.internal.application.ApplicationClassLoaderFactory;
import org.mule.tooling.client.internal.application.DomainClassLoaderFactory;
import org.mule.tooling.client.internal.cache.CacheStorageMapWrapperFactory;
import org.mule.tooling.client.internal.metadata.MetadataMediatorRuntimeToolingServiceDecorator;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.service.ServiceRegistry;

import java.util.Optional;

/**
 * Default implementation of {@link ToolingArtifactContext}.
 */
public class DefaultToolingArtifactContext implements ToolingArtifactContext {

  private final MuleArtifactResourcesRegistry muleArtifactResourcesRegistry;
  private final SpiRuntimeToolingServiceProvider runtimeToolingServiceProvider = new SpiRuntimeToolingServiceProvider();
  private transient volatile boolean runtimeToolingServiceInitialized;
  private RuntimeToolingService runtimeToolingService;

  private Optional<AgentConfiguration> agentConfigurationOptional;
  private ApplicationCache applicationCache;
  private DomainCache domainCache;
  private MavenClient mavenClient;
  private MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
  private Optional<MetadataCacheFactory> metadataCacheFactory;
  private Optional<CacheStorageMapWrapperFactory> metadataCacheStorageMapWrapperFactory;
  private Optional<CacheStorageMapWrapperFactory> valueProvidersCacheStorageMapWrapperFactory;
  private Serializer serializer;
  private ServiceRegistry serviceRegistry;

  public DefaultToolingArtifactContext(MuleArtifactResourcesRegistry muleArtifactResourcesRegistry) {
    this.muleArtifactResourcesRegistry = muleArtifactResourcesRegistry;
  }

  public void setAgentConfiguration(Optional<AgentConfiguration> agentConfigurationOptional) {
    this.agentConfigurationOptional = agentConfigurationOptional;
  }

  @Override
  public Optional<AgentConfiguration> getAgentConfiguration() {
    return agentConfigurationOptional;
  }

  @Override
  public RuntimeToolingService getRuntimeToolingService() {
    if (!this.runtimeToolingServiceInitialized) {
      synchronized (this) {
        if (!this.runtimeToolingServiceInitialized) {
          this.runtimeToolingService = runtimeToolingServiceProvider.lookUpRuntimeToolingService();

          if (!agentConfigurationOptional.isPresent()) {
            throw new MissingToolingConfigurationException("Tooling Client has to be configured with a configuration to access REST Tooling API in order to resolve operations");
          }

          AgentConfiguration agentConfiguration = agentConfigurationOptional.get();
          runtimeToolingService.setToolingApiUrl(agentConfiguration.getToolingApiUrl(),
                                                 agentConfiguration.getDefaultConnectTimeout(),
                                                 agentConfiguration.getDefaultReadTimeout(),
                                                 agentConfiguration.getSslConfiguration(),
                                                 agentConfiguration.getProxyConfig(),
                                                 agentConfiguration.getAuthorizationToken());

          // Defines the API version of the Agent
          runtimeToolingService.setMuleVersion(agentConfiguration.muleVersion().orElse(null));
          muleArtifactResourcesRegistry.getTargetMuleVersion().ifPresent(targetMuleVersion -> runtimeToolingService =
              new MetadataMediatorRuntimeToolingServiceDecorator(targetMuleVersion, runtimeToolingService));
          this.runtimeToolingServiceInitialized = true;
        }
      }
    }
    return runtimeToolingService;
  }

  @Override
  public PluginDependenciesResolver getPluginDependenciesResolver() {
    return muleArtifactResourcesRegistry.getPluginDependenciesResolver();
  }

  @Override
  public ApplicationCache getApplicationCache() {
    return applicationCache;
  }

  public void setApplicationCache(ApplicationCache applicationCache) {
    this.applicationCache = applicationCache;
  }

  @Override
  public DomainCache getDomainCache() {
    return domainCache;
  }

  public void setDomainCache(DomainCache domainCache) {
    this.domainCache = domainCache;
  }

  public void setMavenClient(MavenClient mavenClient) {
    this.mavenClient = mavenClient;
  }

  @Override
  public MavenClient getMavenClient() {
    return mavenClient;
  }

  public void setMuleRuntimeExtensionModelProvider(MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider) {
    this.muleRuntimeExtensionModelProvider = muleRuntimeExtensionModelProvider;
  }

  @Override
  public MuleRuntimeExtensionModelProvider getMuleRuntimeExtensionModelProvider() {
    return muleRuntimeExtensionModelProvider;
  }

  @Override
  public DomainClassLoaderFactory getDomainClassLoaderFactory() {
    return muleArtifactResourcesRegistry.getDomainClassLoaderFactory();
  }

  @Override
  public ApplicationClassLoaderFactory getApplicationClassLoaderFactory() {
    return muleArtifactResourcesRegistry.getApplicationClassLoaderFactory();
  }

  public DomainDescriptorFactory getDomainDescriptorFactory() {
    return muleArtifactResourcesRegistry.getDomainDescriptorFactory();
  }

  public AbstractArtifactDescriptorFactory<MuleApplicationModel, ApplicationDescriptor> getApplicationDescriptorFactory() {
    return muleArtifactResourcesRegistry.getApplicationDescriptorFactory();
  }

  public void setMetadataCacheStorageFactory(Optional<CacheStorageMapWrapperFactory> metadataCacheFactory) {
    this.metadataCacheStorageMapWrapperFactory = metadataCacheFactory;
  }

  public void setValueProvidersCacheStorageFactory(Optional<CacheStorageMapWrapperFactory> valueProvidersCacheStorageFactory) {
    this.valueProvidersCacheStorageMapWrapperFactory = valueProvidersCacheStorageFactory;
  }

  public Optional<CacheStorageMapWrapperFactory> getMetadataCacheStorageMapWrapperFactory() {
    return metadataCacheStorageMapWrapperFactory;
  }

  public Optional<CacheStorageMapWrapperFactory> getValueProvidersCacheStorageMapWrapperFactory() {
    return valueProvidersCacheStorageMapWrapperFactory;
  }

  public void setSerializer(Serializer serializer) {
    this.serializer = serializer;
  }

  public Serializer getSerializer() {
    return serializer;
  }

  public void setServiceRegistry(ServiceRegistry serviceRegistry) {
    this.serviceRegistry = serviceRegistry;
  }

  public ServiceRegistry getServiceRegistry() {
    return serviceRegistry;
  }

  public void setMetadataCacheFactory(Optional<MetadataCacheFactory> metadataCacheFactoryOptional) {
    this.metadataCacheFactory = metadataCacheFactoryOptional;
  }

  public Optional<MetadataCacheFactory> getMetadataCacheFactory() {
    return metadataCacheFactory;
  }
}
