/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static org.mule.runtime.deployment.model.api.artifact.ArtifactDescriptorFactoryProvider.artifactDescriptorFactoryProvider;
import static org.mule.runtime.deployment.model.api.builder.DeployableArtifactClassLoaderFactoryProvider.regionPluginClassLoadersFactory;
import static org.mule.runtime.globalconfig.api.maven.MavenClientFactory.setMavenClientProvider;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.AetherMavenClientProvider;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.api.artifact.DescriptorLoaderRepositoryFactory;
import org.mule.runtime.deployment.model.api.builder.RegionPluginClassLoadersFactory;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.api.plugin.resolver.PluginDependenciesResolver;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.DeployableArtifactClassLoaderFactory;
import org.mule.runtime.module.artifact.api.classloader.MuleDeployableArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.AbstractArtifactDescriptorFactory;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorValidatorBuilder;
import org.mule.runtime.module.artifact.api.descriptor.DescriptorLoaderRepository;
import org.mule.runtime.module.deployment.impl.internal.domain.DomainDescriptorFactory;
import org.mule.runtime.module.deployment.impl.internal.plugin.ArtifactPluginDescriptorLoader;
import org.mule.tooling.client.internal.application.ApplicationClassLoaderFactory;
import org.mule.tooling.client.internal.application.DomainClassLoaderFactory;

import java.io.File;
import java.util.List;
import java.util.Optional;

/**
 * Mule resources registry that has implementations for factories and utilities for creating artifacts.
 */
public class MuleArtifactResourcesRegistry {

  private String toolingVersion;
  private Optional<MuleVersion> targetMuleVersion;
  private ModuleRepository moduleRepository;
  private final File workingDirectory;

  private AbstractArtifactDescriptorFactory<MulePluginModel, ArtifactPluginDescriptor> artifactPluginDescriptorFactory;
  private ArtifactPluginDescriptorLoader artifactPluginDescriptorLoader;
  private PluginDependenciesResolver pluginDependenciesResolver;

  private ArtifactClassLoader containerArtifactClassLoader;

  private AbstractArtifactDescriptorFactory<MuleApplicationModel, ApplicationDescriptor> applicationDescriptorFactory;
  private DomainDescriptorFactory domainDescriptorFactory;

  private RegionPluginClassLoadersFactory regionPluginClassLoadersFactory;

  private DomainClassLoaderFactory domainClassLoaderFactory;
  private ApplicationClassLoaderFactory applicationClassLoaderFactory;

  private DescriptorLoaderRepository descriptorLoaderRepository;

  /**
   * Creates an instance of the registry.
   */
  public MuleArtifactResourcesRegistry(String toolingVersion, Optional<MuleVersion> targetMuleVersion, MavenClient mavenClient,
                                       ModuleRepository moduleRepository,
                                       ArtifactClassLoader containerArtifactClassLoader, File workingDirectory) {
    this.toolingVersion = toolingVersion;
    this.targetMuleVersion = targetMuleVersion;
    this.moduleRepository = moduleRepository;
    this.containerArtifactClassLoader = containerArtifactClassLoader;
    this.workingDirectory = workingDirectory;

    setMavenClientProvider(() -> new AetherMavenClientProvider() {

      @Override
      public MavenClient createMavenClient(MavenConfiguration mavenConfiguration) {
        return mavenClient;
      }
    });

    init();
  }

  public DescriptorLoaderRepository getDescriptorLoaderRepository() {
    return descriptorLoaderRepository;
  }

  private void init() {
    descriptorLoaderRepository =
        new DescriptorLoaderRepositoryFactory().createDescriptorLoaderRepository();

    ArtifactDescriptorValidatorBuilder artifactDescriptorValidatorBuilder = ArtifactDescriptorValidatorBuilder.builder()
        .validateMinMuleVersion().validateMinMuleVersion(() -> toolingVersion)
        .validateMinMuleVersionUsingSemanticVersion();
    artifactPluginDescriptorFactory =
        artifactDescriptorFactoryProvider()
            .createArtifactPluginDescriptorFactory(descriptorLoaderRepository, artifactDescriptorValidatorBuilder);
    artifactPluginDescriptorLoader = new ArtifactPluginDescriptorLoader(artifactPluginDescriptorFactory);

    applicationDescriptorFactory = artifactDescriptorFactoryProvider()
        .createApplicationDescriptorFactory(artifactPluginDescriptorFactory, descriptorLoaderRepository,
                                            artifactDescriptorValidatorBuilder);

    domainDescriptorFactory = new DomainDescriptorFactory(artifactPluginDescriptorLoader, descriptorLoaderRepository,
                                                          artifactDescriptorValidatorBuilder);

    regionPluginClassLoadersFactory = regionPluginClassLoadersFactory(moduleRepository);
    pluginDependenciesResolver =
        artifactDescriptorFactoryProvider().createBundlePluginDependenciesResolver(artifactPluginDescriptorFactory);

    domainClassLoaderFactory =
        new DomainClassLoaderFactory(containerArtifactClassLoader, regionPluginClassLoadersFactory, pluginDependenciesResolver);
    applicationClassLoaderFactory =
        new ApplicationClassLoaderFactory(containerArtifactClassLoader, regionPluginClassLoadersFactory,
                                          pluginDependenciesResolver);
  }

  public Optional<MuleVersion> getTargetMuleVersion() {
    return targetMuleVersion;
  }

  public File getWorkingDirectory() {
    return workingDirectory;
  }

  public RegionPluginClassLoadersFactory getRegionPluginClassLoadersFactory() {
    return regionPluginClassLoadersFactory;
  }

  public DeployableArtifactClassLoaderFactory<ArtifactDescriptor> newTemporaryArtifactClassLoaderFactory() {
    return new TemporaryArtifactClassLoaderFactory();
  }

  public ArtifactClassLoader getContainerArtifactClassLoader() {
    return containerArtifactClassLoader;
  }

  public AbstractArtifactDescriptorFactory<MuleApplicationModel, ApplicationDescriptor> getApplicationDescriptorFactory() {
    return applicationDescriptorFactory;
  }

  public DomainDescriptorFactory getDomainDescriptorFactory() {
    return domainDescriptorFactory;
  }

  public PluginDependenciesResolver getPluginDependenciesResolver() {
    return pluginDependenciesResolver;
  }

  public ArtifactPluginDescriptorLoader getArtifactPluginDescriptorLoader() {
    return artifactPluginDescriptorLoader;
  }

  public DomainClassLoaderFactory getDomainClassLoaderFactory() {
    return domainClassLoaderFactory;
  }

  public ApplicationClassLoaderFactory getApplicationClassLoaderFactory() {
    return applicationClassLoaderFactory;
  }

  /**
   * Creates a class loader instance for a temporary artifact.
   */
  private class TemporaryArtifactClassLoaderFactory
      implements DeployableArtifactClassLoaderFactory<ArtifactDescriptor> {

    /**
     * {@inheritDoc}
     */
    @Override
    public ArtifactClassLoader create(String artifactId, ArtifactClassLoader parent,
                                      ArtifactDescriptor descriptor,
                                      List<ArtifactClassLoader> artifactPluginClassLoaders) {
      return new MuleDeployableArtifactClassLoader(artifactId, descriptor, descriptor.getClassLoaderModel().getUrls(),
                                                   parent.getClassLoader(),
                                                   parent.getClassLoaderLookupPolicy(), artifactPluginClassLoaders);
    }

  }


}
