/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.lang.Boolean.valueOf;
import static java.lang.System.getProperty;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.disposeIfNeeded;

import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.ast.api.ArtifactAst;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Optional;

import javax.validation.constraints.NotNull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Represents an ApplicationModel hierarchy, it could have a parent {@link ToolingApplicationModel}.
 *
 * @since 4.1
 */
public class ToolingApplicationModel {

  private static final String TOOLING_CLIENT_ALLOW_ACCESS_EXTERNAL_RESOURCES =
      "tooling.client.configuration.allowAccessExternalResources";

  private static final Logger LOGGER = LoggerFactory.getLogger(ToolingApplicationModel.class);

  private final ToolingApplicationModel parent;
  private final ArtifactAst muleApplicationModel;
  private final ConfigurationProperties configurationProperties;
  private final ClassLoader classLoader;

  /**
   * Creates an instance of this application model.
   *
   * @param parent                  defines the parent {@link ToolingApplicationModel}, can be null.
   * @param muleApplicationModel    the Mule {@link ArtifactAst}. Non null.
   * @param configurationProperties {@link ConfigurationProperties} for the artifact.
   * @param classLoader             the artifact class loader to resolve resources. Non null.
   */
  public ToolingApplicationModel(ToolingApplicationModel parent, ArtifactAst muleApplicationModel,
                                 ConfigurationProperties configurationProperties, ClassLoader classLoader) {
    requireNonNull(muleApplicationModel, "muleApplicationModel cannot be null");
    this.parent = parent;
    this.muleApplicationModel = muleApplicationModel;
    this.configurationProperties = configurationProperties;
    this.classLoader = classLoader;
  }

  public Optional<ToolingApplicationModel> getParent() {
    return ofNullable(parent);
  }

  public ArtifactAst getMuleApplicationModel() {
    return muleApplicationModel;
  }

  public ConfigurationProperties getConfigurationProperties() {
    return configurationProperties;
  }

  public Optional<URI> findResource(String resource) {
    URI uri = URI.create(resource);
    if (uri.getScheme() == null) {
      return getResourceFromClassLoader(uri.getPath());
    } else {
      if (uri.getScheme().equalsIgnoreCase("resource")) {
        return getResourceFromClassLoader(resource);
      }
    }
    if (valueOf(getProperty(TOOLING_CLIENT_ALLOW_ACCESS_EXTERNAL_RESOURCES, "false"))) {
      return of(uri);
    }
    if (LOGGER.isWarnEnabled()) {
      LOGGER
          .warn("Trying to access an external resources '{}' when Tooling Client is not enabled to access external resources, " +
              "use the system properties '{}' to enable it ", resource, TOOLING_CLIENT_ALLOW_ACCESS_EXTERNAL_RESOURCES);
    }
    return empty();
  }

  @NotNull
  private Optional<URI> getResourceFromClassLoader(String resource) {
    try {
      final URL classLoaderResource = classLoader.getResource(resource);
      if (classLoaderResource != null) {
        return ofNullable(classLoaderResource.toURI());
      }
      return empty();
    } catch (URISyntaxException e) {
      return empty();
    }
  }

  public void dispose() {
    disposeIfNeeded(configurationProperties, LOGGER);
  }
}
