/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dataweave;

import static java.lang.String.format;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JAVA;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.tooling.client.api.el.Severity.ERROR;
import static org.mule.tooling.client.api.el.Severity.WARNING;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import org.mule.datasense.api.metadataprovider.ToolingHelper;
import org.mule.runtime.api.el.DataFormat;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.util.LazyValue;
import org.mule.tooling.client.api.dataweave.DataWeaveModuleOption;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.api.dataweave.validation.DataWeaveValidationRequest;
import org.mule.tooling.client.api.el.Location;
import org.mule.tooling.client.api.el.Position;
import org.mule.tooling.client.api.el.ValidationMessage;
import org.mule.tooling.client.api.el.ValidationResult;
import org.mule.tooling.client.internal.Command;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class DefaultDataWeaveService implements DataWeaveService, Command {

  private final LazyValue<ClassLoader> executionClassLaoder;
  private ExpressionLanguageMetadataService expressionLanguageMetadataService;
  private ModulesAnalyzer modulesAnalyzer;
  private DataWeaveRunner runner;
  private Serializer serializer;

  public DefaultDataWeaveService(LazyValue<ClassLoader> executionClassLoader, DataWeaveRunner runner,
                                 ExpressionLanguageMetadataService expressionLanguageMetadataService,
                                 ModulesAnalyzer modulesAnalyzer, Serializer serializer) {
    this.executionClassLaoder = executionClassLoader;
    this.expressionLanguageMetadataService = expressionLanguageMetadataService;
    this.runner = runner;
    this.modulesAnalyzer = modulesAnalyzer;
    this.serializer = serializer;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeavePreviewResponse execute(DataWeavePreviewRequest parameters) {
    final EventModel event = parameters.getEvent();
    if (event != null && anyInputIsJavaBased(event)) {
      throw new IllegalArgumentException("Java input not supported, serialize to DW script");
    }
    return runner.execute(parameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValidationResult validate(DataWeaveValidationRequest request) {
    return withContextClassLoader(executionClassLaoder.get(), () -> {
      List<ValidationMessage> validationMessages = new ArrayList<>();
      expressionLanguageMetadataService.getOutputType(
                                                      ToolingHelper
                                                          .resolveTypeBindings(request.getEventType(),
                                                                               request.getGlobalBindings(),
                                                                               request.getFunctionBindings()),
                                                      request.getScript(),
                                                      new ExpressionLanguageMetadataService.MessageCallback() {

                                                        @Override
                                                        public void warning(String message,
                                                                            ExpressionLanguageMetadataService.MessageLocation location) {
                                                          validationMessages
                                                              .add(new ValidationMessage(WARNING, message,
                                                                                         toLocation(location)));

                                                        }

                                                        @Override
                                                        public void error(String message,
                                                                          ExpressionLanguageMetadataService.MessageLocation location) {
                                                          validationMessages
                                                              .add(new ValidationMessage(ERROR, message,
                                                                                         toLocation(location)));
                                                        }

                                                      });
      return new ValidationResult(validationMessages);
    });
  }

  private Location toLocation(ExpressionLanguageMetadataService.MessageLocation location) {
    final ExpressionLanguageMetadataService.MessagePosition startPosition = location.getStartPosition();
    final ExpressionLanguageMetadataService.MessagePosition endPosition = location.getEndPosition();
    return new Location(new Position(startPosition.getLine(), startPosition.getColumn(), startPosition.getOffset()),
                        new Position(endPosition.getLine(), endPosition.getColumn(), endPosition.getOffset()));
  }

  private boolean anyInputIsJavaBased(final EventModel event) {
    final MessageModel message = event.getMessage();
    if (isJavaBased(message.getPayload()) || isJavaBased(message.getAttributes())) {
      return true;
    }

    final Map<String, TypedValueModel> variables = event.getVariables();
    if (variables != null && variables.values().stream().anyMatch(this::isJavaBased)) {
      return true;
    }

    // otherwise
    return false;
  }

  private boolean isJavaBased(final TypedValueModel model) {
    return model != null && MediaType.parse(model.getDataType().getMediaType()).equals(APPLICATION_JAVA);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<DataWeaveModuleOption> getReaderOptions(String contentType) {
    return modulesAnalyzer.getOptions(contentType, DataFormat::getReaderOptions);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<DataWeaveModuleOption> getWriterOptions(String contentType) {
    return modulesAnalyzer.getOptions(contentType, DataFormat::getWriterOptions);
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "execute": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(DataWeavePreviewRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(execute(serializer.deserialize(arguments[0])));
      }
      case "validate": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(DataWeaveValidationRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(validate(serializer.deserialize(arguments[0])));
      }
      case "getReaderOptions": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(getReaderOptions(serializer.deserialize(arguments[0])));
      }
      case "getWriterOptions": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(getWriterOptions(serializer.deserialize(arguments[0])));
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
