/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ComponentModelVisitor;
import org.mule.runtime.api.meta.model.OutputModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModelVisitor;
import org.mule.runtime.api.meta.model.nested.NestedChainModel;
import org.mule.runtime.api.meta.model.nested.NestedComponentModel;
import org.mule.runtime.api.meta.model.nested.NestedRouteModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.extension.api.model.ImmutableOutputModel;
import org.mule.runtime.extension.api.model.construct.ImmutableConstructModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedChainModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedComponentModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedRouteModel;
import org.mule.runtime.extension.api.model.operation.ImmutableOperationModel;
import org.mule.runtime.extension.api.model.parameter.ImmutableParameterGroupModel;
import org.mule.runtime.extension.api.model.parameter.ImmutableParameterModel;
import org.mule.runtime.extension.api.model.source.ImmutableSourceModel;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;

public class ComponentModelMediator<T extends ComponentModel, M> {

  private static final Logger LOGGER = getLogger(ComponentModelMediator.class);

  private ToolingCacheIdGenerator<M> cacheIdGenerator;
  private Map<String, MetadataType> metadataTypesCache;
  private M component;
  private String componentIdentifier;
  private T componentModel;

  public ComponentModelMediator(Map<String, MetadataType> metadataTypesCache,
                                ToolingCacheIdGenerator<M> cacheIdGenerator,
                                M component,
                                String componentIdentifier,
                                T componentModel) {
    this.metadataTypesCache = metadataTypesCache;
    this.cacheIdGenerator = cacheIdGenerator;
    this.component = component;
    this.componentIdentifier = componentIdentifier;
    this.componentModel = componentModel;
  }

  public Optional<T> enrichComponentModel() {
    Reference<T> filteredModel = new Reference<>();
    try {
      componentModel.accept(new ComponentModelVisitor() {

        @Override
        public void visit(ConstructModel constructModel) {
          filteredModel.set((T) resolveConstructModel(constructModel));
        }

        @Override
        public void visit(OperationModel operationModel) {
          filteredModel.set((T) resolveOperationModel(operationModel));
        }

        @Override
        public void visit(SourceModel sourceModel) {
          filteredModel.set((T) resolveSourceModel(sourceModel));
        }

        @Override
        public void visit(NestedComponentModel model) {
          // no - op
        }

        @Override
        public void visit(NestedChainModel model) {
          // no - op
        }

        @Override
        public void visit(NestedRouteModel model) {
          // no - op
        }
      });
    } catch (MissingDynamicMetadataException e) {
      return empty();
    }

    return ofNullable(filteredModel.get());
  }

  private List<ParameterGroupModel> resolveParameterGroupModels(List<ParameterGroupModel> parameterGroupModels) {
    return parameterGroupModels.stream()
        .map(parameterGroupModel -> new ImmutableParameterGroupModel(parameterGroupModel.getName(),
                                                                     parameterGroupModel.getDescription(),
                                                                     resolveParameterModels(parameterGroupModel
                                                                         .getParameterModels()),
                                                                     parameterGroupModel.getExclusiveParametersModels(),
                                                                     parameterGroupModel.isShowInDsl(),
                                                                     parameterGroupModel.getDisplayModel().orElse(null),
                                                                     parameterGroupModel.getLayoutModel().orElse(null),
                                                                     parameterGroupModel.getModelProperties()))
        .collect(toList());
  }

  private List<ParameterModel> resolveParameterModels(List<ParameterModel> parameterModels) {
    return parameterModels.stream()
        .map(parameterModel -> {
          MetadataType type = parameterModel.getType();
          if (parameterModel.hasDynamicType()) {
            String componentInputMetadataKey =
                cacheIdGenerator.getIdForComponentInputMetadata(component, parameterModel.getName())
                    .orElseThrow(() -> new IllegalArgumentException(format("Couldn't generate a MetadataCacheId for %s parameter name: %s",
                                                                           componentIdentifier,
                                                                           parameterModel.getName())));
            if (!metadataTypesCache.containsKey(componentInputMetadataKey)) {
              if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(format("Dynamic MetadataType for parameter: %s (componentModel: %s) not present in cache",
                                    parameterModel.getName(), componentIdentifier));
              }
              throw new MissingDynamicMetadataException();
            }
            type = metadataTypesCache.get(componentInputMetadataKey);
          }
          return new ImmutableParameterModel(parameterModel.getName(),
                                             parameterModel.getDescription(),
                                             type,
                                             parameterModel.hasDynamicType(),
                                             parameterModel.isRequired(),
                                             parameterModel.isOverrideFromConfig(),
                                             parameterModel.isComponentId(),
                                             parameterModel.getExpressionSupport(),
                                             parameterModel.getDefaultValue(),
                                             parameterModel.getRole(),
                                             parameterModel.getDslConfiguration(),
                                             parameterModel.getDisplayModel()
                                                 .orElse(null),
                                             parameterModel.getLayoutModel()
                                                 .orElse(null),
                                             parameterModel.getValueProviderModel()
                                                 .orElse(null),
                                             parameterModel.getAllowedStereotypes(),
                                             parameterModel.getModelProperties(),
                                             parameterModel.getDeprecationModel()
                                                 .orElse(null));
        })
        .collect(toList());
  }

  private List<? extends NestableElementModel> resolveNestableElementModels(List<? extends NestableElementModel> components) {
    return components.stream()
        .map(this::resolveNestedComponentModel)
        .collect(toList());
  }

  private NestableElementModel resolveNestedComponentModel(NestableElementModel nestedModel) {
    final NestableElementModelFilterVisitor nestableElementModelVisitor =
        new NestableElementModelFilterVisitor();
    nestedModel.accept(nestableElementModelVisitor);
    return nestableElementModelVisitor.getNestableElementModelFiltered();
  }

  private class NestableElementModelFilterVisitor implements NestableElementModelVisitor {

    private NestableElementModel nestableElementModel = null;

    public NestableElementModel getNestableElementModelFiltered() {
      return nestableElementModel;
    }

    @Override
    public void visit(NestedComponentModel nestedComponentModel) {
      nestableElementModel = new ImmutableNestedComponentModel(
                                                               nestedComponentModel.getName(),
                                                               nestedComponentModel.getDescription(),
                                                               resolveParameterGroupModels(nestedComponentModel
                                                                   .getParameterGroupModels()),
                                                               nestedComponentModel.getMinOccurs(),
                                                               nestedComponentModel.getMaxOccurs().orElse(null),
                                                               nestedComponentModel.getAllowedStereotypes(),
                                                               resolveNestableElementModels(nestableElementModel
                                                                   .getNestedComponents()),
                                                               nestedComponentModel.getDisplayModel().orElse(null),
                                                               nestedComponentModel.getErrorModels(),
                                                               nestedComponentModel.getStereotype(),
                                                               nestedComponentModel.getModelProperties(),
                                                               nestedComponentModel.getDeprecationModel().orElse(null),
                                                               nestedComponentModel.getSemanticTerms());
    }

    @Override
    public void visit(NestedChainModel nestedChainModel) {
      nestableElementModel = new ImmutableNestedChainModel(
                                                           nestedChainModel.getName(),
                                                           nestedChainModel.getDescription(),
                                                           resolveParameterGroupModels(nestedChainModel
                                                               .getParameterGroupModels()),
                                                           nestedChainModel.isRequired(),
                                                           nestedChainModel.getAllowedStereotypes(),
                                                           resolveNestableElementModels(nestedChainModel.getNestedComponents()),
                                                           nestedChainModel.getDisplayModel().orElse(null),
                                                           nestedChainModel.getErrorModels(),
                                                           nestedChainModel.getStereotype(),
                                                           nestedChainModel.getModelProperties(),
                                                           nestedChainModel.getDeprecationModel().orElse(null),
                                                           nestedChainModel.getSemanticTerms());
    }

    @Override
    public void visit(NestedRouteModel nestedRouteModel) {
      nestableElementModel = new ImmutableNestedRouteModel(
                                                           nestedRouteModel.getName(),
                                                           nestedRouteModel.getDescription(),
                                                           resolveParameterGroupModels(nestedRouteModel
                                                               .getParameterGroupModels()),
                                                           nestedRouteModel.getDisplayModel().orElse(null),
                                                           nestedRouteModel.getMinOccurs(),
                                                           nestedRouteModel.getMaxOccurs().orElse(null),
                                                           resolveNestableElementModels(nestedRouteModel.getNestedComponents()),
                                                           nestedRouteModel.getStereotype(),
                                                           nestedRouteModel.getModelProperties(),
                                                           nestedRouteModel.getDeprecationModel().orElse(null),
                                                           nestedRouteModel.getSemanticTerms());
    }
  }


  private ImmutableOperationModel resolveOperationModel(OperationModel operationModel) {
    return new ImmutableOperationModel(operationModel.getName(),
                                       operationModel.getDescription(),
                                       resolveParameterGroupModels(operationModel.getParameterGroupModels()),
                                       resolveNestableElementModels(operationModel.getNestedComponents()),
                                       resolveOutputModel(operationModel.getOutput()),
                                       resolveOutputAttributesModel(operationModel.getOutputAttributes()),
                                       operationModel.isBlocking(),
                                       operationModel.getExecutionType(),
                                       operationModel.requiresConnection(),
                                       operationModel.isTransactional(),
                                       operationModel.supportsStreaming(),
                                       operationModel.getDisplayModel().orElse(null),
                                       operationModel.getErrorModels(),
                                       operationModel.getStereotype(),
                                       operationModel.getModelProperties(),
                                       operationModel.getNotificationModels(),
                                       operationModel.getDeprecationModel().orElse(null));
  }

  private OutputModel resolveOutputModel(OutputModel outputModel) {
    MetadataType type = outputModel.getType();
    if (outputModel.hasDynamicType()) {
      String componentOutputMetadataKey = cacheIdGenerator.getIdForComponentOutputMetadata(component)
          .orElseThrow(() -> new IllegalArgumentException(format("Couldn't generate a MetadataCacheId for %s output",
                                                                 componentIdentifier)));
      if (!metadataTypesCache.containsKey(componentOutputMetadataKey)) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug(format("Dynamic output MetadataType for componentModel: %s not present in cache",
                              componentIdentifier));
        }
        throw new MissingDynamicMetadataException();
      }
      type = metadataTypesCache.get(componentOutputMetadataKey);
    }
    return new ImmutableOutputModel(outputModel.getDescription(), type, outputModel.hasDynamicType(),
                                    outputModel.getModelProperties());
  }

  private OutputModel resolveOutputAttributesModel(OutputModel outputModel) {
    MetadataType type = outputModel.getType();
    if (outputModel.hasDynamicType()) {
      String componentAttributesMetadataKey = cacheIdGenerator.getIdForComponentAttributesMetadata(component)
          .orElseThrow(() -> new IllegalArgumentException(format("Couldn't generate a MetadataCacheId for %s output attributes",
                                                                 componentIdentifier)));
      if (!metadataTypesCache.containsKey(componentAttributesMetadataKey)) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug(format("Dynamic output attributes MetadataType for componentModel: %s not present in cache",
                              componentIdentifier));
        }
        throw new MissingDynamicMetadataException();
      }
      type = metadataTypesCache.get(componentAttributesMetadataKey);
    }
    return new ImmutableOutputModel(outputModel.getDescription(), type, outputModel.hasDynamicType(),
                                    outputModel.getModelProperties());
  }

  private ImmutableSourceModel resolveSourceModel(SourceModel sourceModel) {
    return new ImmutableSourceModel(sourceModel.getName(),
                                    sourceModel.getDescription(),
                                    sourceModel.hasResponse(),
                                    sourceModel.runsOnPrimaryNodeOnly(),
                                    resolveParameterGroupModels(sourceModel.getParameterGroupModels()),
                                    resolveNestableElementModels(sourceModel.getNestedComponents()),
                                    resolveOutputModel(sourceModel.getOutput()),
                                    resolveOutputAttributesModel(sourceModel.getOutputAttributes()),
                                    // TODO: MULE-17263
                                    sourceModel.getSuccessCallback(),
                                    // TODO: MULE-17263
                                    sourceModel.getErrorCallback(),
                                    sourceModel.getTerminateCallback(),
                                    sourceModel.requiresConnection(),
                                    sourceModel.isTransactional(),
                                    sourceModel.supportsStreaming(),
                                    sourceModel.getDisplayModel().orElse(null),
                                    sourceModel.getStereotype(),
                                    sourceModel.getErrorModels(), sourceModel.getModelProperties(),
                                    sourceModel.getNotificationModels(),
                                    sourceModel.getDeprecationModel().orElse(null));
  }


  private ImmutableConstructModel resolveConstructModel(ConstructModel constructModel) {
    return new ImmutableConstructModel(constructModel.getName(),
                                       constructModel.getDescription(),
                                       resolveParameterGroupModels(constructModel.getParameterGroupModels()),
                                       resolveNestableElementModels(constructModel.getNestedComponents()),
                                       constructModel.allowsTopLevelDeclaration(),
                                       constructModel.getDisplayModel().orElse(null),
                                       constructModel.getErrorModels(),
                                       constructModel.getStereotype(),
                                       constructModel.getModelProperties(),
                                       constructModel.getDeprecationModel().orElse(null));
  }

  public static class MissingDynamicMetadataException extends RuntimeException {
    // Marker interface
  }

}
