/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.config.api.dsl.model.metadata.ComponentBasedValueProviderCacheIdGenerator;
import org.mule.runtime.config.api.dsl.model.metadata.ModelBasedMetadataCacheIdGeneratorFactory;
import org.mule.runtime.core.internal.locator.ComponentLocator;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheIdGenerator;
import org.mule.runtime.core.internal.value.cache.ValueProviderCacheIdGenerator;
import org.mule.tooling.client.internal.values.ValueProviderCache;

import java.util.Optional;

import com.vdurmont.semver4j.Semver;
import com.vdurmont.semver4j.SemverException;
import org.slf4j.Logger;

/**
 * Generates keys to use in a {@link MetadataCache} and {@link ValueProviderCache}
 *
 * @since 4.4.0
 */
public class DefaultToolingCacheIdGenerator implements ToolingCacheIdGenerator<ComponentAst> {

  private static final Logger LOGGER = getLogger(DefaultToolingCacheIdGenerator.class);
  private static final String ENRICHED_PART_SEPARATOR = "#";

  private static final ModelBasedMetadataCacheIdGeneratorFactory cacheIdGeneratorFactory =
      new ModelBasedMetadataCacheIdGeneratorFactory();

  private MetadataCacheIdGenerator<ComponentAst> metadataDelegate;
  private ValueProviderCacheIdGenerator<ComponentAst> valueProviderDelegate;
  private DslResolvingContext dslResolvingContext;

  /**
   * Returns whether or not the {@link String}s represent keys that are related. That is true if the @param child has a reference
   * to it's parent.
   * </p>
   * For example, if an operation (child) has a reference to a config (parent)
   *
   * @param parentValue
   * @param childValue
   * @return {@link Boolean#TRUE} if they are related, {@link Boolean#FALSE} otherwise.
   */
  public static boolean areRelated(String parentValue, String childValue) {
    Pair<String, String> parentSplitInfo = splitInformation(parentValue); // EnrichedValue, DefaultValue
    Pair<String, String> childSplitInfo = splitInformation(childValue); // EnrichedValue, DefaultValue
    if (parentSplitInfo.getFirst().equals(childSplitInfo.getFirst())) {
      return childSplitInfo.getSecond().contains(parentSplitInfo.getSecond());
    }
    return false;
  }

  private static Pair<String, String> splitInformation(String idValue) {
    String enrichedPart = "";
    String valuePart;
    String[] parts = idValue.split(ENRICHED_PART_SEPARATOR);
    if (parts.length == 1) {
      valuePart = parts[0];
    } else {
      enrichedPart = parts[0];
      valuePart = parts[1];
    }
    return new Pair<>(enrichedPart, valuePart);
  }

  private String betweenBrackets(String value) {
    return "[" + value + "]";
  }

  public DefaultToolingCacheIdGenerator(DslResolvingContext context,
                                        ComponentLocator<ComponentAst> locator) {
    this.dslResolvingContext = context;
    this.metadataDelegate = cacheIdGeneratorFactory.create(context, locator);
    this.valueProviderDelegate = new ComponentBasedValueProviderCacheIdGenerator(context, locator);
  }

  public Optional<String> getIdForComponentOutputMetadata(ComponentAst component) {
    return metadataDelegate.getIdForComponentOutputMetadata(component).map(id -> generateCacheId(component, id.getValue()));
  }

  public Optional<String> getIdForComponentAttributesMetadata(ComponentAst component) {
    return metadataDelegate.getIdForComponentAttributesMetadata(component).map(id -> generateCacheId(component, id.getValue()));
  }

  public Optional<String> getIdForComponentInputMetadata(ComponentAst component, String parameterName) {
    return metadataDelegate.getIdForComponentInputMetadata(component, parameterName)
        .map(id -> generateCacheId(component, id.getValue()));
  }

  public Optional<String> getIdForComponentMetadata(ComponentAst component) {
    return metadataDelegate.getIdForComponentMetadata(component).map(id -> generateCacheId(component, id.getValue()));
  }

  public Optional<String> getIdForMetadataKeys(ComponentAst component) {
    return metadataDelegate.getIdForMetadataKeys(component).map(id -> generateCacheId(component, id.getValue()));
  }

  public Optional<String> getIdForGlobalMetadata(ComponentAst component) {
    return metadataDelegate.getIdForGlobalMetadata(component).map(id -> generateCacheId(component, id.getValue()));
  }

  @Override
  public Optional<String> getIdForResolvedValues(ComponentAst component, String parameterName) {
    return valueProviderDelegate.getIdForResolvedValues(component, parameterName)
        .map(id -> generateCacheId(component, id.getValue()));
  }

  @Override
  public Optional<String> getIdForResolvedValues(ComponentAst component, String parameterName, String targetSelector) {
    return valueProviderDelegate.getIdForResolvedValues(component, parameterName, targetSelector)
        .map(id -> generateCacheId(component, id.getValue()));
  }

  private String generateCacheId(ComponentAst componentAst,
                                 String originalId) {
    Reference<String> idToReturn = new Reference<>(originalId);
    getExtensionModel(componentAst).ifPresent(em -> {
      try {
        String major = getMajor(em.getVersion());
        idToReturn.set(betweenBrackets(em.getName().toLowerCase()) +
            "-" +
            betweenBrackets(major) +
            ENRICHED_PART_SEPARATOR +
            originalId);
      } catch (SemverException e) {
        if (LOGGER.isErrorEnabled()) {
          LOGGER.error("Unable to properly create a cache key id for " + componentAst
              + " ,extension model version does not follow semver, was: "
              + em.getVersion());
        }
      }
    });
    return idToReturn.get();
  }

  private Optional<ExtensionModel> getExtensionModel(ComponentAst componentAst) {
    return dslResolvingContext.getExtensions().stream()
        .filter(em -> componentAst.getIdentifier().getNamespace().equals(em.getXmlDslModel().getPrefix())).findAny();
  }

  private String getMajor(String version) {
    Semver semver = new Semver(version);
    return semver.getMajor().toString();
  }
}
