/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.lang.String.format;
import static java.util.stream.Collectors.toSet;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.failure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.success;
import static org.mule.tooling.client.internal.metadata.DefaultToolingCacheIdGenerator.areRelated;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.HasOutputModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataFailure;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.tooling.client.api.exception.ToolingException;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;


public class DefaultToolingMetadataCache implements MetadataCache {

  private Map<String, MetadataType> metadataTypesCache;
  private Map<String, List<MetadataFailure>> failureMetadataComponentCache;
  private Map<String, MetadataResult<MetadataKeysContainer>> metadataKeysCache;
  private ToolingCacheIdGenerator<ComponentAst> cacheIdGenerator;

  public DefaultToolingMetadataCache(Map<String, MetadataType> metadataTypesCache,
                                     Map<String, List<MetadataFailure>> failureMetadataComponentCache,
                                     Map<String, MetadataResult<MetadataKeysContainer>> metadataKeysCache,
                                     ToolingCacheIdGenerator<ComponentAst> metadataCacheIdGenerator) {
    this.metadataTypesCache = metadataTypesCache;
    this.failureMetadataComponentCache = failureMetadataComponentCache;
    this.metadataKeysCache = metadataKeysCache;
    this.cacheIdGenerator = metadataCacheIdGenerator;
  }

  @Override
  public MetadataResult<OperationModel> getOperationMetadata(ComponentAst componentAst,
                                                             Callable<MetadataResult<OperationModel>> resolver) {
    OperationModel operationModel = componentAst.getModel(OperationModel.class)
        .orElseThrow(() -> new IllegalArgumentException(format("Operation model should be present for component: %s",
                                                               componentAst.getLocation())));

    return getComponentMetadata(operationModel, componentAst, resolver);
  }

  @Override
  public MetadataResult<SourceModel> getSourceMetadata(ComponentAst componentAst,
                                                       Callable<MetadataResult<SourceModel>> resolver) {
    SourceModel sourceModel = componentAst.getModel(SourceModel.class)
        .orElseThrow(() -> new IllegalArgumentException(format("Source model should be present for component: %s",
                                                               componentAst.getLocation())));

    return getComponentMetadata(sourceModel, componentAst, resolver);
  }

  private <T extends ComponentModel> MetadataResult<T> getComponentMetadata(T staticComponentModel, ComponentAst componentAst,
                                                                            Callable<MetadataResult<T>> resolver) {
    String failureComponentKey = cacheIdGenerator.getIdForComponentMetadata(componentAst)
        .orElseThrow(() -> new IllegalArgumentException(format("Couldn't create a metadata cache id for component: %s",
                                                               componentAst.getLocation())));
    if (failureMetadataComponentCache.containsKey(failureComponentKey)) {
      return failure(failureMetadataComponentCache.get(failureComponentKey));
    }

    ComponentModelMediator<T, ComponentAst> componentModelMediator =
        new ComponentModelMediator<>(metadataTypesCache,
                                     cacheIdGenerator,
                                     componentAst,
                                     componentAst.getLocation().toString(),
                                     staticComponentModel);

    Optional<T> enrichComponentModel = componentModelMediator.enrichComponentModel();
    if (enrichComponentModel.isPresent()) {
      return success(enrichComponentModel.get());
    }

    MetadataResult<T> result = executeHandling(resolver);
    if (!result.isSuccess()) {
      failureMetadataComponentCache.putIfAbsent(failureComponentKey, result.getFailures());
      return result;
    }
    new MetadataCachePopulator<>(metadataTypesCache, cacheIdGenerator, componentAst,
                                 result.get()).populateCache();
    return result;
  }


  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(ComponentAst componentAst,
                                                               Callable<MetadataResult<MetadataKeysContainer>> resolver) {
    return cacheIdGenerator.getIdForMetadataKeys(componentAst).map(
                                                                   id -> metadataKeysCache.computeIfAbsent(id,
                                                                                                           k -> executeHandling(resolver)))
        .orElseThrow(() -> new IllegalArgumentException(format("Couldn't create a metadata cache id for component: %s",
                                                               componentAst.getLocation())));
  }

  @Override
  public void dispose(ComponentAst componentAst) {
    Optional<String> componentCacheId = cacheIdGenerator.getIdForComponentMetadata(componentAst);

    // Remove cached component in case there is a failure present
    componentCacheId.ifPresent(cid -> failureMetadataComponentCache.keySet().remove(cid));


    componentAst
        .getModel(ConfigurationModel.class)
        .map(cm -> {
          disposeConfig(componentAst);
          return null;
        })
        .orElseGet(() -> {
          componentAst.getModel(ParameterizedModel.class).ifPresent(pm -> disposeTypes(componentAst, pm.getAllParameterModels()));
          return null;
        });

  }

  @Override
  public void invalidateMetadataKeysFor(ComponentAst componentAst) {
    Optional<String> componentKeysCacheId = cacheIdGenerator.getIdForMetadataKeys(componentAst);
    componentKeysCacheId.ifPresent(cid -> metadataKeysCache.keySet().remove(cid));
  }

  private void disposeConfig(ComponentAst configAst) {
    Optional<String> configId = cacheIdGenerator.getIdForComponentMetadata(configAst);
    configId.ifPresent(
                       id -> {
                         disposeAllConfigRelated(id, metadataKeysCache);
                         disposeAllConfigRelated(id, metadataTypesCache);
                       });
  }

  private void disposeAllConfigRelated(String configId, Map<String, ?> externalCache) {
    externalCache.keySet().removeIf(k -> areRelated(configId, k));
  }

  private void disposeTypes(ComponentAst component, List<ParameterModel> parameterModels) {
    Set<String> typesToRemove = new HashSet<>();

    // TODO: MULE-17263
    // dispose Source callbacks if they are cached

    // dispose output types
    component.getModel(HasOutputModel.class).ifPresent(om -> {
      if (om.getOutput().hasDynamicType()) {
        cacheIdGenerator.getIdForComponentOutputMetadata(component).ifPresent(typesToRemove::add);
      }
      if (om.getOutputAttributes().hasDynamicType()) {
        cacheIdGenerator.getIdForComponentAttributesMetadata(component).ifPresent(typesToRemove::add);
      }
    });

    // dispose input types
    for (ParameterModel parameterModel : parameterModels) {
      if (parameterModel.hasDynamicType()) {
        cacheIdGenerator.getIdForComponentInputMetadata(component, parameterModel.getName())
            .ifPresent(typesToRemove::add);
      }
    }

    metadataTypesCache.keySet().removeAll(typesToRemove);

    // dispose keys
    cacheIdGenerator.getIdForMetadataKeys(component).ifPresent(id -> metadataKeysCache.keySet().remove(id));
  }

  private <T> MetadataResult<T> executeHandling(Callable<MetadataResult<T>> callable) {
    try {
      return callable.call();
    } catch (Exception e) {
      throw new ToolingException(e);
    }
  }
}
