/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.service;

import static java.lang.String.format;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;
import static org.mule.runtime.core.api.util.StringUtils.isEmpty;

import org.mule.runtime.api.el.DefaultExpressionLanguageFactoryService;
import org.mule.runtime.api.el.ExpressionLanguageCapabilitiesService;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;
import org.mule.runtime.api.service.Service;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.apikit.metadata.api.MetadataService;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of service registry for Tooling.
 */
public class DefaultServiceRegistry implements ServiceRegistry {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultServiceRegistry.class);

  private List<Service> services;

  private LazyValue<ExpressionLanguageMetadataService> expressionLanguageMetadataService;
  private LazyValue<ExpressionLanguageCapabilitiesService> expressionLanguageCapabilitiesService;
  private LazyValue<MetadataService> apikitMetadataService;
  private LazyValue<DefaultExpressionLanguageFactoryService> defaultExpressionLanguageFactoryService;

  public DefaultServiceRegistry(List<Service> services) {
    this.services = new ArrayList<>(services);
    initServices();
    startServices();
  }

  private void initServices() {
    expressionLanguageMetadataService = new LazyValue<>(() -> (ExpressionLanguageMetadataService) services.stream()
        .filter(service -> service.getContractName().equals(ExpressionLanguageMetadataService.class.getSimpleName()))
        .findFirst().orElseThrow(() -> new RuntimeException("Weave expression language metadata service cannot be found")));

    expressionLanguageCapabilitiesService = new LazyValue<>(() -> (ExpressionLanguageCapabilitiesService) services.stream()
        .filter(service -> service.getContractName().equals(ExpressionLanguageCapabilitiesService.class.getSimpleName()))
        .findFirst().orElseThrow(() -> new RuntimeException("Weave expression language capabilities service cannot be found")));

    apikitMetadataService = new LazyValue<>(() -> ((MetadataService) services.stream()
        .filter(service -> service.getContractName().equals(MetadataService.class.getSimpleName()))
        .findFirst().orElseThrow(() -> new RuntimeException("Apikit metadata service cannot be found"))));

    defaultExpressionLanguageFactoryService = new LazyValue<>(() -> (DefaultExpressionLanguageFactoryService) services
        .stream()
        .filter(service -> service.getContractName().equals(DefaultExpressionLanguageFactoryService.class.getSimpleName()))
        .findFirst()
        .orElseThrow(() -> new RuntimeException("Default Expression Language Factory service cannot be found")));

  }

  private void startServices() {
    for (Service service : services) {
      try {
        startIfNeeded(service);
      } catch (MuleException e) {
        if (!isEmpty(service.getContractName())) {
          LOGGER
              .warn(format("Failed to start service '%s - %s': %s", service.getName(), service.getContractName(), e.getMessage()),
                    e);
        } else {
          LOGGER.warn(format("Failed to start service '%s': %s", service.getName(), e.getMessage()),
                      e);
        }
      }
    }
  }

  @Override
  public void dispose() {
    services.forEach(service -> {
      try {
        stopIfNeeded(service);
      } catch (MuleException e) {
        LOGGER.warn(format("Failed to stop service '%s': %s", service.getName(), e.getMessage()), e);
      }
    });
  }

  @Override
  public ExpressionLanguageMetadataService getExpressionLanguageMetadataService() {
    return expressionLanguageMetadataService.get();
  }

  @Override
  public ExpressionLanguageCapabilitiesService getExpressionLanguageCapabilitiesService() {
    return expressionLanguageCapabilitiesService.get();
  }

  @Override
  public MetadataService getApikitMetadataService() {
    return apikitMetadataService.get();
  }

  @Override
  public DefaultExpressionLanguageFactoryService getDefaultExpressionLanguageFactoryService() {
    return defaultExpressionLanguageFactoryService.get();
  }

}
