/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.cache;

import static java.lang.String.format;
import static java.util.Comparator.comparingInt;
import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.api.metadata.resolving.FailureCode.INVALID_METADATA_KEY;
import static org.mule.runtime.api.metadata.resolving.MetadataFailure.Builder.newFailure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.failure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.success;
import static org.mule.tooling.client.internal.utils.FunctionalUtils.executeHandling;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.SimpleType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.EnrichableModel;
import org.mule.runtime.api.meta.model.HasOutputModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataTypesDescriptor;
import org.mule.runtime.api.metadata.descriptor.InputMetadataDescriptor;
import org.mule.runtime.api.metadata.descriptor.OutputMetadataDescriptor;
import org.mule.runtime.api.metadata.descriptor.ParameterMetadataDescriptor;
import org.mule.runtime.api.metadata.descriptor.TypeMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.FailureCode;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.app.declaration.api.ElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterGroupElementDeclaration;
import org.mule.runtime.core.internal.util.OneTimeWarning;
import org.mule.runtime.extension.api.property.MetadataKeyIdModelProperty;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;
import org.mule.runtime.module.tooling.internal.utils.ArtifactHelperUtils;
import org.mule.sdk.api.annotation.metadata.MetadataKeyId;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyIdModel;
import org.mule.tooling.client.internal.metadata.ComponentModelMediator;
import org.mule.tooling.client.internal.metadata.ToolingCacheIdGenerator;
import org.mule.tooling.client.internal.session.ExtensionModelProvider;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import net.sf.saxon.ma.trie.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultDeclarationMetadataCache implements DeclarationMetadataCache {

  private Map<String, MetadataType> typesStorage;
  private Map<String, MetadataResult<MetadataKeysContainer>> keysStorage;
  private ToolingCacheIdGenerator<ElementDeclaration> cacheIdGenerator;
  private ExtensionModelProvider extensionModelProvider;

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultDeclarationMetadataCache.class);

  public DefaultDeclarationMetadataCache(ToolingCacheIdGenerator<ElementDeclaration> cacheIdGenerator,
                                         Map<String, MetadataType> typesStorage,
                                         Map<String, MetadataResult<MetadataKeysContainer>> keysStorage,
                                         ExtensionModelProvider extensionModelProvider) {
    this.cacheIdGenerator = cacheIdGenerator;
    this.typesStorage = typesStorage;
    this.keysStorage = keysStorage;
    this.extensionModelProvider = extensionModelProvider;
  }

  @Override
  public MetadataResult<ComponentMetadataTypesDescriptor> getComponentMetadata(ComponentElementDeclaration<?> componentElementDeclaration,
                                                                               Callable<MetadataResult<ComponentMetadataTypesDescriptor>> resolver,
                                                                               boolean forceResolution) {
    ComponentModel staticComponentModel = getModel(componentElementDeclaration);

    Optional<? extends ComponentModel> enrichedComponentModel = empty();
    if (!forceResolution) {
      ComponentModelMediator<? extends ComponentModel, ElementDeclaration> componentModelMediator =
          new ComponentModelMediator<>(typesStorage,
                                       cacheIdGenerator,
                                       componentElementDeclaration,
                                       componentElementDeclaration.getName(),
                                       staticComponentModel);
      enrichedComponentModel = componentModelMediator.enrichComponentModel();
    }
    if (enrichedComponentModel.isPresent()) {
      return success(descriptorFromModel(enrichedComponentModel.get()));
    }

    MetadataResult<ComponentMetadataTypesDescriptor> result =
        executeHandling(resolver, e -> failure(newFailure(e).onComponent()));
    if (result.isSuccess()) {
      populateTypesCache(componentElementDeclaration, result.get());
    }
    return result;
  }

  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(ComponentElementDeclaration elementDeclaration,
                                                               Callable<MetadataResult<MetadataKeysContainer>> resolver,
                                                               boolean forceResolution) {
    return doGetMetadataKeys(elementDeclaration, resolver, forceResolution);
  }

  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeysPartialFetch(ParameterizedModel parameterizedModel,
                                                                           ComponentElementDeclaration elementDeclaration,
                                                                           Callable<MetadataResult<MetadataKeysContainer>> resolver,
                                                                           boolean forceResolution) {
    MetadataResult<MetadataKeysContainer> metadataKeysContainerMetadataResult =
        doGetMetadataKeys(elementDeclaration, resolver, forceResolution);
    if (forceResolution && !metadataKeysContainerMetadataResult.isSuccess()
        && metadataKeysContainerMetadataResult.getFailures().stream()
            .filter(f -> f.getFailureCode().equals(INVALID_METADATA_KEY)).findAny().isPresent()) {
      // if there was an error means that the upper levels of the key may have been removed from external system so we need to
      // invalidate them from cache
      if (!(parameterizedModel instanceof EnrichableModel)) {
        LOGGER.warn(format("ParameterizeModel '{}' from '{}:{}' should be an EnrichableModel", parameterizedModel.getName(),
                           elementDeclaration.getDeclaringExtension(), elementDeclaration.getName()));
        return metadataKeysContainerMetadataResult;
      }
      ((EnrichableModel) parameterizedModel).getModelProperty(MetadataKeyIdModelProperty.class)
          .ifPresent(metadataKeyIdModelProperty -> {
            List<ParameterModel> metadataKeyPartParameterModels = parameterizedModel.getAllParameterModels()
                .stream()
                .filter(p -> p.getModelProperty(MetadataKeyPartModelProperty.class).isPresent())
                .sorted(comparingInt(p -> ((EnrichableModel) p).getModelProperty(MetadataKeyPartModelProperty.class).get()
                    .getOrder()).reversed())
                .collect(toList());

            Reference<List<ParameterElementDeclaration>> parameterElementDeclarationsReference = new Reference<>();

            metadataKeyIdModelProperty.getType().accept(new MetadataTypeVisitor() {

              @Override
              public void visitObject(ObjectType objectType) {
                Optional<ParameterGroupElementDeclaration> elementDeclarationParameterGroupOptional =
                    elementDeclaration.getParameterGroup(metadataKeyIdModelProperty.getParameterName());
                if (!elementDeclarationParameterGroupOptional.isPresent()) {
                  LOGGER
                      .warn("Could not find metadataKeyIdModelProperty.name='{}' as parameterGroup on parameterizableModel '{}' for element '{}:{}'",
                            metadataKeyIdModelProperty.getName(),
                            parameterizedModel.getName(),
                            elementDeclaration.getDeclaringExtension(), elementDeclaration.getName());
                }
                elementDeclarationParameterGroupOptional.ifPresent(elementDeclarationParameterGroup -> {
                  List<ParameterElementDeclaration> parameterElementDeclarations =
                      elementDeclarationParameterGroup.getParameters();
                  metadataKeyPartParameterModels.stream().forEach(keyParameterModel -> {
                    parameterElementDeclarations.removeIf(parameterElementDeclaration -> parameterElementDeclaration.getName()
                        .equals(keyParameterModel.getName()));
                    cacheIdGenerator.getIdForMetadataKeys(elementDeclaration)
                        .map(id -> keysStorage.keySet().remove(id))
                        .orElseGet(() -> {
                          LOGGER.warn("Couldn't create a metadata cache id for component: %s",
                                      elementDeclaration.getName());
                          return null;
                        });
                  });
                });
              }

              @Override
              protected void defaultVisit(MetadataType metadataType) {
                LOGGER
                    .warn("Could not invalidate upper level of combinations for partial type key resolver from cache as metadataKeyIdModelProperty.name='{}' is not an ObjectType, instead: '{}' for element '{}:{}'",
                          metadataKeyIdModelProperty.getName(),
                          metadataType.getClass().getName(),
                          elementDeclaration.getDeclaringExtension(), elementDeclaration.getName());
              }
            });
          });
    }
    return metadataKeysContainerMetadataResult;
  }

  private MetadataResult<MetadataKeysContainer> doGetMetadataKeys(ComponentElementDeclaration elementDeclaration,
                                                                  Callable<MetadataResult<MetadataKeysContainer>> resolver,
                                                                  boolean forceResolution) {
    return cacheIdGenerator.getIdForMetadataKeys(elementDeclaration)
        .map(id -> {
          if (forceResolution) {
            keysStorage.keySet().remove(id);
          }
          if (keysStorage.containsKey(id)) {
            try {
              MetadataResult<MetadataKeysContainer> result = keysStorage.get(id);
              if (result != null) {
                return result;
              }
            } catch (Exception e) {
              // In case if the cache cannot deserialize the data when retrieved
            }
          }
          MetadataResult<MetadataKeysContainer> result = executeHandling(resolver, e -> failure(newFailure(e).onKeys()));
          if (result.isSuccess()) {
            keysStorage.put(id, result);
          }
          return result;
        })
        .orElseThrow(() -> new IllegalArgumentException(format("Couldn't create a metadata cache id for component: %s",
                                                               elementDeclaration.getName())));
  }

  private <T extends ParameterizedModel & EnrichableModel> T getModel(ElementDeclaration elementDeclaration)
      throws IllegalArgumentException {
    return extensionModelProvider.get(elementDeclaration.getDeclaringExtension())
        .map(em -> ArtifactHelperUtils.<T>findModel(em, elementDeclaration)
            .orElseThrow(() -> new IllegalArgumentException(format("Could not find ComponentModel for:  %s in extension with name: %s",
                                                                   elementDeclaration.getName(),
                                                                   em.getName()))))
        .orElseThrow(() -> new IllegalArgumentException(format("Could not find ExtensionModel for extension with name: %s. Available: %s",
                                                               elementDeclaration.getDeclaringExtension(),
                                                               extensionModelProvider.getAllNames())));
  }

  private void populateTypesCache(ElementDeclaration elementDeclaration, ComponentMetadataTypesDescriptor types) {
    types.getInputMetadata().forEach(
                                     (parameterName, parameterType) -> cacheIdGenerator
                                         .getIdForComponentInputMetadata(elementDeclaration, parameterName)
                                         .ifPresent(id -> typesStorage.put(id, parameterType)));
    types.getOutputAttributesMetadata().ifPresent(
                                                  outputAttributes -> cacheIdGenerator
                                                      .getIdForComponentAttributesMetadata(elementDeclaration)
                                                      .ifPresent(id -> typesStorage.put(id, outputAttributes)));
    types.getOutputMetadata().ifPresent(
                                        outputMetadata -> cacheIdGenerator.getIdForComponentOutputMetadata(elementDeclaration)
                                            .ifPresent(id -> typesStorage.put(id, outputMetadata)));
  }

  private <T extends ComponentModel> ComponentMetadataTypesDescriptor descriptorFromModel(T model) {
    ComponentMetadataTypesDescriptor.ComponentMetadataTypesDescriptorBuilder builder = ComponentMetadataTypesDescriptor.builder();

    // We could only add dynamic types here, but they are already being filtered by ComponentMetadataTypesDescriptor.

    if (model instanceof HasOutputModel) {
      HasOutputModel outputModel = (HasOutputModel) model;
      builder.withOutputMetadataDescriptor(OutputMetadataDescriptor.builder()
          .withReturnType(TypeMetadataDescriptor.builder().withType(outputModel.getOutput().getType())
              .dynamic(outputModel.getOutput().hasDynamicType()).build())
          .withAttributesType(TypeMetadataDescriptor.builder().withType(outputModel.getOutputAttributes().getType())
              .dynamic(outputModel.getOutputAttributes().hasDynamicType()).build())
          .build());
    }

    InputMetadataDescriptor.InputMetadataDescriptorBuilder inputBuilder = InputMetadataDescriptor.builder();

    model.getAllParameterModels().forEach(
                                          pm -> inputBuilder.withParameter(pm.getName(),
                                                                           ParameterMetadataDescriptor
                                                                               .builder(pm.getName())
                                                                               .withType(pm.getType())
                                                                               .dynamic(pm.hasDynamicType())
                                                                               .build()));

    return builder.withInputMetadataDescriptor(inputBuilder.build()).build();
  }

}
