/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.factory;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newArtifact;
import static org.mule.tooling.client.internal.session.factory.RemoteDeclarationSessionDeployment.failure;
import static org.mule.tooling.client.internal.session.factory.RemoteDeclarationSessionDeployment.success;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.app.declaration.api.ConnectionElementDeclaration;
import org.mule.runtime.app.declaration.api.IdentifiableElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.runtime.app.declaration.api.fluent.ArtifactDeclarer;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.api.request.session.DeclarationSessionCreationRequest;
import org.mule.tooling.api.request.session.model.Exclusion;
import org.mule.tooling.client.api.descriptors.dependency.Dependency;
import org.mule.tooling.client.internal.session.DeclarationSessionConfig;
import org.mule.tooling.client.internal.session.ExtensionModelProvider;

import java.util.Optional;
import java.util.Set;

import javax.validation.ValidationException;

import org.jetbrains.annotations.NotNull;

public class RemoteDeclarationSessionProvider {

  private LazyValue<RemoteDeclarationSessionDeployment> declarationSessionDeploymentLazyValue;

  public RemoteDeclarationSessionProvider(ExtensionModelProvider extensionModelProvider,
                                          RuntimeToolingService runtimeToolingService,
                                          DeclarationSessionConfig sessionConfig) {
    declarationSessionDeploymentLazyValue = new LazyValue<>(() -> {
      ArtifactDeclarer artifactDeclarer = newArtifact();
      sessionConfig.getConfigurationElementDeclarations().stream()
          .forEach(configurationElementDeclaration -> artifactDeclarer.withGlobalElement(configurationElementDeclaration));
      sessionConfig.getGlobalParameterDeclarations().stream()
          .forEach(globalParameterDeclaration -> artifactDeclarer.withGlobalElement(globalParameterDeclaration));

      try {
        validateRequiredParameters(extensionModelProvider, sessionConfig);
        return success(runtimeToolingService
            .createDeclarationSession(new DeclarationSessionCreationRequest(sessionConfig.getDependencies().stream()
                .map(dependency -> toAgentPluginDependency(dependency)).collect(toList()), artifactDeclarer.getDeclaration(),
                                                                            sessionConfig.getSessionProperties())));
      } catch (Exception e) {
        return failure(e);
      }
    });
  }

  public RemoteDeclarationSessionDeployment getRemoteDeclarationSession() {
    return declarationSessionDeploymentLazyValue.get();
  }

  public boolean isDeployed() {
    return declarationSessionDeploymentLazyValue.isComputed() && declarationSessionDeploymentLazyValue.get().isSuccess();
  }

  private void validateRequiredParameters(ExtensionModelProvider extensionModelProvider, DeclarationSessionConfig sessionConfig) {
    sessionConfig.getConfigurationElementDeclarations().forEach(configurationElementDeclaration -> {
      ExtensionModel extensionModel =
          validateIdentifiableElementDeclaration(extensionModelProvider, configurationElementDeclaration);

      ConfigurationModel configurationModel = extensionModel.getConfigurationModel(configurationElementDeclaration.getName())
          .orElseThrow(() -> new ValidationException(format("Could not find component: '%s:%s'",
                                                            configurationElementDeclaration.getDeclaringExtension(),
                                                            configurationElementDeclaration.getName())));

      validateRequiredParameters(configurationElementDeclaration, configurationModel, of("name"),
                                 of(configurationElementDeclaration.getRefName()), empty());

      configurationElementDeclaration.getConnection()
          .ifPresent(connectionElementDeclaration -> validateConnectionElementDeclaration(extensionModelProvider,
                                                                                          connectionElementDeclaration,
                                                                                          configurationElementDeclaration
                                                                                              .getRefName()));
    });

    sessionConfig.getGlobalParameterDeclarations().forEach(globalParameter -> {
      if (globalParameter.getValue() == null) {
        throw new ValidationException(format("GlobalParameter with refName: '%s' for '%s:%s' does not has a value",
                                             globalParameter.getRefName(),
                                             globalParameter.getDeclaringExtension(),
                                             globalParameter.getName()));
      }
    });

  }

  private void validateRequiredParameters(ParameterizedElementDeclaration parameterizedElementDeclaration,
                                          ParameterizedModel parameterizedModel,
                                          Optional<String> nameProviderParameter,
                                          Optional<String> refName,
                                          Optional<String> containerElementName) {
    Set<String> declaredParameterNames = parameterizedElementDeclaration.getParameterGroups().stream()
        .flatMap(parameterGroupElementDeclaration -> parameterGroupElementDeclaration.getParameters().stream())
        .map(parameterElementDeclaration -> parameterElementDeclaration.getName())
        .collect(toSet());

    nameProviderParameter.ifPresent(nameParameter -> declaredParameterNames.add(nameParameter));

    Optional<ParameterModel> missingRequiredParameter = parameterizedModel.getAllParameterModels().stream()
        .filter(parameterModel -> parameterModel.isRequired())
        .filter(parameterModel -> !declaredParameterNames.contains(parameterModel.getName()))
        .findFirst();
    if (missingRequiredParameter.isPresent()) {
      StringBuilder messageBuilder = new StringBuilder();
      messageBuilder.append("Parameter '%s' is required but was not found on element: '%s:%s'");
      refName.ifPresent(ref -> messageBuilder.append(", refName: '").append(ref).append("'"));
      containerElementName
          .ifPresent(containerName -> messageBuilder.append(", containerElement: '").append(containerName).append("'"));
      throw new ValidationException(format(messageBuilder.toString(),
                                           missingRequiredParameter.get().getName(),
                                           parameterizedElementDeclaration.getDeclaringExtension(),
                                           parameterizedElementDeclaration.getName()));
    }
  }

  private void validateConnectionElementDeclaration(ExtensionModelProvider extensionModelProvider,
                                                    ConnectionElementDeclaration connectionElementDeclaration,
                                                    String containerRefName) {
    ExtensionModel extensionModel = validateIdentifiableElementDeclaration(extensionModelProvider, connectionElementDeclaration);

    Optional<ConnectionProviderModel> connectionProviderModel;
    connectionProviderModel =
        extensionModel.getConnectionProviders().stream()
            .filter(cp -> cp.getName().equals(connectionElementDeclaration.getName()))
            .findFirst();

    if (!connectionProviderModel.isPresent()) {
      connectionProviderModel = extensionModel.getConfigurationModels().stream()
          .flatMap(cm -> cm.getConnectionProviders().stream())
          .filter(cp -> cp.getName().equals(connectionElementDeclaration.getName()))
          .findFirst();
    }

    if (!connectionProviderModel.isPresent()) {
      throw new ValidationException(format("Could not find component: '%s:%s'",
                                           connectionElementDeclaration.getDeclaringExtension(),
                                           connectionElementDeclaration.getName()));
    }

    validateRequiredParameters(connectionElementDeclaration, connectionProviderModel.get(), empty(), empty(),
                               of(containerRefName));
  }

  @NotNull
  private ExtensionModel validateIdentifiableElementDeclaration(ExtensionModelProvider extensionModelProvider,
                                                                IdentifiableElementDeclaration elementDeclaration) {
    String declaringExtension = elementDeclaration.getDeclaringExtension();
    return extensionModelProvider.get(declaringExtension)
        .orElseThrow(() -> new ValidationException(format("ElementDeclaration is defined for extension: '%s' which is not part of the context: '%s'",
                                                          declaringExtension,
                                                          extensionModelProvider.getAllNames())));
  }

  private static org.mule.tooling.api.request.session.model.Dependency toAgentPluginDependency(Dependency fromDependency) {
    org.mule.tooling.api.request.session.model.Dependency toDependency =
        new org.mule.tooling.api.request.session.model.Dependency();
    toDependency.setGroupId(fromDependency.getGroupId());
    toDependency.setArtifactId(fromDependency.getArtifactId());
    toDependency.setVersion(fromDependency.getVersion());
    toDependency.setClassifier(fromDependency.getClassifier());
    toDependency.setType(fromDependency.getType());
    toDependency.setScope(fromDependency.getScope());
    toDependency.setSystemPath(fromDependency.getSystemPath());
    toDependency.setOptional(fromDependency.getOptional());
    toDependency.setExclusions(fromDependency.getExclusions().stream().map(fromExclusion -> {
      Exclusion toExclusion = new Exclusion();
      toExclusion.setGroupId(fromExclusion.getGroupId());
      toExclusion.setArtifactId(fromExclusion.getArtifactId());
      return toExclusion;
    }).collect(toList()));
    return toDependency;
  }

}
