/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.filter;

import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toMap;
import static org.mule.tooling.client.internal.session.filter.FilterUtils.getParameterConfiguredValue;

import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.runtime.module.tooling.internal.artifact.params.ParameterSimpleValueExtractor;
import org.mule.tooling.client.internal.session.filter.exception.MissingLevelException;
import org.mule.tooling.client.internal.session.filter.exception.UnknownLevelValueException;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

public class ValueFilter {

  public Set<Value> filter(ValueProviderModel valueProviderModel, ParameterizedModel parameterizedModel,
                           ParameterizedElementDeclaration parameterizedElementDeclaration,
                           Set<Value> values) {
    Map<Integer, String> orderParts = orderParts(valueProviderModel, parameterizedModel.getAllParameterModels());
    Map<Integer, Set<Value>> valuesByPartOrder = new HashMap<>();
    try {
      filterValues(valuesByPartOrder, values, orderParts, parameterizedElementDeclaration, 1);
    } catch (MissingLevelException e) {
      Set<Value> level = valuesByPartOrder.get(valueProviderModel.getPartOrder());
      if (level == null) {
        throw e;
      }
      return level;
    }
    return emptySet();
  }

  private Map<Integer, String> orderParts(ValueProviderModel valueProviderModel, List<ParameterModel> parameters) {
    return parameters.stream()
        .filter(p -> p.getValueProviderModel().map(vpm -> vpm.getProviderName().equals(valueProviderModel.getProviderName()))
            .orElse(false))
        .collect(toMap(param -> param.getValueProviderModel().get().getPartOrder(), NamedObject::getName));
  }

  private void filterValues(Map<Integer, Set<Value>> valuesByPartOrder,
                            Set<Value> values, Map<Integer, String> orderParts,
                            ParameterizedElementDeclaration parameterizedElementDeclaration,
                            int level) {
    if (!values.stream().findFirst().isPresent()) {
      return;
    }
    valuesByPartOrder.put(level, values);
    final String levelName = orderParts.get(level);
    final Optional<String> optionalValuePartConfiguredValue =
        getParameterConfiguredValue(parameterizedElementDeclaration, levelName)
            .map(ParameterSimpleValueExtractor::extractSimpleValue);
    if (!optionalValuePartConfiguredValue.isPresent()) {
      throw new MissingLevelException(levelName);
    } else {
      final String valuePartConfiguredValue = optionalValuePartConfiguredValue.get();

      final Optional<Value> matchingValue =
          values.stream().filter(v -> v.getId().equals(valuePartConfiguredValue)).findFirst();

      if (matchingValue.isPresent()) {
        filterValues(valuesByPartOrder, matchingValue.get().getChilds(), orderParts, parameterizedElementDeclaration, level + 1);
      } else {
        throw new UnknownLevelValueException(levelName, valuePartConfiguredValue);
      }
    }
  }

}
