/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.utils;

import static java.util.Optional.of;
import static java.util.Optional.ofNullable;

import org.mule.runtime.api.util.Reference;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.tooling.client.api.component.location.Location;

import java.util.Optional;

/**
 * Utility class used to locat a {@link ComponentAst} inside an {@link ArtifactAst}
 */
public class ComponentModelLocatorHelper {

  /**
   * Returns the {@link ComponentAst} located at the specified {@link Location}
   *
   * @param applicationModel The {@link ArtifactAst} in which we want to locate a component.
   * @param location         The {@link Location}
   * @return The component located at the specified location if available or empty if none matches.
   */
  public Optional<ComponentAst> findComponentModel(ArtifactAst applicationModel, Location location) {
    final Reference<ComponentAst> foundComponentModelReference = new Reference<>();
    Optional<ComponentAst> globalComponent = applicationModel.topLevelComponentsStream()
        .filter(cm -> location.getGlobalName().equals(cm.getComponentId().orElse(null))).findFirst();
    globalComponent.ifPresent(componentModel -> findComponentWithLocation(componentModel, location)
        .ifPresent(foundComponentModel -> foundComponentModelReference.set(foundComponentModel)));
    return ofNullable(foundComponentModelReference.get());
  }

  private Optional<ComponentAst> findComponentWithLocation(ComponentAst componentModel, Location location) {
    if (componentModel.getLocation().getLocation().equals(location.toString())) {
      return of(componentModel);
    }

    return componentModel.directChildrenStream()
        .map(childComponent -> findComponentWithLocation(childComponent, location))
        .filter(Optional::isPresent)
        .map(Optional::get)
        .findFirst();
  }
}
