/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClient.Builder;
import org.mule.tooling.client.api.cache.CacheStorageFactory;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.datasense.MetadataCacheFactory;
import org.mule.tooling.client.api.datasense.storage.MetadataCacheStorageFactory;
import org.mule.tooling.client.api.feature.Action;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.internal.action.DispacheableAction;
import org.mule.tooling.client.internal.cache.CacheStorageMapWrapperFactory;
import org.mule.tooling.client.internal.dsl.DslSyntaxServiceCache;
import org.mule.tooling.client.internal.metadata.MetadataCacheStorageMapWrapperFactory;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.service.ServiceRegistry;

import java.io.File;
import java.util.Optional;

/**
 * Default implementation of {@link Builder} that creates a {@link ToolingRuntimeClient}.
 *
 * @since 4.0
 */
public class DefaultToolingRuntimeClientBuilder implements Builder, Command {

  private Optional<MuleVersion> targetMuleVersion = empty();
  private String toolingVersion;
  private Serializer serializer;
  private MavenClient mavenClient;
  private ModuleRepository moduleRepository;
  private ArtifactClassLoader containerClassLoaderFactory;
  private Optional<ExtensionModelServiceCache> extensionModelServiceCache;
  private ApplicationCache applicationCache;
  private DomainCache domainCache;
  private Optional<MetadataCacheFactory> metadataCacheFactoryOptional = empty();
  private Optional<CacheStorageMapWrapperFactory> metadataCacheStorageMapWrapperFactory = empty();
  private Optional<CacheStorageMapWrapperFactory> valueProvidersCacheStorageMapWrapperFactory = empty();

  private DslSyntaxServiceCache dslSyntaxServiceCache;
  private ServiceRegistry serviceRegistry;

  private AgentConfiguration agentConfiguration;

  private File workingDirectory;

  public DefaultToolingRuntimeClientBuilder(String toolingVersion,
                                            Serializer serializer,
                                            ModuleRepository moduleRepository,
                                            ArtifactClassLoader containerClassLoaderFactory,
                                            Optional<ExtensionModelServiceCache> extensionModelServiceCache,
                                            ApplicationCache applicationCache,
                                            DomainCache domainCache,
                                            DslSyntaxServiceCache dslSyntaxServiceCache,
                                            ServiceRegistry serviceRegistry,
                                            File workingDirectory) {
    this.toolingVersion = toolingVersion;
    this.serializer = serializer;
    this.moduleRepository = moduleRepository;
    this.containerClassLoaderFactory = containerClassLoaderFactory;
    this.extensionModelServiceCache = extensionModelServiceCache;
    this.applicationCache = applicationCache;
    this.domainCache = domainCache;
    this.dslSyntaxServiceCache = dslSyntaxServiceCache;
    this.serviceRegistry = serviceRegistry;
    this.workingDirectory = workingDirectory;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Builder withRemoteAgentConfiguration(AgentConfiguration agentConfiguration) {
    this.agentConfiguration = agentConfiguration;
    return this;
  }

  @Override
  public Builder withMavenConfiguration(MavenConfiguration mavenConfiguration) {
    this.mavenClient = discoverProvider(DefaultToolingRuntimeClientBuilder.class.getClassLoader())
        .createMavenClient(mavenConfiguration);
    return this;
  }

  @Override
  public Feature<Action<String>> withTargetRuntimeVersion() {
    return Feature.enabled(new DispacheableAction<>(targetRuntimeVersion -> {
      requireNonNull(targetRuntimeVersion, "targetRuntimeVersion cannot be null");
      DefaultToolingRuntimeClientBuilder.this.targetMuleVersion = of(new MuleVersion(targetRuntimeVersion));
    }, String.class, serializer));
  }

  @Override
  public Builder withMetadataCacheFactory(MetadataCacheFactory metadataCacheFactory) {
    this.metadataCacheFactoryOptional = ofNullable(metadataCacheFactory);
    return this;
  }

  @Override
  public Feature<Action<MetadataCacheStorageFactory>> withMetadataCacheStorageFactory() {
    /*
     * This method should not be called on this class. The storage factory should be configured with a Map wrapper by reflection.
     * This method is only intended to be implemented by client-side Builders
     */
    return Feature.enabled((f) -> {
      throw new UnsupportedOperationException();
    });
  }

  @Override
  public Feature<Action<CacheStorageFactory>> withCacheStorageFactoryForMetadata() {
    /*
     * This method should not be called on this class. The storage factory should be configured with a Map wrapper by reflection.
     * This method is only intended to be implemented by client-side Builders
     */
    return Feature.enabled((f) -> {
      throw new UnsupportedOperationException();
    });
  }

  @Override
  public Feature<Action<CacheStorageFactory>> withCacheStorageFactoryForValueProviders() {
    /*
     * This method should not be called on this class. The storage factory should be configured with a Map wrapper by reflection.
     * This method is only intended to be implemented by client-side Builders
     */
    return Feature.enabled((f) -> {
      throw new UnsupportedOperationException();
    });
  }

  public Builder withMetadataCacheStorageMapWrapperFactory(MetadataCacheStorageMapWrapperFactory metadataCacheStorageMapWrapperFactory) {
    /*
     * This method is only part of this specific implementation because it's supposed to be called internally by reflection on an
     * instance of this class
     */
    this.metadataCacheStorageMapWrapperFactory = ofNullable(metadataCacheStorageMapWrapperFactory);
    return this;
  }

  public Builder withCacheStorageMapWrapperFactoryForMetadata(CacheStorageMapWrapperFactory metadataCacheStorageMapWrapperFactory) {
    /*
     * This method is only part of this specific implementation because it's supposed to be called internally by reflection on an
     * instance of this class
     */
    this.metadataCacheStorageMapWrapperFactory = ofNullable(metadataCacheStorageMapWrapperFactory);
    return this;
  }

  public Builder withCacheStorageMapWrapperFactoryForValueProviders(CacheStorageMapWrapperFactory valueProvidersCacheStorageMapWrapperFactory) {
    /*
     * This method is only part of this specific implementation because it's supposed to be called internally by reflection on an
     * instance of this class
     */
    this.valueProvidersCacheStorageMapWrapperFactory = ofNullable(valueProvidersCacheStorageMapWrapperFactory);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingRuntimeClient build() {
    MuleArtifactResourcesRegistry muleArtifactResourcesRegistry =
        new MuleArtifactResourcesRegistry(toolingVersion, targetMuleVersion, mavenClient, moduleRepository,
                                          containerClassLoaderFactory, workingDirectory);
    InternalExtensionModelService extensionModelService =
        new DefaultExtensionModelService(muleArtifactResourcesRegistry);
    MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider =
        new DefaultMuleRuntimeExtensionModelProvider(mavenClient, extensionModelService);
    if (extensionModelServiceCache.isPresent()) {
      muleRuntimeExtensionModelProvider = new CachedExtensionModelService(mavenClient, extensionModelServiceCache.get(),
                                                                          extensionModelService);
    }

    if (muleArtifactResourcesRegistry.getTargetMuleVersion().isPresent()) {
      muleRuntimeExtensionModelProvider =
          new ExtensionModelTargetVersionMediator(muleArtifactResourcesRegistry.getTargetMuleVersion().get(),
                                                  muleRuntimeExtensionModelProvider);
    }

    return new DefaultToolingRuntimeClient(mavenClient,
                                           serializer,
                                           ofNullable(agentConfiguration),
                                           muleRuntimeExtensionModelProvider,
                                           muleArtifactResourcesRegistry,
                                           applicationCache,
                                           domainCache,
                                           metadataCacheFactoryOptional,
                                           metadataCacheStorageMapWrapperFactory,
                                           valueProvidersCacheStorageMapWrapperFactory,
                                           dslSyntaxServiceCache,
                                           serviceRegistry);
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "withRemoteAgentConfiguration": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(AgentConfiguration.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        this.withRemoteAgentConfiguration(serializer.deserialize(arguments[0]));
        return this;
      }
      case "withMavenConfiguration": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(MavenConfiguration.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        this.withMavenConfiguration(serializer.deserialize(arguments[0]));
        return this;
      }
      case "withTargetRuntimeVersion": {
        checkState(arguments.length == 0,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        return this.withTargetRuntimeVersion();
      }
      case "build": {
        return this.build();
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }
}
