/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.io.Files.createTempDir;
import static java.lang.Boolean.valueOf;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.mule.runtime.container.api.ContainerClassLoaderProvider.createContainerClassLoader;
import static org.mule.runtime.container.api.ModuleRepository.createModuleRepository;
import static org.mule.runtime.core.api.util.FileUtils.unzip;
import static org.mule.runtime.globalconfig.api.maven.MavenClientFactory.setMavenClientProvider;
import static org.mule.tooling.client.internal.util.Preconditions.checkState;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.AetherMavenClientProvider;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.deployment.model.api.artifact.DescriptorLoaderRepositoryFactory;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorValidatorBuilder;
import org.mule.runtime.module.artifact.api.descriptor.DescriptorLoaderRepository;
import org.mule.runtime.module.service.api.discoverer.ServiceResolutionError;
import org.mule.runtime.module.service.internal.artifact.ServiceClassLoaderFactory;
import org.mule.runtime.module.service.internal.discoverer.DefaultServiceDiscoverer;
import org.mule.runtime.module.service.internal.discoverer.FileSystemServiceProviderDiscoverer;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClientBuilderFactory;
import org.mule.tooling.client.internal.dsl.DslSyntaxServiceCache;
import org.mule.tooling.client.internal.log4j.ToolingLog4jContextFactory;
import org.mule.tooling.client.internal.serialization.KryoServerSerializer;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.serialization.XStreamServerSerializer;
import org.mule.tooling.client.internal.service.DefaultServiceRegistry;
import org.mule.tooling.client.internal.service.ServiceRegistry;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URLClassLoader;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableList;

import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;

/**
 * Default implementation of {@link ToolingRuntimeClientBuilderFactory} for creating {@link ToolingRuntimeClient}s.
 *
 * @since 4.0
 */
public class DefaultToolingRuntimeClientFactory implements ToolingRuntimeClientBuilderFactory, Command, Disposable {

  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISABLE =
      "tooling.client.ExtensionModelServiceCache.disable";

  private static final String SERVICES_FOLDER = "services";

  private final String toolingVersion;
  private ArtifactClassLoader containerClassLoaderFactory;
  private ModuleRepository moduleRepository;
  private Optional<ExtensionModelServiceCache> extensionModelServiceCache = empty();
  private ApplicationCache applicationCache;
  private DomainCache domainCache;
  private DslSyntaxServiceCache dslSyntaxServiceCache;
  // Previous versions of Tooling API when bootstrapping Tooling Client will not call #setSerialization(name), therefore has to be
  // initialized here
  private LazyValue<Serializer> serializer = new LazyValue<>(() -> new XStreamServerSerializer(getToolingAPIClassLoader()));

  private File workingDirectory;
  private boolean deleteWorkingDirectoryOnDispose = false;
  private ServiceRegistry serviceRegistry;

  public DefaultToolingRuntimeClientFactory(String toolingVersion) {
    this(toolingVersion, createTemporaryWorkingDirectory());
  }

  private static File createTemporaryWorkingDirectory() {
    File workingDirectory = createTempDir();
    workingDirectory.deleteOnExit();
    return workingDirectory;
  }

  public DefaultToolingRuntimeClientFactory(String toolingVersion, File workingDirectory) {
    preInitialize();
    this.toolingVersion = toolingVersion;
    if (workingDirectory != null) {
      this.workingDirectory = workingDirectory;
    } else {
      this.workingDirectory = createTemporaryWorkingDirectory();
      this.deleteWorkingDirectoryOnDispose = true;
    }

    if (!valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISABLE, "false"))) {
      extensionModelServiceCache = of(new ExtensionModelServiceCache(toolingVersion));
    }
    initialise();
  }

  private void preInitialize() {
    // XmlBeans library for XmlTypeLoader has class loading issues and static fields are not initialized,
    // therefore we should enforce this call before the rest of the classes are loaded
    try {
      XmlObject.Factory.parse("<example>data</example>");
    } catch (XmlException e) {
      throw new IllegalStateException("Could not pre initialize XmlBeans library", e);
    }
  }

  private void initialise() {
    moduleRepository = createModuleRepository(DefaultToolingRuntimeClientFactory.class.getClassLoader(), workingDirectory);
    containerClassLoaderFactory =
        createContainerClassLoader(moduleRepository, DefaultToolingRuntimeClientFactory.class.getClassLoader());
    applicationCache = new ApplicationCache();
    domainCache = new DomainCache();
    dslSyntaxServiceCache = new DslSyntaxServiceCache();

    discoverServices();
  }

  private void discoverServices() {
    File servicesFolder = new File(workingDirectory, SERVICES_FOLDER);

    List<File> serviceJarFiles = Arrays.stream(((URLClassLoader) this.getClass().getClassLoader()).getURLs())
        .map(url -> FileUtils.toFile(url))
        .filter(file -> file.getName().endsWith("mule-service.jar"))
        .collect(Collectors.toList());

    serviceJarFiles.stream().forEach(serviceJarFile -> {
      try {
        unzip(serviceJarFile, new File(servicesFolder, serviceJarFile.getName()));
      } catch (IOException e) {
        throw new UncheckedIOException(e);
      }
    });

    setMavenClientProvider(() -> new AetherMavenClientProvider() {

      @Override
      public MavenClient createMavenClient(MavenConfiguration mavenConfiguration) {
        return null;
      }
    });
    DescriptorLoaderRepository descriptorLoaderRepository =
        new DescriptorLoaderRepositoryFactory().createDescriptorLoaderRepository();

    final FileSystemServiceProviderDiscoverer fileSystemServiceProviderDiscoverer =
        new FileSystemServiceProviderDiscoverer(containerClassLoaderFactory, new ServiceClassLoaderFactory(),
                                                descriptorLoaderRepository, ArtifactDescriptorValidatorBuilder.builder(),
                                                servicesFolder);

    try {
      this.serviceRegistry =
          new DefaultServiceRegistry(new DefaultServiceDiscoverer(fileSystemServiceProviderDiscoverer)
              .discoverServices());
    } catch (ServiceResolutionError serviceResolutionError) {
      throw new IllegalStateException(serviceResolutionError);
    }
  }

  /**
   * Changes the serialization implementation, as the API and implementations cannot be changed I had to add this method that will
   * only be called using reflection and whenever the API supports more than one serialization, if not it will continue using
   * xStream as the default serializer.
   *
   * @param name the serialization name to be enabled.
   */
  public void setSerialization(String name) {
    checkState(getSupportedSerialization().contains(name),
               String.format("Not supported serialization '%s', the ones supported by this implementation are: %s", name,
                             getSupportedSerialization()));
    if (XStreamServerSerializer.NAME.equals(name)) {
      this.serializer = new LazyValue<>(() -> new XStreamServerSerializer(getToolingAPIClassLoader()));
    } else if (KryoServerSerializer.NAME.equals(name)) {
      this.serializer =
          new LazyValue<>(() -> new KryoServerSerializer(this.getClass().getClassLoader(), getToolingAPIClassLoader()));
    }
  }

  private ClassLoader getToolingAPIClassLoader() {
    // Parent is FilteringClassLoader, so next would be the one that loaded Tooling API (client's class loader)
    return this.getClass().getClassLoader().getParent().getParent();
  }

  /**
   * @return {@link List} of serialization names supported by this implementation.
   */
  private List<String> getSupportedSerialization() {
    return ImmutableList.<String>builder().add(XStreamServerSerializer.NAME, KryoServerSerializer.NAME).build();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingRuntimeClient.Builder create() {
    return new DefaultToolingRuntimeClientBuilder(toolingVersion, serializer.get(), moduleRepository,
                                                  containerClassLoaderFactory,
                                                  extensionModelServiceCache,
                                                  applicationCache,
                                                  domainCache,
                                                  dslSyntaxServiceCache,
                                                  serviceRegistry, workingDirectory);
  }

  @Override
  public void dispose() {
    extensionModelServiceCache.ifPresent(service -> dispose(service));
    dispose(domainCache);
    dispose(applicationCache);
    dispose(dslSyntaxServiceCache);

    if (LogManager.getFactory() instanceof ToolingLog4jContextFactory) {
      ((ToolingLog4jContextFactory) LogManager.getFactory()).dispose();
    }

    if (serviceRegistry != null) {
      serviceRegistry.dispose();
    }
    if (deleteWorkingDirectoryOnDispose) {
      deleteQuietly(workingDirectory);
    }
  }

  private void dispose(Disposable disposable) {
    try {
      disposable.dispose();
    } catch (Exception e) {
      // Nothing to do...
    }
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "dispose": {
        this.dispose();
        return null;
      }
      case "create": {
        return create();
      }
    }
    throw new IllegalArgumentException(format("Invalid arguments passed for calling method '%s' on '%s'", methodName,
                                              this.getClass()));
  }


}
