/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.Boolean.parseBoolean;
import static java.lang.String.join;
import static java.lang.System.getProperty;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Comparator.comparing;
import static java.util.Comparator.comparingInt;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toCollection;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.mule.runtime.api.meta.ExternalLibraryType.DEPENDENCY;
import static org.mule.runtime.api.meta.ExternalLibraryType.JAR;
import static org.mule.runtime.api.meta.ExternalLibraryType.NATIVE;
import static org.mule.tooling.client.api.extension.model.ExternalLibraryType.dependencyExternalLibraryType;
import static org.mule.tooling.client.api.extension.model.ExternalLibraryType.jarExternalLibraryType;
import static org.mule.tooling.client.api.extension.model.ExternalLibraryType.nativeExternalLibraryType;
import static org.mule.tooling.client.api.extension.model.value.ValuesResolverModel.metadataKeyResolverName;
import static org.mule.tooling.client.api.extension.model.value.ValuesResolverModel.valueProviderResolverName;
import static org.mule.tooling.client.internal.CollectionUtils.copy;

import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.Typed;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.EnrichableModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.extension.api.property.DefaultImplementingTypeModelProperty;
import org.mule.runtime.extension.api.property.InfrastructureParameterModelProperty;
import org.mule.runtime.extension.api.property.MetadataKeyIdModelProperty;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;
import org.mule.runtime.extension.api.property.QNameModelProperty;
import org.mule.runtime.extension.api.property.ResolverInformation;
import org.mule.runtime.extension.api.property.TypeResolversInformationModelProperty;
import org.mule.tooling.client.api.extension.model.ClassValueModel;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExpressionSupport;
import org.mule.tooling.client.api.extension.model.ExternalLibraryModel;
import org.mule.tooling.client.api.extension.model.ExternalLibraryType;
import org.mule.tooling.client.api.extension.model.LayoutModel;
import org.mule.tooling.client.api.extension.model.OutputModel;
import org.mule.tooling.client.api.extension.model.ParameterDslConfiguration;
import org.mule.tooling.client.api.extension.model.PathModel;
import org.mule.tooling.client.api.extension.model.PathModelLocation;
import org.mule.tooling.client.api.extension.model.PathModelType;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionManagementType;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.data.sample.SampleDataProviderModel;
import org.mule.tooling.client.api.extension.model.deprecated.DeprecationModel;
import org.mule.tooling.client.api.extension.model.function.FunctionModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyIdModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyPartModel;
import org.mule.tooling.client.api.extension.model.metadata.TypeResolversInformationModel;
import org.mule.tooling.client.api.extension.model.nested.NestableElementModel;
import org.mule.tooling.client.api.extension.model.nested.NestedChainModel;
import org.mule.tooling.client.api.extension.model.nested.NestedComponentModel;
import org.mule.tooling.client.api.extension.model.nested.NestedRouteModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.parameter.ActingParameterModel;
import org.mule.tooling.client.api.extension.model.parameter.ExclusiveParametersModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterRole;
import org.mule.tooling.client.api.extension.model.property.DefaultImplementingTypeModel;
import org.mule.tooling.client.api.extension.model.property.InfrastructureParameterModel;
import org.mule.tooling.client.api.extension.model.property.QNameModel;
import org.mule.tooling.client.api.extension.model.source.SourceCallbackModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.api.extension.model.value.FieldValueProviderModel;
import org.mule.tooling.client.api.extension.model.value.FieldValuesResolverModel;
import org.mule.tooling.client.api.extension.model.value.ValueProviderModel;
import org.mule.tooling.client.api.extension.model.value.ValuesResolverModel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

/**
 * Utils for creating the Extension Model DTOs from the Mule-API objects
 *
 * @since 4.0
 */
public class ExtensionModelPartsFactory {

  private static final String DOT = ".";
  private static final String CONFIG_REF_PARAMETER = "config-ref";
  private static final String NAME_PARAMETER = "name";
  private static final MuleVersion MULE_VERSION_4_2_0 = new MuleVersion("4.2.0");

  public static final String FILTER_PARAMETERS_RESERVED_NAMES = "tooling.client.configuration.filter.parameters.reserved.names";
  private static final String VALUE_PROVIDER = "ValueProvider";

  private final Optional<MuleVersion> targetRuntimeVersion;

  public ExtensionModelPartsFactory() {
    this(empty());
  }

  public ExtensionModelPartsFactory(Optional<MuleVersion> targetRuntimeVersion) {
    this.targetRuntimeVersion = targetRuntimeVersion;
  }

  public Set<ExternalLibraryModel> toExternalLibraryModelsDTO(
                                                              Set<org.mule.runtime.api.meta.model.ExternalLibraryModel> externalLibraryModels) {
    return externalLibraryModels.stream().map(externalLibraryModel -> new ExternalLibraryModel(externalLibraryModel.getName(),
                                                                                               externalLibraryModel
                                                                                                   .getDescription(),
                                                                                               externalLibraryModel
                                                                                                   .getRequiredClassName()
                                                                                                   .orElse(null),
                                                                                               externalLibraryModel
                                                                                                   .getRegexMatcher()
                                                                                                   .orElse(null),
                                                                                               toExternalLibraryTypeDTO(externalLibraryModel
                                                                                                   .getType()),
                                                                                               externalLibraryModel
                                                                                                   .getSuggestedCoordinates()
                                                                                                   .orElse(null),
                                                                                               externalLibraryModel.isOptional(),
                                                                                               targetRuntimeVersion
                                                                                                   .map(muleVersion -> muleVersion
                                                                                                       .atLeast(MULE_VERSION_4_2_0))
                                                                                                   .orElse(true)))
        .collect(toCollection(LinkedHashSet::new));
  }

  private static ExternalLibraryType toExternalLibraryTypeDTO(org.mule.runtime.api.meta.ExternalLibraryType type) {
    if (type == NATIVE) {
      return nativeExternalLibraryType(type.name());
    }
    if (type == JAR) {
      return jarExternalLibraryType(type.name());
    }
    if (type == DEPENDENCY) {
      return dependencyExternalLibraryType(type.name());
    }
    return new ExternalLibraryType(type.name());
  }


  public ConnectionProviderModel toConnectionProviderDTO(org.mule.runtime.api.meta.model.connection.ConnectionProviderModel connectionProviderModel) {
    return new ConnectionProviderModel(connectionProviderModel.getName(),
                                       connectionProviderModel.getDescription(),
                                       toDisplayModelDTO(connectionProviderModel.getDisplayModel()),
                                       toConnectionManagementTypeDTO(connectionProviderModel.getConnectionManagementType()),
                                       connectionProviderModel.supportsConnectivityTesting(),
                                       toParameterGroupModelsDTO(createValueProviderEnricher(connectionProviderModel),
                                                                 connectionProviderModel.getParameterGroupModels()),
                                       toExternalLibraryModelsDTO(connectionProviderModel.getExternalLibraryModels()),
                                       toStereotypeDTO(connectionProviderModel.getStereotype()),
                                       toDeprecationModelDTO(connectionProviderModel.getDeprecationModel().orElse(null)),
                                       connectionProviderModel.getSemanticTerms());
  }

  private static ConnectionManagementType toConnectionManagementTypeDTO(
                                                                        org.mule.runtime.api.meta.model.connection.ConnectionManagementType connectionManagementType) {
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.POOLING) {
      return ConnectionManagementType.poolingConnectionManagementType(connectionManagementType.name());
    }
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.CACHED) {
      return ConnectionManagementType.cachedConnectionManagementType(connectionManagementType.name());
    }
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.NONE) {
      return ConnectionManagementType.noneConnectionManagementType(connectionManagementType.name());
    }

    return new ConnectionManagementType(connectionManagementType.name());
  }


  public SourceModel toSourceModelDTO(org.mule.runtime.api.meta.model.source.SourceModel sourceModel) {
    if (sourceModel == null) {
      return null;
    }
    return new SourceModel(sourceModel.getName(),
                           sourceModel.getDescription(),
                           toParameterGroupModelsDTO(valueProviderEnricher(sourceModel),
                                                     sourceModel.getParameterGroupModels()),
                           toDisplayModelDTO(sourceModel.getDisplayModel()),
                           toOutputModelDTO(sourceModel.getOutput()),
                           toOutputModelDTO(sourceModel.getOutputAttributes()),
                           sourceModel.isTransactional(),
                           sourceModel.requiresConnection(),
                           sourceModel.supportsStreaming(),
                           toStereotypeDTO(sourceModel.getStereotype()),
                           sourceModel.hasResponse(),
                           sourceModel.runsOnPrimaryNodeOnly(),
                           toSourceCallbackModelDTO(sourceModel.getSuccessCallback()),
                           toSourceCallbackModelDTO(sourceModel.getErrorCallback()),
                           toSourceCallbackModelDTO(sourceModel.getTerminateCallback()),
                           toErrorModelsDTO(sourceModel.getErrorModels()),
                           toNestedComponentModelsDTO(sourceModel.getNestedComponents()),
                           toMetadataKeyIdModel(sourceModel.getModelProperty(MetadataKeyIdModelProperty.class)),
                           toDeprecationModelDTO(sourceModel.getDeprecationModel().orElse(null)),
                           toTypeResolversInformationModel(sourceModel.getAllParameterModels(), sourceModel
                               .getModelProperty(TypeResolversInformationModelProperty.class).orElse(null)),
                           toSampleDataProviderModel(sourceModel.getSampleDataProviderModel().orElse(null)),
                           sourceModel.getSemanticTerms());
  }

  private SampleDataProviderModel toSampleDataProviderModel(org.mule.runtime.api.meta.model.data.sample.SampleDataProviderModel sampleDataProviderModel) {
    if (sampleDataProviderModel == null) {
      return null;
    }
    return new SampleDataProviderModel(toActingParameterModelsDTO(sampleDataProviderModel.getParameters()),
                                       sampleDataProviderModel.getProviderId(),
                                       sampleDataProviderModel.requiresConfiguration(),
                                       sampleDataProviderModel.requiresConnection());
  }

  static List<ActingParameterModel> toActingParameterModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ActingParameterModel> actingParameterModels) {
    return actingParameterModels.stream()
        .map(param -> new ActingParameterModel(param.getName(),
                                               param.isRequired(),
                                               param.getExtractionExpression()))
        .collect(toList());
  }

  private TypeResolversInformationModel toTypeResolversInformationModel(List<org.mule.runtime.api.meta.model.parameter.ParameterModel> parameterModels,
                                                                        TypeResolversInformationModelProperty typeResolversInformationModelProperty) {
    if (typeResolversInformationModelProperty == null) {
      return null;
    }
    Map<String, org.mule.tooling.client.api.extension.model.metadata.ResolverInformation> inputResolvers =
        parameterModels.stream()
            .filter(Typed::hasDynamicType)
            .filter(pm -> typeResolversInformationModelProperty.getParameterResolver(pm.getName()).isPresent())
            .collect(toMap(NamedObject::getName, pm -> toResolverInformation(typeResolversInformationModelProperty
                .getParameterResolver(pm.getName()).get())));

    return new TypeResolversInformationModel(typeResolversInformationModelProperty.getCategoryName(),
                                             inputResolvers,
                                             typeResolversInformationModelProperty.getOutputResolver()
                                                 .map(this::toResolverInformation).orElse(null),
                                             typeResolversInformationModelProperty.getAttributesResolver()
                                                 .map(this::toResolverInformation).orElse(null),
                                             typeResolversInformationModelProperty.getKeysResolver()
                                                 .map(this::toResolverInformation).orElse(null),
                                             typeResolversInformationModelProperty.isPartialTypeKeyResolver());
  }

  private org.mule.tooling.client.api.extension.model.metadata.ResolverInformation toResolverInformation(ResolverInformation resolverInformation) {
    return new org.mule.tooling.client.api.extension.model.metadata.ResolverInformation(resolverInformation
        .getResolverName(), resolverInformation.isRequiresConnection(), resolverInformation.isRequiresConnection());
  }

  static List<FieldValueProviderModel> toFieldValueProviderModelsDTO(org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel) {
    return parameterModel.getFieldValueProviderModels()
        .stream()
        .map(fvpm -> new FieldValueProviderModel(
                                                 fvpm.getActingParameters(),
                                                 toActingParameterModelsDTO(fvpm.getParameters()),
                                                 fvpm.requiresConfiguration(),
                                                 fvpm.requiresConnection(),
                                                 fvpm.isOpen(),
                                                 fvpm.getPartOrder(),
                                                 fvpm.getProviderName(),
                                                 fvpm.getProviderId(),
                                                 fvpm.getTargetSelector()))
        .collect(toList());
  }

  static List<FieldValuesResolverModel> toFieldValuesResolverModelDTO(org.mule.runtime.api.meta.model.parameter.ParameterModel parameterModel) {
    return parameterModel.getFieldValueProviderModels()
        .stream()
        .map(fvpm -> {
          List<ActingParameterModel> actingParameters = toActingParameterModelsDTO(fvpm.getParameters());
          if (fvpm.getPartOrder() > 1) {
            parameterModel.getFieldValueProviderModels()
                .stream()
                .filter(f -> f.getProviderId().equals(fvpm.getProviderId()))
                .filter(f -> f.getPartOrder() < fvpm.getPartOrder())
                .sorted(
                        Comparator.<org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel>comparingInt(p -> p
                            .getPartOrder()).reversed())
                // The extraction expression should be formed by the name of this parameter, followed by the target path for this
                // field
                // as for the name of the acting parameter, is not really important because it's not actually mapped
                // to a real acting parameter in the ValueProvider
                .forEach(f -> actingParameters.add(new ActingParameterModel(f.getTargetSelector(),
                                                                            true,
                                                                            join(DOT, parameterModel.getName(),
                                                                                 f.getTargetSelector()))));
          }
          return new FieldValuesResolverModel(fvpm.getTargetSelector(),
                                              actingParameters,
                                              fvpm.requiresConfiguration(),
                                              fvpm.requiresConnection(),
                                              fvpm.isOpen(),
                                              valueProviderResolverName(fvpm.getProviderId()));
        })
        .collect(toList());
  }

  private Function<org.mule.runtime.api.meta.model.parameter.ParameterModel, Optional<ValuesResolverModel>> valueProviderEnricher(ComponentModel componentModel) {
    Optional<TypeResolversInformationModelProperty> typeResolversInformationModelPropertyOptional =
        componentModel.getModelProperty(TypeResolversInformationModelProperty.class);

    final Function<org.mule.runtime.api.meta.model.parameter.ParameterModel, Optional<ValuesResolverModel>> metadataKeyEnricher =
        createMetadataKeyEnricher(componentModel, typeResolversInformationModelPropertyOptional);

    final Function<org.mule.runtime.api.meta.model.parameter.ParameterModel, Optional<ValuesResolverModel>> valueProviderEnricher =
        createValueProviderEnricher(componentModel);

    return parameterModel -> ofNullable(metadataKeyEnricher.apply(parameterModel)
        .orElseGet(() -> valueProviderEnricher.apply(parameterModel).orElse(null)));
  }

  static Function<org.mule.runtime.api.meta.model.parameter.ParameterModel, Optional<ValuesResolverModel>> createValueProviderEnricher(ParameterizedModel parameterizedModel) {
    return parameterModel -> parameterModel.getValueProviderModel().map(valueProviderModel -> {
      List<ActingParameterModel> parameters = new ArrayList<>(toActingParameterModelsDTO(valueProviderModel.getParameters()));
      if (valueProviderModel.getPartOrder() > 1) {
        parameterizedModel
            .getAllParameterModels()
            .stream()
            .filter(p -> !p.getName().equals(parameterModel.getName()))
            .filter(p -> p.getValueProviderModel()
                .map(vpm -> vpm.getProviderName().equals(valueProviderModel.getProviderName()))
                .orElse(false))
            .filter(p -> p.getValueProviderModel().get().getPartOrder() < valueProviderModel.getPartOrder())
            .sorted(
                    Comparator.<org.mule.runtime.api.meta.model.parameter.ParameterModel>comparingInt(p -> p
                        .getValueProviderModel().get().getPartOrder()).reversed())
            // Acting parameters for MultiPart Values should always be required. We need to remember that in the context of a
            // ValueResolver,
            // we are always attempting to resolve only one value (There is no more Tree). It can be either a parameter, or a part
            // of a ParameterGroup like this case.
            // Either way, even if the valueProviderResolver can work without any acting parameters set (It should return the
            // whole tree), we need
            // the acting parameters values in order to filter out the actual parameter that we want resolved.
            .forEach(p -> parameters.add(new ActingParameterModel(p.getName(), true)));
      }
      return new ValuesResolverModel(parameters,
                                     valueProviderModel.requiresConfiguration(),
                                     valueProviderModel.requiresConnection(),
                                     valueProviderModel.isOpen(),
                                     valueProviderResolverName(valueProviderModel.getProviderId()));
    });
  }

  private Function<org.mule.runtime.api.meta.model.parameter.ParameterModel, Optional<ValuesResolverModel>> createMetadataKeyEnricher(ComponentModel componentModel,
                                                                                                                                      Optional<TypeResolversInformationModelProperty> typeResolversInformationModelPropertyOptional) {
    if (typeResolversInformationModelPropertyOptional.isPresent()) {
      TypeResolversInformationModelProperty typeResolversInformationModelProperty =
          typeResolversInformationModelPropertyOptional.get();
      Optional<MetadataKeyIdModelProperty> metadataKeyIdModelPropertyOptional =
          componentModel.getModelProperty(MetadataKeyIdModelProperty.class);

      if (metadataKeyIdModelPropertyOptional.isPresent()) {
        Optional<ResolverInformation> keyResolverInformationOptional = typeResolversInformationModelProperty.getKeysResolver();

        if (keyResolverInformationOptional.isPresent()) {
          return parameterModel -> {
            ResolverInformation resolverInformation = keyResolverInformationOptional.get();
            Optional<MetadataKeyPartModelProperty> metadataKeyPartModelPropertyOptional =
                parameterModel.getModelProperty(MetadataKeyPartModelProperty.class);
            if (metadataKeyPartModelPropertyOptional.isPresent()) {
              MetadataKeyPartModelProperty metadataKeyPartModelProperty = metadataKeyPartModelPropertyOptional.get();
              if (metadataKeyPartModelProperty.isProvidedByKeyResolver()) {
                List<ActingParameterModel> parameters = new ArrayList<>();
                if (metadataKeyPartModelProperty.getOrder() > 1) {
                  componentModel.getAllParameterModels()
                      .stream()
                      .filter(p -> !p.getName().equals(parameterModel.getName()))
                      .filter(p -> p.getModelProperty(MetadataKeyPartModelProperty.class).isPresent())
                      .filter(p -> p.getModelProperty(MetadataKeyPartModelProperty.class).get()
                          .getOrder() < metadataKeyPartModelProperty.getOrder())
                      .sorted(
                              Comparator.<EnrichableModel>comparingInt(p2 -> p2
                                  .getModelProperty(MetadataKeyPartModelProperty.class).get().getOrder()).reversed())
                      // Acting parameters for MetadataKeys should always be required. We need to remember that in the context of
                      // a ValueResolver,
                      // we are always attempting to resolve only one value (There is no more Tree). It can be either a parameter,
                      // or a part of a ParameterGroup like this case.
                      // Either way, even if the metadataKeyResolver can work without any acting parameters set (It should return
                      // the whole tree), we need
                      // the acting parameters values in order to filter out the actual parameter that we want resolved.
                      .forEach(actingParameter -> parameters.add(new ActingParameterModel(actingParameter.getName(), true)));
                }
                return of(new ValuesResolverModel(parameters,
                                                  resolverInformation.isRequiresConfiguration(),
                                                  resolverInformation.isRequiresConnection(),
                                                  // By definition MetadataKeys are close.
                                                  false,
                                                  metadataKeyResolverName(typeResolversInformationModelProperty.getCategoryName(),
                                                                          keyResolverInformationOptional.get()
                                                                              .getResolverName())));
              }
            }
            return empty();
          };
        }
      }
    }
    return parameterModel -> empty();
  }

  private List<? extends NestableElementModel> toNestedComponentModelsDTO(
                                                                          List<? extends org.mule.runtime.api.meta.model.nested.NestableElementModel> components) {
    return components.stream().map(this::toNestedComponentModelDTO).collect(toList());
  }

  private NestableElementModel toNestedComponentModelDTO(org.mule.runtime.api.meta.model.nested.NestableElementModel nestedModel) {
    final NestableElementModelDTOVisitor nestableElementModelVisitor = new NestableElementModelDTOVisitor();
    nestedModel.accept(nestableElementModelVisitor);
    return nestableElementModelVisitor.getNestableElementModelDTO();
  }

  private class NestableElementModelDTOVisitor
      implements org.mule.runtime.api.meta.model.nested.NestableElementModelVisitor {

    private NestableElementModel nestableElementModelDTO = null;

    public NestableElementModel getNestableElementModelDTO() {
      return nestableElementModelDTO;
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.nested.NestedComponentModel nestedComponentModel) {
      nestableElementModelDTO = new NestedComponentModel(nestedComponentModel.getName(),
                                                         nestedComponentModel.getDescription(),
                                                         toDisplayModelDTO(nestedComponentModel.getDisplayModel()),
                                                         nestedComponentModel.isRequired(),
                                                         toStereotypesDTO(nestedComponentModel.getAllowedStereotypes()),
                                                         toParameterGroupModelsDTO(p -> empty(),
                                                                                   nestedComponentModel
                                                                                       .getParameterGroupModels()),
                                                         toDeprecationModelDTO(nestedComponentModel.getDeprecationModel()
                                                             .orElse(null)),
                                                         toErrorModelsDTO(nestedComponentModel.getErrorModels()),
                                                         nestedComponentModel.getMinOccurs(),
                                                         nestedComponentModel.getMaxOccurs().orElse(null),
                                                         toNestedComponentModelsDTO(nestedComponentModel.getNestedComponents()),
                                                         toStereotypeDTO(nestedComponentModel.getStereotype()),
                                                         nestedComponentModel.getSemanticTerms());
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.nested.NestedChainModel nestedChainModel) {
      nestableElementModelDTO = new NestedChainModel(nestedChainModel.getName(),
                                                     nestedChainModel.getDescription(),
                                                     toDisplayModelDTO(nestedChainModel.getDisplayModel()),
                                                     nestedChainModel.isRequired(),
                                                     toStereotypesDTO(nestedChainModel.getAllowedStereotypes()),
                                                     toParameterGroupModelsDTO(p -> empty(),
                                                                               nestedChainModel.getParameterGroupModels()),
                                                     toDeprecationModelDTO(nestedChainModel.getDeprecationModel().orElse(null)),
                                                     toErrorModelsDTO(nestedChainModel.getErrorModels()),
                                                     nestedChainModel.getMinOccurs(),
                                                     nestedChainModel.getMaxOccurs().orElse(null),
                                                     toNestedComponentModelsDTO(nestedChainModel.getNestedComponents()),
                                                     toStereotypeDTO(nestedChainModel.getStereotype()),
                                                     nestedChainModel.getSemanticTerms());
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.nested.NestedRouteModel nestedRouteModel) {
      nestableElementModelDTO = new NestedRouteModel(nestedRouteModel.getName(),
                                                     nestedRouteModel.getDescription(),
                                                     toDisplayModelDTO(nestedRouteModel.getDisplayModel()),
                                                     nestedRouteModel.getMinOccurs(),
                                                     nestedRouteModel.getMaxOccurs()
                                                         .orElse(null),
                                                     toParameterGroupModelsDTO(p -> empty(),
                                                                               nestedRouteModel.getParameterGroupModels()),
                                                     toNestedComponentModelsDTO(nestedRouteModel.getNestedComponents()),
                                                     toDeprecationModelDTO(nestedRouteModel.getDeprecationModel().orElse(null)),
                                                     toErrorModelsDTO(nestedRouteModel.getErrorModels()),
                                                     toStereotypeDTO(nestedRouteModel.getStereotype()),
                                                     nestedRouteModel.getSemanticTerms());
    }
  }

  private Set<ErrorModel> toErrorModelsDTO(Set<org.mule.runtime.api.meta.model.error.ErrorModel> errorModels) {
    return errorModels.stream()
        .map(this::toErrorModelDTO)
        .collect(toCollection(LinkedHashSet::new));
  }

  private ErrorModel toErrorModelDTO(org.mule.runtime.api.meta.model.error.ErrorModel errorModel) {
    if (errorModel == null) {
      return null;
    }
    return new ErrorModel(errorModel.getType(), errorModel.getNamespace(), errorModel.isHandleable(),
                          toErrorModelDTO(errorModel.getParent().orElse(null)));
  }

  private SourceCallbackModel toSourceCallbackModelDTO(Optional<org.mule.runtime.api.meta.model.source.SourceCallbackModel> sourceCallbackModelOptional) {
    return sourceCallbackModelOptional
        .map(sourceCallbackModel -> new SourceCallbackModel(sourceCallbackModel.getName(),
                                                            sourceCallbackModel.getDescription(),
                                                            toParameterGroupModelsDTO(p -> empty(),
                                                                                      sourceCallbackModel
                                                                                          .getParameterGroupModels()),
                                                            toDisplayModelDTO(sourceCallbackModel.getDisplayModel())))
        .orElse(null);
  }

  private Set<StereotypeModel> toStereotypesDTO(Collection<org.mule.runtime.api.meta.model.stereotype.StereotypeModel> stereotypes) {
    // Mule Extension Models (json written by hand) has null value for this property
    if (stereotypes == null) {
      return emptySet();
    }
    return stereotypes.stream()
        .map(stereotype -> new StereotypeModel(stereotype.getType(), stereotype.getNamespace(),
                                               stereotype.getParent().map(ExtensionModelPartsFactory::toStereotypeDTO)
                                                   .orElse(null)))
        .collect(toCollection(LinkedHashSet::new));
  }

  public static <T extends Collection<StereotypeModel>> T toStereotypesDTO(Collection<org.mule.runtime.api.meta.model.stereotype.StereotypeModel> stereotypes,
                                                                           T output) {
    // Mule Extension Models (json written by hand) has null value for this property
    if (stereotypes == null) {
      return output;
    }
    return stereotypes.stream()
        .map(stereotype -> new StereotypeModel(stereotype.getType(), stereotype.getNamespace(),
                                               stereotype.getParent().map(ExtensionModelPartsFactory::toStereotypeDTO)
                                                   .orElse(null)))
        .collect(toCollection(() -> output));
  }

  public static StereotypeModel toStereotypeDTO(org.mule.runtime.api.meta.model.stereotype.StereotypeModel stereotype) {
    // Mule Extension Models (json written by hand) has null value for this property
    if (stereotype == null) {
      return null;
    }
    return new StereotypeModel(stereotype.getType(), stereotype.getNamespace(),
                               stereotype.getParent().map(ExtensionModelPartsFactory::toStereotypeDTO).orElse(null));
  }

  public static DeprecationModel toDeprecationModelDTO(org.mule.runtime.api.meta.model.deprecated.DeprecationModel deprecationModel) {
    if (deprecationModel == null) {
      return null;
    }
    return new DeprecationModel(deprecationModel.getMessage(), deprecationModel.getDeprecatedSince(),
                                deprecationModel.getToRemoveIn().orElse(null));
  }

  private static OutputModel toOutputModelDTO(org.mule.runtime.api.meta.model.OutputModel output) {
    return new OutputModel(output.getDescription(), output.getType(), output.hasDynamicType());
  }

  public OperationModel toOperationModelDTO(org.mule.runtime.api.meta.model.operation.OperationModel operationModel) {
    if (operationModel == null) {
      return null;
    }

    return new OperationModel(operationModel.getName(),
                              operationModel.getDescription(),
                              toParameterGroupModelsDTO(valueProviderEnricher(operationModel),
                                                        operationModel.getParameterGroupModels()),
                              toDisplayModelDTO(operationModel.getDisplayModel()),
                              operationModel.isBlocking(),
                              toErrorModelsDTO(operationModel.getErrorModels()),
                              toOutputModelDTO(operationModel.getOutput()),
                              toOutputModelDTO(operationModel.getOutputAttributes()),
                              operationModel.isTransactional(),
                              operationModel.requiresConnection(),
                              operationModel.supportsStreaming(),
                              toStereotypeDTO(operationModel.getStereotype()),
                              toNestedComponentModelsDTO(operationModel.getNestedComponents()),
                              toMetadataKeyIdModel(operationModel.getModelProperty(MetadataKeyIdModelProperty.class)),
                              toDeprecationModelDTO(operationModel.getDeprecationModel().orElse(null)),
                              toTypeResolversInformationModel(operationModel.getAllParameterModels(), operationModel
                                  .getModelProperty(TypeResolversInformationModelProperty.class).orElse(null)),
                              toSampleDataProviderModel(operationModel.getSampleDataProviderModel().orElse(null)),
                              operationModel.getSemanticTerms());
  }

  public static List<ParameterGroupModel> toParameterGroupModelsDTO(Function<org.mule.runtime.api.meta.model.parameter.ParameterModel, Optional<ValuesResolverModel>> valuesResolverEnricher,
                                                                    List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels) {
    return parameterGroupModels.stream()
        .map(parameterGroupModel -> new ParameterGroupModel(parameterGroupModel.getName(),
                                                            parameterGroupModel.getDescription(),
                                                            toDisplayModelDTO(parameterGroupModel.getDisplayModel()),
                                                            toLayoutModelDTO(parameterGroupModel.getLayoutModel()),
                                                            toParameterModelsDTO(valuesResolverEnricher,
                                                                                 parameterGroupModel.getParameterModels()),
                                                            toExclusiveParameterModelsDTO(parameterGroupModel
                                                                .getExclusiveParametersModels()),
                                                            parameterGroupModel.isShowInDsl()))
        .collect(toList());
  }

  private static List<ExclusiveParametersModel> toExclusiveParameterModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ExclusiveParametersModel> exclusiveParametersModels) {
    return exclusiveParametersModels.stream()
        .map(exclusiveParametersModel -> new ExclusiveParametersModel(copy(exclusiveParametersModel
            .getExclusiveParameterNames()),
                                                                      exclusiveParametersModel.isOneRequired()))
        .collect(toList());
  }

  private static List<ParameterModel> toParameterModelsDTO(Function<org.mule.runtime.api.meta.model.parameter.ParameterModel, Optional<ValuesResolverModel>> valuesResolverEnricher,
                                                           List<org.mule.runtime.api.meta.model.parameter.ParameterModel> parameterModels) {
    boolean filterParameters = parseBoolean(getProperty(FILTER_PARAMETERS_RESERVED_NAMES, "true"));
    return parameterModels.stream()
        .filter(parameterModel -> !filterParameters ||
            !((parameterModel.getName().equals(NAME_PARAMETER) && parameterModel.isComponentId()) ||
                parameterModel.getName().equals(CONFIG_REF_PARAMETER)))
        .map(parameterModel -> new ParameterModel(parameterModel.getName(),
                                                  parameterModel.getDescription(),
                                                  toDisplayModelDTO(parameterModel.getDisplayModel()),
                                                  parameterModel.getType(),
                                                  parameterModel.hasDynamicType(),
                                                  parameterModel.isRequired(),
                                                  parameterModel.isOverrideFromConfig(),
                                                  parameterModel.isComponentId(),
                                                  toExpressionSupportDTO(parameterModel.getExpressionSupport()),
                                                  parameterModel.getDefaultValue(),
                                                  toParameterRoleDTO(parameterModel.getRole()),
                                                  parameterModel.getDslConfiguration() != null
                                                      ? toParameterDslConfigurationDTO(parameterModel.getDslConfiguration())
                                                      : null,
                                                  toLayoutModelDTO(parameterModel.getLayoutModel()),
                                                  toMetadataKeyPartModel(parameterModel
                                                      .getModelProperty(MetadataKeyPartModelProperty.class)),
                                                  toQNameModel(parameterModel.getModelProperty(QNameModelProperty.class)),
                                                  toInfrastructureParameterModel(parameterModel
                                                      .getModelProperty(InfrastructureParameterModelProperty.class)),
                                                  toDefaultImplementingTypeModel(parameterModel
                                                      .getModelProperty(DefaultImplementingTypeModelProperty.class)),
                                                  toValueProviderModelDTO(parameterModel.getValueProviderModel()),
                                                  toStereotypesDTO(parameterModel.getAllowedStereotypes(), new ArrayList<>()),
                                                  toDeprecationModelDTO(parameterModel.getDeprecationModel().orElse(null)),
                                                  valuesResolverEnricher.apply(parameterModel).orElse(null),
                                                  toFieldValueProviderModelsDTO(parameterModel),
                                                  toFieldValuesResolverModelDTO(parameterModel),
                                                  parameterModel.getSemanticTerms()))
        .collect(toList());
  }

  private static QNameModel toQNameModel(Optional<QNameModelProperty> modelProperty) {
    return modelProperty.map(qNameProperty -> new QNameModel(qNameProperty.getValue()))
        .orElse(null);
  }

  private static DefaultImplementingTypeModel toDefaultImplementingTypeModel(Optional<DefaultImplementingTypeModelProperty> modelProperty) {
    return modelProperty
        .map(defaultImplementingTypeModelProperty -> new DefaultImplementingTypeModel(defaultImplementingTypeModelProperty
            .value()))
        .orElse(null);
  }

  private static InfrastructureParameterModel toInfrastructureParameterModel(Optional<InfrastructureParameterModelProperty> modelProperty) {
    return modelProperty
        .map(infrastructureParameterModelProperty -> new InfrastructureParameterModel(infrastructureParameterModelProperty
            .getSequence()))
        .orElse(null);
  }

  private static MetadataKeyPartModel toMetadataKeyPartModel(Optional<MetadataKeyPartModelProperty> modelProperty) {
    return modelProperty.map(metadataKeyPartModelProperty -> new MetadataKeyPartModel(metadataKeyPartModelProperty.getOrder(),
                                                                                      metadataKeyPartModelProperty
                                                                                          .isProvidedByKeyResolver()))
        .orElse(null);
  }

  private static MetadataKeyIdModel toMetadataKeyIdModel(Optional<MetadataKeyIdModelProperty> modelProperty) {
    return modelProperty
        .map(metadataKeyIdModel -> new MetadataKeyIdModel(metadataKeyIdModel.getType(), metadataKeyIdModel.getParameterName(),
                                                          metadataKeyIdModel.getCategoryName().orElse(null)))
        .orElse(null);
  }

  public static ExpressionSupport toExpressionSupportDTO(org.mule.runtime.api.meta.ExpressionSupport expressionSupport) {
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED) {
      return ExpressionSupport.supportedExpressionSupport(expressionSupport.name());
    }
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED) {
      return ExpressionSupport.notSupportedExpressionSupport(expressionSupport.name());
    }
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.REQUIRED) {
      return ExpressionSupport.requiredExpressionSupport(expressionSupport.name());
    }
    return new ExpressionSupport(expressionSupport.name());
  }

  private static ParameterDslConfiguration toParameterDslConfigurationDTO(
                                                                          org.mule.runtime.api.meta.model.ParameterDslConfiguration dslConfiguration) {
    return new ParameterDslConfiguration(dslConfiguration.allowsInlineDefinition(), dslConfiguration.allowTopLevelDefinition(),
                                         dslConfiguration.allowsReferences());
  }

  public static LayoutModel toLayoutModelDTO(Optional<org.mule.runtime.api.meta.model.display.LayoutModel> layoutModelOptional) {
    return layoutModelOptional.map(layoutModel -> new LayoutModel(layoutModel.getTabName().orElse(null),
                                                                  layoutModel.getOrder().orElse(null),
                                                                  layoutModel.isPassword(),
                                                                  layoutModel.isText(),
                                                                  layoutModel.isQuery()))
        .orElse(null);
  }

  private static ParameterRole toParameterRoleDTO(org.mule.runtime.api.meta.model.parameter.ParameterRole parameterRole) {
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR) {
      return ParameterRole.behaviourParameterRole(parameterRole.name());
    }
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.CONTENT) {
      return ParameterRole.contentParameterRole(parameterRole.name());
    }
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.PRIMARY_CONTENT) {
      return ParameterRole.primaryContentParameterRole(parameterRole.name());
    }
    return new ParameterRole(parameterRole.name());
  }

  public static DisplayModel toDisplayModelDTO(Optional<org.mule.runtime.api.meta.model.display.DisplayModel> displayModelOptional) {
    return displayModelOptional.map(displayModel -> new DisplayModel(displayModel.getDisplayName(), displayModel.getSummary(),
                                                                     displayModel.getExample(),
                                                                     toPathModelDTO(displayModel.getPathModel()),
                                                                     toClassValueModelDTO(displayModel.getClassValueModel())))
        .orElse(null);
  }

  public static ClassValueModel toClassValueModelDTO(Optional<org.mule.runtime.api.meta.model.display.ClassValueModel> classValue) {
    return classValue.map(cv -> new ClassValueModel(cv.getAssignableFrom())).orElse(null);
  }

  public static PathModel toPathModelDTO(Optional<org.mule.runtime.api.meta.model.display.PathModel> pathModelOptional) {
    return pathModelOptional.map(pathModel -> new PathModel(toPathModelTypeDTO(pathModel.getType()), pathModel.acceptsUrls(),
                                                            toPathModelLocationDTO(pathModel.getLocation()),
                                                            pathModel.getFileExtensions().toArray(new String[0])))
        .orElse(null);
  }

  private static PathModelType toPathModelTypeDTO(org.mule.runtime.api.meta.model.display.PathModel.Type type) {
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.DIRECTORY) {
      return PathModelType.directoryPathModelType(type.name());
    }
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.FILE) {
      return PathModelType.filePathModelType(type.name());
    }
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.ANY) {
      return PathModelType.anyPathModelType(type.name());
    }
    return new PathModelType(type.name());
  }

  private static PathModelLocation toPathModelLocationDTO(org.mule.runtime.api.meta.model.display.PathModel.Location location) {
    if (location == org.mule.runtime.api.meta.model.display.PathModel.Location.EMBEDDED) {
      return PathModelLocation.embeddedPathModelLocation(location.name());
    }
    if (location == org.mule.runtime.api.meta.model.display.PathModel.Location.EXTERNAL) {
      return PathModelLocation.externalPathModelLocation(location.name());
    }
    if (location == org.mule.runtime.api.meta.model.display.PathModel.Location.ANY) {
      return PathModelLocation.anyPathModelLocation(location.name());
    }
    return new PathModelLocation(location.name());
  }

  private static ValueProviderModel toValueProviderModelDTO(Optional<org.mule.runtime.api.meta.model.parameter.ValueProviderModel> valueProviderModel) {
    return valueProviderModel
        .map(model -> new ValueProviderModel(model.getActingParameters(),
                                             toActingParameterModelsDTO(model.getParameters()),
                                             model.requiresConfiguration(),
                                             model.requiresConnection(),
                                             model.isOpen(),
                                             model.getPartOrder(),
                                             model.getProviderName(),
                                             model.getProviderId()))
        .orElse(null);
  }

  public FunctionModel toFunctionModelDTO(org.mule.runtime.api.meta.model.function.FunctionModel functionModel) {
    return new FunctionModel(functionModel.getName(), functionModel.getDescription(),
                             toParameterGroupModelsDTO(p -> empty(), functionModel.getParameterGroupModels()),
                             toDisplayModelDTO(functionModel.getDisplayModel()), toOutputModelDTO(functionModel.getOutput()),
                             toDeprecationModelDTO(functionModel.getDeprecationModel().orElse(null)));
  }


  public ConstructModel toConstructModelDTO(org.mule.runtime.api.meta.model.construct.ConstructModel constructModel) {
    return new ConstructModel(constructModel.getName(), constructModel.getDescription(),
                              toParameterGroupModelsDTO(p -> createValueProviderEnricher(constructModel).apply(p),
                                                        constructModel.getParameterGroupModels()),
                              toDisplayModelDTO(constructModel.getDisplayModel()),
                              toStereotypeDTO(constructModel.getStereotype()),
                              toNestedComponentModelsDTO(constructModel.getNestedComponents()),
                              constructModel.allowsTopLevelDeclaration(),
                              toErrorModelsDTO(constructModel.getErrorModels()),
                              toDeprecationModelDTO(constructModel.getDeprecationModel().orElse(null)),
                              toTypeResolversInformationModel(constructModel.getAllParameterModels(), constructModel
                                  .getModelProperty(TypeResolversInformationModelProperty.class).orElse(null)),
                              constructModel.getSemanticTerms());
  }

}
