/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static org.mule.runtime.deployment.model.api.builder.DeployableArtifactClassLoaderFactoryProvider.applicationClassLoaderFactory;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;

import org.mule.runtime.deployment.model.api.builder.RegionPluginClassLoadersFactory;
import org.mule.runtime.deployment.model.api.plugin.resolver.PluginDependenciesResolver;
import org.mule.runtime.deployment.model.internal.tooling.ToolingApplicationClassLoaderBuilder;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.ApplicationDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.tooling.client.api.exception.ToolingException;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

/**
 * Factory that knows how to create a {@link ArtifactClassLoader} for applications.
 *
 * @since 4.0
 */
public class ApplicationClassLoaderFactory {

  private final ArtifactClassLoader containerArtifactClassLoader;
  private final RegionPluginClassLoadersFactory regionPluginClassLoadersFactory;
  private final PluginDependenciesResolver pluginDependenciesResolver;

  public ApplicationClassLoaderFactory(ArtifactClassLoader containerArtifactClassLoader,
                                       RegionPluginClassLoadersFactory regionPluginClassLoadersFactory,
                                       PluginDependenciesResolver pluginDependenciesResolver) {
    requireNonNull(containerArtifactClassLoader, "containerArtifactClassLoader cannot be null");
    requireNonNull(regionPluginClassLoadersFactory, "regionPluginClassLoadersFactory cannot be null");
    requireNonNull(pluginDependenciesResolver, "pluginDependenciesResolver cannot be null");

    this.containerArtifactClassLoader = containerArtifactClassLoader;
    this.regionPluginClassLoadersFactory = regionPluginClassLoadersFactory;
    this.pluginDependenciesResolver = pluginDependenciesResolver;
  }

  public ToolingArtifactClassLoader createApplicationClassLoader(ApplicationDescriptor applicationDescriptor,
                                                                 File workingDirectory,
                                                                 ToolingArtifactClassLoader parentToolingArtifactClassLoader) {
    try {
      ToolingApplicationClassLoaderBuilder builder =
          new ToolingApplicationClassLoaderBuilder(applicationClassLoaderFactory(name -> new File(workingDirectory, name)),
                                                   regionPluginClassLoadersFactory);

      Set<ArtifactPluginDescriptor> domainPluginDescriptors = new HashSet<>();
      boolean hasDomainDependency = applicationDescriptor.getDomainDescriptor().isPresent();
      if (!hasDomainDependency) {
        builder.setParentClassLoader(containerArtifactClassLoader);
      } else {
        final BundleDescriptor declaredDomainDescriptor = applicationDescriptor.getDomainDescriptor().get();
        if (hasDomainDependency && parentToolingArtifactClassLoader == null) {
          throw new IllegalStateException(format("Application '%s' declares a domain dependency '%s' that should have been already resolved",
                                                 applicationDescriptor.getArtifactDeclaration().getName(),
                                                 declaredDomainDescriptor));
        }

        for (ArtifactClassLoader artifactPluginClassLoader : parentToolingArtifactClassLoader.getArtifactPluginClassLoaders()) {
          domainPluginDescriptors.add(artifactPluginClassLoader.getArtifactDescriptor());
        }

        builder.setDomainParentClassLoader(parentToolingArtifactClassLoader.getRegionClassLoader());
      }

      pluginDependenciesResolver.resolve(domainPluginDescriptors, new ArrayList<>(applicationDescriptor.getPlugins()), true)
          .stream()
          .forEach(builder::addArtifactPluginDescriptors);

      builder.setArtifactDescriptor(applicationDescriptor);
      return builder.build();
    } catch (Exception e) {
      throw new ToolingException("Error while creating application class loader", e);
    }
  }
}
