/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static org.mule.runtime.config.api.properties.ConfigurationPropertiesResolverProvider.builder;
import static org.mule.runtime.core.api.util.ClassUtils.memoize;

import static java.lang.String.format;
import static java.util.Optional.of;

import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.config.api.properties.ConfigurationPropertiesResolverProvider;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.DomainDescriptor;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.DefaultApplicationModelFactory;
import org.mule.tooling.client.internal.ToolingArtifactContext;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;

/**
 * Represents a Mule domain and allows to be deployed into Mule Runtime.
 *
 * @since 4.0
 */
public class DefaultDomain extends AbstractArtifact<DomainDescriptor> implements Domain {

  private final ReentrantReadWriteLock deploymentLock = new ReentrantReadWriteLock();

  public DefaultDomain(String id, ArtifactResources artifactResources, DomainDescriptor domainDescriptor,
                       ToolingArtifactContext context,
                       Map<String, String> properties) {
    super(id, artifactResources, domainDescriptor, context, properties);
  }

  @Override
  public DomainDescriptor getDescriptor() {
    return artifactDescriptor;
  }

  @Override
  protected LazyValue<ToolingArtifactClassLoader> newToolingArtifactClassLoaderLazyValue() {
    return new LazyValue<>(() -> context.getDomainClassLoaderFactory()
        .createDomainClassLoader(artifactDescriptor, artifactResources.getWorkingDirectory()));
  }

  @Override
  protected LazyValue<ToolingApplicationModel> newToolingApplicationModelLazyValue() {
    return new LazyValue<>(() -> {
      // Cache the provider generated for the ast during its parse to be able to store it in the ToolingApplicationModel created
      // below
      Function<ArtifactAst, ConfigurationPropertiesResolverProvider> configurationPropertiesResolver =
          memoize(originalAst -> builder().from(originalAst)
              .withParentProperties(of(noFailureConfigurationProperties))
              .withDeploymentProperties(getProperties())
              .loadingResourcesWith(getArtifactClassLoader().getClassLoader())
              .build(),
                  new HashMap<>());

      final ArtifactAst ast = new DefaultApplicationModelFactory()
          .createApplicationModel(artifactDescriptor,
                                  getAstXmlParser(),
                                  getArtifactClassLoader()
                                      .getClassLoader(),
                                  configurationPropertiesResolver
                                      .andThen(ConfigurationPropertiesResolverProvider::getConfigurationPropertiesResolver))
          .orElseThrow(() -> new ToolingException(format("Couldn't create ApplicationModel from %s",
                                                         this)));

      return new ToolingApplicationModel(null,
                                         ast,
                                         configurationPropertiesResolver.apply(ast),
                                         getArtifactClassLoader().getClassLoader());
    });
  }


  @Override
  public <R> R evaluateWithRemoteDomain(DomainRemoteFunction<R> function) {
    checkState();
    if (!deployed) {
      Lock writeLock = deploymentLock.writeLock();
      if (writeLock.tryLock()) {
        try {
          if (!deployed) {
            deployed = true;
            remoteArtifactId = artifactDeployer.deploy(null);
            return function.apply(remoteArtifactId, runtimeToolingService.get());
          }
        } finally {
          writeLock.unlock();
        }
      }
    }

    Lock readLock = deploymentLock.readLock();
    readLock.lock();
    try {
      return function.apply(remoteArtifactId, runtimeToolingService.get());
    } finally {
      readLock.unlock();
    }
  }
}
