/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dsl;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.utils.ExtensionModelUtils.find;

import org.mule.maven.client.api.MavenClient;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.tooling.client.api.artifact.dsl.DslElementSyntax;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolver;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverFactory;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverService;
import org.mule.tooling.client.api.artifact.dsl.request.DslSyntaxResolverFactoryRequest;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.internal.Command;
import org.mule.tooling.client.internal.MuleRuntimeExtensionModelProvider;
import org.mule.tooling.client.internal.serialization.Serializer;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

/**
 * Default implementation of a {@link DslSyntaxResolverService}
 *
 * @since 1.0
 */
public class DefaultDslSyntaxResolverService implements DslSyntaxResolverService, Command {

  private final DslSyntaxServiceCache dslSyntaxServiceCache;
  private final MavenClient mavenClient;
  private final MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
  private final Serializer serializer;

  public DefaultDslSyntaxResolverService(DslSyntaxServiceCache dslSyntaxServiceCache, MavenClient mavenClient,
                                         MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                                         Serializer serializer) {
    this.dslSyntaxServiceCache = dslSyntaxServiceCache;
    this.mavenClient = mavenClient;
    this.muleRuntimeExtensionModelProvider = muleRuntimeExtensionModelProvider;
    this.serializer = serializer;
  }

  @Override
  public DslSyntaxResolverFactory getDslSyntaxResolverFactory(DslSyntaxResolverFactoryRequest request) {
    checkArgument(request != null, "The resolution request cannot be null");
    List<ArtifactDescriptor> pluginArtifactDescriptors = request.getPluginArtifactDescriptors();
    checkArgument(pluginArtifactDescriptors != null, "pluginArtifactDescriptors cannot be null");

    final DefaultDslSyntaxResolverService delegate = this;
    return new InternalDslSyntaxResolverFactory(delegate, pluginArtifactDescriptors);
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "getDslSyntaxResolverFactory": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(DslSyntaxResolverFactoryRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return getDslSyntaxResolverFactory(serializer.deserialize(arguments[0]));
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

  public void invalidate(List<ArtifactDescriptor> pluginArtifactDescriptors, ExtensionModel toolingModel) {
    dslSyntaxServiceCache.invalidate(pluginArtifactDescriptors, toolingModel);
  }

  public DslElementSyntax resolve(SourceModel component, ExtensionModel toolingModel,
                                  List<ArtifactDescriptor> pluginArtifactDescriptors) {
    return doResolve(extension -> find(extension, component), toolingModel, pluginArtifactDescriptors);
  }

  public DslElementSyntax resolve(OperationModel component, ExtensionModel toolingModel,
                                  List<ArtifactDescriptor> pluginArtifactDescriptors) {
    return doResolve(extension -> find(extension, component), toolingModel, pluginArtifactDescriptors);
  }

  public DslElementSyntax resolve(ConstructModel component, ExtensionModel toolingModel,
                                  List<ArtifactDescriptor> pluginArtifactDescriptors) {
    return doResolve(extension -> find(extension, component), toolingModel, pluginArtifactDescriptors);
  }

  public DslElementSyntax resolve(ConnectionProviderModel component, ExtensionModel toolingModel,
                                  List<ArtifactDescriptor> pluginArtifactDescriptors) {
    return doResolve(extension -> find(extension, component), toolingModel, pluginArtifactDescriptors);
  }

  public DslElementSyntax resolve(ConfigurationModel component, ExtensionModel toolingModel,
                                  List<ArtifactDescriptor> pluginArtifactDescriptors) {
    return doResolve(extension -> find(extension, component), toolingModel, pluginArtifactDescriptors);
  }

  public Set<ObjectType> getSubTypes(ObjectType type, List<ArtifactDescriptor> pluginArtifactDescriptors) {
    final DslResolvingContext context =
        dslSyntaxServiceCache.getContext(pluginArtifactDescriptors, mavenClient, muleRuntimeExtensionModelProvider);

    return context.getTypeCatalog().getSubTypes(type);
  }

  Optional<DslElementSyntax> resolve(MetadataType type,
                                     ExtensionModel toolingModel,
                                     List<ArtifactDescriptor> pluginArtifactDescriptors) {
    final DslResolvingContext context =
        dslSyntaxServiceCache.getContext(pluginArtifactDescriptors, mavenClient, muleRuntimeExtensionModelProvider);

    return dslSyntaxServiceCache.getResolver(findExtension(toolingModel, context), context)
        .resolve(type)
        .map(DefaultDslSyntaxResolverService::toDslDto);
  }

  private DslElementSyntax doResolve(Function<org.mule.runtime.api.meta.model.ExtensionModel, NamedObject> componentProvider,
                                     ExtensionModel toolingModel,
                                     List<ArtifactDescriptor> pluginArtifactDescriptors) {
    final DslResolvingContext context =
        dslSyntaxServiceCache.getContext(pluginArtifactDescriptors, mavenClient, muleRuntimeExtensionModelProvider);
    final org.mule.runtime.api.meta.model.ExtensionModel extension = findExtension(toolingModel, context);

    return toDslDto(dslSyntaxServiceCache.getResolver(extension, context)
        .resolve(componentProvider.apply(extension)));
  }

  private org.mule.runtime.api.meta.model.ExtensionModel findExtension(ExtensionModel toolingExtensionModel,
                                                                       DslResolvingContext resolvingContext) {
    return resolvingContext.getExtension(toolingExtensionModel.getName())
        .orElseThrow(() -> new IllegalArgumentException(
                                                        format("Extension with name [%s] not found in the current context. Available extensions are %s",
                                                               resolvingContext.getExtensions().stream().map(e -> e.getName())
                                                                   .collect(toList()))));
  }

  public static DslElementSyntax toDslDto(org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax origin) {
    List<MetadataType> metadataTypes = new ArrayList<>();
    List<DslElementSyntax> dslElementsSyntax = new ArrayList<>();
    origin.getGenerics().entrySet().stream().forEach(entry -> {
      metadataTypes.add(entry.getKey());
      dslElementsSyntax.add(toDslDto(entry.getValue()));
    });

    return new DslElementSyntax(origin.getAttributeName(),
                                origin.getElementName(),
                                origin.getPrefix(),
                                origin.getNamespace(),
                                origin.isWrapped(),
                                origin.supportsAttributeDeclaration(),
                                origin.supportsChildDeclaration(),
                                origin.supportsTopLevelDeclaration(),
                                origin.requiresConfig(),
                                metadataTypes,
                                dslElementsSyntax,
                                origin.getContainedElementsByName().entrySet().stream()
                                    .collect(toMap(Map.Entry::getKey, e -> toDslDto(e.getValue()))));
  }

  private class InternalDslSyntaxResolverFactory implements DslSyntaxResolverFactory, Command {

    private final DefaultDslSyntaxResolverService delegate;
    private final List<ArtifactDescriptor> pluginArtifactDescriptors;

    public InternalDslSyntaxResolverFactory(DefaultDslSyntaxResolverService delegate,
                                            List<ArtifactDescriptor> pluginArtifactDescriptors) {
      this.delegate = delegate;
      this.pluginArtifactDescriptors = pluginArtifactDescriptors;
    }

    @Override
    public DslSyntaxResolver createDslResolver(ExtensionModel toolingModel) {
      return new ServiceDelegatingDslSyntaxResolver(delegate, toolingModel, pluginArtifactDescriptors, serializer);
    }

    @Override
    public void dispose() {
      dslSyntaxServiceCache.invalidate(pluginArtifactDescriptors);
    }

    @Override
    public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
      switch (methodName) {
        case "createDslResolver": {
          checkState(arguments.length == 1,
                     format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
          checkState(classes.length == 1 && classes[0].equals(ExtensionModel.class.getName()),
                     format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
          return createDslResolver(serializer.deserialize(arguments[0]));
        }
        case "dispose": {
          dispose();
          return null;
        }
      }
      throw methodNotFound(this.getClass(), methodName);
    }

  }
}
