/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static java.util.Collections.synchronizedList;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.converters.collections.MapConverter;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.mapper.Mapper;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Implementation that allows to serialize/deserialize objects using xStream.
 *
 * @since 4.0
 */
public class XStreamServerSerializer implements Serializer {

  public static String NAME = "xStream";

  private ClassLoader targetClassLoader;
  private List<Class> typeAnnotationsNotSupportedByToolingRuntimeClient;

  public XStreamServerSerializer(ClassLoader targetClassLoader) {
    this.targetClassLoader = targetClassLoader;
    this.typeAnnotationsNotSupportedByToolingRuntimeClient = synchronizedList(new ArrayList<>());
  }

  private static XStream createXStream(ClassLoader targetClassLoader,
                                       List<Class> typeAnnotationsNotsupportedByToolingRuntimeClient) {
    final XStream xStream = XStreamFactory.createXStream();
    xStream
        .registerConverter(new TypeAnnotationsMapConverter(targetClassLoader, typeAnnotationsNotsupportedByToolingRuntimeClient,
                                                           xStream.getMapper()));
    xStream.registerConverter(new XmlArtifactDeserializationRequestConverter(xStream.getMapper()));
    xStream.ignoreUnknownElements();
    return xStream;
  }

  private static class TypeAnnotationsMapConverter extends MapConverter {

    private TypeAnnotationMapWriter typeAnnotationMapWriter;
    private TypeAnnotationMapReader typeAnnotationMapReader;

    public TypeAnnotationsMapConverter(ClassLoader targetClassLoader,
                                       List<Class> typeAnnotationsNotSupportedByToolingRuntimeClient, Mapper mapper) {
      super(mapper);
      this.typeAnnotationMapWriter =
          new TypeAnnotationMapWriter(targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient);
      this.typeAnnotationMapReader = new TypeAnnotationMapReader();
    }

    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
      super.marshal(typeAnnotationMapWriter.write((Map) source), writer, context);
    }

    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
      Map output = (Map) super.unmarshal(reader, context);
      return typeAnnotationMapReader.read(output);
    }
  }

  @Override
  public String serialize(Object object) {
    return createXStream(targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient).toXML(object);
  }

  @Override
  public <T> T deserialize(String content) {
    return (T) createXStream(targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient).fromXML(content);
  }

}
