/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.filter;

import static java.util.Collections.emptySet;
import static java.util.function.UnaryOperator.identity;
import static java.util.stream.Collectors.toMap;
import static org.mule.runtime.api.metadata.DataType.STRING;
import static org.mule.runtime.module.extension.internal.loader.utils.FieldValueProviderNameUtils.getParameterName;
import static org.mule.tooling.client.internal.session.filter.FilterUtils.getParameterConfiguredValue;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionExecutionException;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.runtime.module.tooling.internal.artifact.params.ParameterExtractor;
import org.mule.tooling.client.internal.session.filter.exception.InvalidLevelValueException;
import org.mule.tooling.client.internal.session.filter.exception.MissingLevelException;
import org.mule.tooling.client.internal.session.filter.exception.UnknownLevelValueException;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

public class FieldValueFilter {

  private final ExpressionLanguage expressionLanguage;

  public FieldValueFilter(ExpressionLanguage expressionLanguage) {
    this.expressionLanguage = expressionLanguage;
  }

  public Set<Value> filter(FieldValueProviderModel fieldValueProviderModel,
                           ParameterModel parameterModel,
                           ParameterizedElementDeclaration parameterizedElementDeclaration,
                           Set<Value> values) {
    BindingContext bindingContext = createBindingContext(fieldValueProviderModel, parameterizedElementDeclaration);
    Map<Integer, FieldValueProviderModel> orderedModels = orderFieldModels(parameterModel, fieldValueProviderModel);
    Map<Integer, Set<Value>> valuesByPartOrder = new HashMap<>();
    try {
      filterValues(valuesByPartOrder, values, orderedModels, bindingContext, 1);
    } catch (MissingLevelException e) {
      Set<Value> level = valuesByPartOrder.get(fieldValueProviderModel.getPartOrder());
      if (level == null) {
        throw e;
      }
      return level;
    }
    return emptySet();
  }

  private Map<Integer, FieldValueProviderModel> orderFieldModels(ParameterModel parameterModel,
                                                                 FieldValueProviderModel requestedModel) {
    return parameterModel
        .getFieldValueProviderModels()
        .stream()
        .filter(m -> m.getProviderId().equals(requestedModel.getProviderId()))
        .filter(m -> m.getProviderName().equals(requestedModel.getProviderName()))
        .collect(toMap(FieldValueProviderModel::getPartOrder, identity()));
  }

  private BindingContext createBindingContext(FieldValueProviderModel fieldValueProviderModel,
                                              ParameterizedElementDeclaration parameterizedElementDeclaration) {
    final String parameterName = getParameterName(fieldValueProviderModel);
    final BindingContext.Builder bindingContextBuilder = BindingContext.builder();
    getParameterConfiguredValue(parameterizedElementDeclaration, parameterName)
        .map(ParameterExtractor::asDataWeaveExpression)
        .ifPresent(tv -> bindingContextBuilder.addBinding(keywordSafeName(parameterName), tv));
    return bindingContextBuilder.build();
  }

  private void filterValues(Map<Integer, Set<Value>> valuesByPartOrder,
                            Set<Value> values,
                            Map<Integer, FieldValueProviderModel> orderedFieldModels,
                            BindingContext bindingContext,
                            int level) {
    if (!values.stream().findFirst().isPresent()) {
      return;
    }
    valuesByPartOrder.put(level, values);
    filterValues(
                 valuesByPartOrder,
                 getMatchingValue(values, orderedFieldModels.get(level), bindingContext).getChilds(),
                 orderedFieldModels,
                 bindingContext,
                 level + 1);
  }

  private Value getMatchingValue(Set<Value> values,
                                 FieldValueProviderModel fieldModel,
                                 BindingContext bindingContext) {
    final String parameterName = getParameterName(fieldModel);
    final String keywordSafeProviderName = keywordSafeName(parameterName);
    final String extractionExpression = parameterName + "." + fieldModel.getTargetSelector();
    final String sanitizedExtractionExpression = keywordSafeProviderName + "." + fieldModel.getTargetSelector();
    // There is no need to execute the expression if there was nothing added to the bindingContext
    final TypedValue<?> valuePartConfiguredValue = bindingContext.lookup(keywordSafeProviderName)
        .map(tv -> {
          try {
            return expressionLanguage.evaluate(sanitizedExtractionExpression, STRING, bindingContext);
          } catch (ExpressionExecutionException e) {
            throw new InvalidLevelValueException(extractionExpression, e);
          }
        }).orElseThrow(() -> new MissingLevelException(parameterName));

    if (valuePartConfiguredValue.getValue() == null) {
      throw new MissingLevelException(extractionExpression);
    }

    if (!String.class.isAssignableFrom(valuePartConfiguredValue.getDataType().getType())) {
      throw new UnknownLevelValueException(extractionExpression, valuePartConfiguredValue.getValue().toString());
    }
    final String stringValue = (String) valuePartConfiguredValue.getValue();
    return values
        .stream()
        .filter(v -> v.getId().equals(stringValue))
        .findAny()
        .orElseThrow(() -> new UnknownLevelValueException(extractionExpression, stringValue));
  }

  private String keywordSafeName(String parameterName) {
    return parameterName + "_";
  }

}
