/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.mediator;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static org.mule.tooling.client.internal.session.validation.SessionCallValidator.INVALID_ELEMENT;
import static org.mule.tooling.client.internal.session.validation.SessionCallValidator.INVALID_PARAMETER;

import org.mule.runtime.api.meta.model.EnrichableModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclarationVisitor;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;
import org.mule.runtime.extension.api.property.ResolverInformation;
import org.mule.runtime.extension.api.property.TypeResolversInformationModelProperty;
import org.mule.tooling.client.api.value.resolver.ValueResolverFailure;
import org.mule.tooling.client.api.value.resolver.ValueResolverResult;
import org.mule.tooling.client.internal.session.ConfigurationDeclarationProvider;
import org.mule.tooling.client.internal.session.mediator.resolver.ValuesResolverFactory;
import org.mule.tooling.client.internal.session.validation.ConnectionAndConfigurationValidator;
import org.mule.tooling.client.internal.session.validation.ParameterExistsValidator;
import org.mule.tooling.client.internal.session.validation.SessionCallValidationException;
import org.mule.tooling.client.internal.session.validation.SessionCallValidator;
import org.mule.tooling.client.internal.session.validation.ValueProviderActingParametersValidator;

import java.util.Optional;

public class ValuesResolverModelMediator {

  private final ConfigurationDeclarationProvider configurationDeclarationProvider;
  private final SessionCallValidator sessionCallValidator;
  private final ParameterizedElementDeclaration parameterizedElementDeclaration;
  private final String parameterName;

  private final ValueResolverResult failure;

  private Reference<ParameterizedModel> parameterizedModel = new Reference<>();
  private Reference<ParameterModel> parameterModel = new Reference<>();
  private ComponentElementDeclaration<?> componentElementDeclaration;

  private ValueProviderModel valueProviderModel;
  private MetadataKeyPartModelProperty metadataKeyPartModelProperty;

  public ValuesResolverModelMediator(ConfigurationDeclarationProvider configurationDeclarationProvider,
                                     ParameterizedElementDeclaration elementDeclaration,
                                     SessionCallValidator sessionCallValidator,
                                     String parameterName) {
    this.configurationDeclarationProvider = configurationDeclarationProvider;
    this.parameterizedElementDeclaration = elementDeclaration;
    this.parameterName = parameterName;
    this.sessionCallValidator = sessionCallValidator;

    this.failure = validateModel();
  }

  public ValueResolverResult resolve(ValuesResolverFactory valuesResolverFactory, boolean ignoreCache) {
    if (failure != null) {
      return failure;
    }

    if (isValueProviderResolver()) {
      return valuesResolverFactory.createValueProviderResolver(
                                                               parameterizedElementDeclaration,
                                                               valueProviderModel,
                                                               ignoreCache)
          .resolve(parameterizedModel.get(), parameterModel.get(), parameterName);
    } else {
      return valuesResolverFactory.createMetadataKeyResolver(
                                                             componentElementDeclaration,
                                                             metadataKeyPartModelProperty,
                                                             ignoreCache)
          .resolve(parameterizedModel.get(), parameterModel.get(), parameterName);
    }
  }

  private boolean isValueProviderResolver() {
    return valueProviderModel != null;
  }

  private ValueResolverResult validateModel() {
    try {
      final Reference<Boolean> requiresConfig = new Reference<>(false);
      final Reference<Boolean> requiresConnection = new Reference<>(false);


      sessionCallValidator.validateComponent(parameterizedElementDeclaration,
                                             c -> {
                                               this.parameterizedModel.set(c.getParameterizedModel());

                                               ParameterExistsValidator parameterExistsValidator =
                                                   new ParameterExistsValidator(parameterName);
                                               parameterExistsValidator.validate(c);
                                               this.parameterModel.set(parameterExistsValidator.getParameterModel());

                                               if (parameterModel.get().getValueProviderModel().isPresent()) {
                                                 this.valueProviderModel = parameterModel.get().getValueProviderModel().get();
                                                 requiresConfig.set(valueProviderModel.requiresConfiguration());
                                                 requiresConnection.set(valueProviderModel.requiresConnection());
                                               } else {
                                                 Optional<MetadataKeyPartModelProperty> metadataKeyPartModelProperty =
                                                     parameterModel.get().getModelProperty(MetadataKeyPartModelProperty.class);
                                                 if (!metadataKeyPartModelProperty.isPresent()) {
                                                   throw new SessionCallValidationException(
                                                                                            format("Parameter: '%s' on element : '%s' for extension: '%s' is not marked as MetadataKeyPartModelProperty and doesn't have a ValueProvider defined",
                                                                                                   parameterName,
                                                                                                   parameterizedElementDeclaration
                                                                                                       .getName(),
                                                                                                   parameterizedElementDeclaration
                                                                                                       .getDeclaringExtension()),
                                                                                            "ParameterModel doesn't support values resolution",
                                                                                            INVALID_PARAMETER);
                                                 }
                                                 this.metadataKeyPartModelProperty = metadataKeyPartModelProperty.get();

                                                 ComponentParameterizedElementDeclarationVisitor componentParameterizedElementDeclarationVisitor =
                                                     new ComponentParameterizedElementDeclarationVisitor();
                                                 parameterizedElementDeclaration
                                                     .accept(componentParameterizedElementDeclarationVisitor);
                                                 if (!componentParameterizedElementDeclarationVisitor.getComponentElement()
                                                     .isPresent()) {
                                                   throw new SessionCallValidationException(
                                                                                            format("Parameter: '%s' on element : '%s' for extension: '%s' is marked as MetadataKeyPartModelProperty but declaration is not a ComponentElementDeclaration. Only ComponentElementDeclaration could support metadata keys.",
                                                                                                   parameterName,
                                                                                                   parameterizedElementDeclaration
                                                                                                       .getName(),
                                                                                                   parameterizedElementDeclaration
                                                                                                       .getDeclaringExtension()),
                                                                                            "ParameterModel doesn't support values resolution",
                                                                                            INVALID_PARAMETER);

                                                 }
                                                 this.componentElementDeclaration =
                                                     componentParameterizedElementDeclarationVisitor.getComponentElement().get();
                                                 EnrichableModel enrichableModel = c.getEnrichableModel();
                                                 Optional<ResolverInformation> keyResolverInfo = enrichableModel
                                                     .getModelProperty(TypeResolversInformationModelProperty.class)
                                                     .flatMap(TypeResolversInformationModelProperty::getKeysResolver);
                                                 if (!keyResolverInfo.isPresent()) {
                                                   throw new SessionCallValidationException(
                                                                                            format("Parameter: '%s' on element : '%s' for extension: '%s' is marked as MetadataKeyPartModelProperty but Component does not have any KeyResolverInformation",
                                                                                                   parameterName,
                                                                                                   parameterizedElementDeclaration
                                                                                                       .getName(),
                                                                                                   parameterizedElementDeclaration
                                                                                                       .getDeclaringExtension()),
                                                                                            "ParameterizedModel does not have KeyResolverInformation",
                                                                                            INVALID_ELEMENT);
                                                 }
                                                 requiresConfig.set(keyResolverInfo.get().isRequiresConfiguration());
                                                 requiresConnection.set(keyResolverInfo.get().isRequiresConnection());
                                               }
                                             },
                                             new ValueProviderActingParametersValidator(parameterName),
                                             new ConnectionAndConfigurationValidator(parameterizedElementDeclaration,
                                                                                     configurationDeclarationProvider,
                                                                                     requiresConnection::get,
                                                                                     requiresConfig::get,
                                                                                     () -> format("Parameter: '%s' on element : '%s' for extension: '%s'",
                                                                                                  parameterName,
                                                                                                  parameterizedElementDeclaration
                                                                                                      .getName(),
                                                                                                  parameterizedElementDeclaration
                                                                                                      .getDeclaringExtension())));
    } catch (SessionCallValidationException e) {
      return ValueResolverResult.failure(new ValueResolverFailure(e.getMessage(), e.getReason(), e.getFailureCode()));
    }
    return null;
  }

  static class ComponentParameterizedElementDeclarationVisitor implements ParameterizedElementDeclarationVisitor {

    private ComponentElementDeclaration<?> componentElementDeclaration;

    @Override
    public void visitComponentElementDeclaration(ComponentElementDeclaration componentElementDeclaration) {
      this.componentElementDeclaration = componentElementDeclaration;
    }

    public Optional<ComponentElementDeclaration<?>> getComponentElement() {
      return ofNullable(componentElementDeclaration);
    }
  }

}
