/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.mediator.resolver;

import static java.util.Collections.emptySet;
import static org.mule.runtime.api.metadata.resolving.FailureCode.INVALID_METADATA_KEY;
import static org.mule.tooling.client.api.value.resolver.ValueResolverResult.failure;
import static org.mule.tooling.client.internal.utils.ValueResolverUtils.toValueResolverResult;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.meta.model.EnrichableModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.metadata.MetadataKey;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataFailure;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;
import org.mule.runtime.extension.api.property.TypeResolversInformationModelProperty;
import org.mule.tooling.client.api.value.resolver.ValueResolverFailure;
import org.mule.tooling.client.api.value.resolver.ValueResolverResult;
import org.mule.tooling.client.internal.session.filter.MetadataKeyFilter;
import org.mule.tooling.client.internal.session.cache.DeclarationMetadataCache;
import org.mule.tooling.client.internal.session.filter.exception.InvalidLevelValueException;
import org.mule.tooling.client.internal.session.filter.exception.MissingLevelException;
import org.mule.tooling.client.internal.session.filter.exception.UnknownLevelValueException;

import java.util.Set;
import java.util.concurrent.Callable;

import org.slf4j.Logger;

public class MetadataKeyResolver extends SkeletalValueResolver<MetadataResult<MetadataKeysContainer>> implements ValuesResolver {

  private static final Logger LOGGER = getLogger(MetadataKeyResolver.class);

  private final ComponentElementDeclaration<?> elementDeclaration;
  private final MetadataKeyPartModelProperty metadataKeyPartModelProperty;
  private final DeclarationMetadataCache metadataCache;
  private final MetadataKeyFilter metadataKeyFilter;


  public MetadataKeyResolver(SessionFunction<MetadataResult<MetadataKeysContainer>> sessionFunction,
                             boolean ignoreCache,
                             ComponentElementDeclaration<?> elementDeclaration,
                             MetadataKeyPartModelProperty metadataKeyPartModelProperty,
                             DeclarationMetadataCache metadataCache,
                             MetadataKeyFilter metadataKeyFilter) {
    super(sessionFunction, ignoreCache);
    this.elementDeclaration = elementDeclaration;
    this.metadataKeyPartModelProperty = metadataKeyPartModelProperty;
    this.metadataCache = metadataCache;
    this.metadataKeyFilter = metadataKeyFilter;
  }

  @Override
  public ValueResolverResult resolve(ParameterizedModel parameterizedModel, ParameterModel parameterModel, String parameterName) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(
                   "Resolving values using Metadata Keys mechanism for parameter: '{}' on element: '{}' for declared for extension: '{}'",
                   parameterName, elementDeclaration.getName(), elementDeclaration.getDeclaringExtension());
    }

    EnrichableModel enrichableModel = (EnrichableModel) parameterizedModel;
    boolean partialTypeKeyResolver =
        enrichableModel.getModelProperty(TypeResolversInformationModelProperty.class)
            .map(TypeResolversInformationModelProperty::isPartialTypeKeyResolver)
            .orElse(false);

    Callable<MetadataResult<MetadataKeysContainer>> keysCallable =
        () -> sessionFunction.apply((runtimeToolingService, sessionId) -> runtimeToolingService
            .getComponentMetadataKeys(sessionId, elementDeclaration, ignoreCache));

    org.mule.runtime.api.metadata.resolving.MetadataResult<MetadataKeysContainer> metadataKeysContainerMetadataResult;
    if (partialTypeKeyResolver) {
      metadataKeysContainerMetadataResult =
          metadataCache.getMetadataKeysPartialFetch(parameterizedModel,
                                                    elementDeclaration,
                                                    keysCallable,
                                                    ignoreCache);
    } else {
      metadataKeysContainerMetadataResult =
          metadataCache.getMetadataKeys(
                                        elementDeclaration,
                                        keysCallable,
                                        ignoreCache);

    }
    if (!metadataKeysContainerMetadataResult.isSuccess()) {
      MetadataFailure metadataFailure = metadataKeysContainerMetadataResult.getFailures().stream().findFirst().get();
      return failure(new ValueResolverFailure(metadataFailure.getMessage(), metadataFailure.getReason(),
                                              transformMetadataKeyFailureCode(metadataFailure)));
    }
    // Metadata Keys resolved for an component have unique category
    Set<MetadataKey> metadataKeys = metadataKeysContainerMetadataResult.get().getKeysByCategory().values().stream().findFirst()
        .orElse(emptySet());
    Set<MetadataKey> requestedLevelKeys = emptySet();
    try {
      requestedLevelKeys = metadataKeyFilter.filter(parameterName, elementDeclaration, metadataKeys);
    } catch (MissingLevelException e) {
      return missingActingParametersFailure(e.getMissingParameterName());
    } catch (InvalidLevelValueException e) {
      return invalidActingParameterFailure(e.getTargetSelector(), e.getMessage());
    } catch (UnknownLevelValueException e) {
      if (metadataKeyPartModelProperty.isProvidedByKeyResolver() &&
          metadataKeyPartModelProperty.getOrder() > 1) {
        return unknownActingParametersFailure(e.getParameterName(), e.getParameterValue());
      }
    }
    return toValueResolverResult(requestedLevelKeys);
  }

  private String transformMetadataKeyFailureCode(MetadataFailure metadataFailure) {
    org.mule.runtime.api.metadata.resolving.FailureCode failureCode = metadataFailure.getFailureCode();
    if (failureCode == null) {
      return null;
    }
    if (failureCode.equals(INVALID_METADATA_KEY)) {
      return INVALID_ACTING_PARAMETER_VALUES;
    }
    return failureCode.getName();
  }
}
