/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.validation;

import static java.lang.String.format;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.mule.runtime.api.metadata.resolving.FailureCode.COMPONENT_NOT_FOUND;
import static org.mule.sdk.api.values.ValueResolvingException.MISSING_REQUIRED_PARAMETERS;
import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.app.declaration.api.ConfigurationElementDeclaration;
import org.mule.runtime.app.declaration.api.ConnectionElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.tooling.client.internal.session.ConfigurationDeclarationProvider;

import java.util.Optional;
import java.util.function.Supplier;

public class ConnectionAndConfigurationValidator implements Validators.ComponentValidator {

  private final ParameterizedElementDeclaration elementDeclaration;
  private final Supplier<Boolean> requiresConnection;
  private final Supplier<Boolean> requiresConfiguration;
  private final ConfigurationDeclarationProvider configurationDeclarationProvider;

  private final Supplier<String> errorLocationPrefix;

  public ConnectionAndConfigurationValidator(ParameterizedElementDeclaration elementDeclaration,
                                             ConfigurationDeclarationProvider configurationDeclarationProvider,
                                             Supplier<Boolean> requiresConnection,
                                             Supplier<Boolean> requiresConfiguration,
                                             Supplier<String> errorLocationPrefix) {
    this.elementDeclaration = elementDeclaration;
    this.configurationDeclarationProvider = configurationDeclarationProvider;
    this.requiresConnection = requiresConnection;
    this.requiresConfiguration = requiresConfiguration;
    this.errorLocationPrefix = errorLocationPrefix;
  }

  @Override
  public void validate(ComponentValidationContext<?> context) throws SessionCallValidationException {
    boolean requiresConfig = requiresConfiguration.get();
    boolean requiresConn = requiresConnection.get();
    String requiresString = requiresConfig ? (requiresConn ? "configuration and connection" : "configuration") : "connection";

    if (elementDeclaration instanceof ComponentElementDeclaration && (requiresConfig || requiresConn)) {
      String configRef = ((ComponentElementDeclaration) elementDeclaration).getConfigRef();
      if (isEmpty(configRef)) {
        throw new SessionCallValidationException(format("%s requires %s for resolution but there is no configuration referenced",
                                                        errorLocationPrefix.get(),
                                                        requiresString),
                                                 "There is no configuration referenced",
                                                 MISSING_REQUIRED_PARAMETERS);
      }
      Optional<ConfigurationElementDeclaration> configurationElementDeclaration =
          configurationDeclarationProvider.getConfigDeclaration(configRef);
      if (!configurationElementDeclaration.isPresent()) {
        throw new SessionCallValidationException(
                                                 format("%s requires %s for resolution but the configuration referenced with ref-name: '%s' is not present",
                                                        errorLocationPrefix.get(),
                                                        requiresString,
                                                        configRef),
                                                 "Configuration referenced is not present",
                                                 COMPONENT_NOT_FOUND.getName());
      }
      if (requiresConn) {
        Optional<ConnectionElementDeclaration> connectionElementDeclaration =
            configurationElementDeclaration.get().getConnection();
        if (!connectionElementDeclaration.isPresent()) {
          throw new SessionCallValidationException(
                                                   format("%s requires connection for resolution but the configuration referenced with ref-name: '%s' does not have a connection",
                                                          errorLocationPrefix.get(),
                                                          configRef),
                                                   "Configuration referenced does not have connection",
                                                   COMPONENT_NOT_FOUND.getName());
        }
      }
    }
  }
}
