/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.validation;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.mule.runtime.api.metadata.resolving.FailureCode.COMPONENT_NOT_FOUND;
import static org.mule.runtime.api.metadata.resolving.FailureCode.INVALID_CONFIGURATION;
import static org.mule.runtime.module.tooling.internal.utils.ArtifactHelperUtils.findConfigAwareModel;
import org.mule.runtime.api.meta.model.EnrichableModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.app.declaration.api.ConfigurationElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.runtime.module.tooling.internal.utils.ArtifactHelperUtils;
import org.mule.tooling.client.internal.session.ConfigurationDeclarationProvider;
import org.mule.tooling.client.internal.session.ExtensionModelProvider;

import java.util.Optional;

public class SessionCallValidator {

  public static final String INVALID_TARGET_EXTENSION = "INVALID_TARGET_EXTENSION";
  public static final String INVALID_ELEMENT = "INVALID_ELEMENT";
  public static final String INVALID_PARAMETER = "INVALID_PARAMETER";

  private final ExtensionModelProvider extensionModelProvider;
  private final ConfigurationDeclarationProvider configurationDeclarationProvider;

  public SessionCallValidator(ExtensionModelProvider extensionModelProvider,
                              ConfigurationDeclarationProvider configurationDeclarationProvider) {
    this.extensionModelProvider = extensionModelProvider;
    this.configurationDeclarationProvider = configurationDeclarationProvider;
  }

  public void validateConfig(String configRef,
                             Validators.ConfigValidator... extraValidators)
      throws SessionCallValidationException {
    Optional<ConfigurationElementDeclaration> configurationElementDeclarationOptional =
        configurationDeclarationProvider.getConfigDeclaration(configRef);

    if (!configurationElementDeclarationOptional.isPresent()) {
      throw new SessionCallValidationException(format("Configuration with ref-name: '%s' does not exist in the session",
                                                      configRef),
                                               "Config not found",
                                               COMPONENT_NOT_FOUND.getName());
    }

    final ConfigurationElementDeclaration configurationElementDeclaration = configurationElementDeclarationOptional.get();
    final ExtensionModel extensionModel = getExtensionModelOrFail(configurationElementDeclaration.getDeclaringExtension());
    final ArtifactHelperUtils.ConfigAwareModel<ConfigurationModel> configurationModel =
        configAwareModelOrFail(extensionModel, configurationElementDeclaration);


    ConfigValidationContext context = new ConfigValidationContext(extensionModel,
                                                                  configurationModel.getModel(),
                                                                  configurationElementDeclaration);

    for (Validators.ConfigValidator customValidator : extraValidators) {
      customValidator.validate(context);
    }
  }

  public void validateComponent(ParameterizedElementDeclaration parameterizedElementDeclaration,
                                Validators.ComponentValidator... extraValidators)
      throws SessionCallValidationException {

    final ExtensionModel extensionModel = getExtensionModelOrFail(parameterizedElementDeclaration.getDeclaringExtension());

    final ArtifactHelperUtils.ConfigAwareModel<?> configAwareComponentModel =
        configAwareModelOrFail(extensionModel, parameterizedElementDeclaration);

    ConfigurationModel configurationModel = null;
    ConfigurationElementDeclaration configurationElementDeclaration = null;

    if (configAwareComponentModel.getConfigModel().isPresent()
        && parameterizedElementDeclaration instanceof ComponentElementDeclaration) {
      configurationModel = configAwareComponentModel.getConfigModel().get();
      String configRef = ((ComponentElementDeclaration) parameterizedElementDeclaration).getConfigRef();
      if (!isEmpty(configRef)) {
        Optional<ConfigurationElementDeclaration> configurationElementDeclarationOptional =
            configurationDeclarationProvider.getConfigDeclaration(configRef);

        if (configurationElementDeclarationOptional.isPresent()) {
          configurationElementDeclaration = configurationElementDeclarationOptional.get();

          if (!configurationModel.getName().equals(configurationElementDeclaration.getName())) {
            throw new SessionCallValidationException(format("Component with name: '%s' is referencing a config with name: '%s' and it should reference a config with name: '%s'",
                                                            parameterizedElementDeclaration.getName(),
                                                            configurationElementDeclaration.getName(),
                                                            configurationModel.getName()),
                                                     "Referencing wrong config",
                                                     INVALID_CONFIGURATION.getName());
          }
        }
      }
    }

    ComponentValidationContext context =
        new ComponentValidationContext<>(extensionModel,
                                         configurationModel,
                                         configurationElementDeclaration,
                                         parameterizedElementDeclaration,
                                         configAwareComponentModel.getModel());

    for (Validators.ComponentValidator customValidator : extraValidators) {
      customValidator.validate(context);
    }
  }

  private ExtensionModel getExtensionModelOrFail(String name) throws SessionCallValidationException {
    return extensionModelProvider.get(name).orElseThrow(
                                                        () -> new SessionCallValidationException(format("ElementDeclaration is declaring an extension: '%s' that is not part of the session: '%s'",
                                                                                                        name,
                                                                                                        extensionModelProvider
                                                                                                            .getAllNames()),
                                                                                                 "ExtensionModel not found",
                                                                                                 INVALID_TARGET_EXTENSION));
  }

  private <T extends ParameterizedModel & EnrichableModel> ArtifactHelperUtils.ConfigAwareModel<T> configAwareModelOrFail(ExtensionModel extensionModel,
                                                                                                                          ParameterizedElementDeclaration declaration)
      throws SessionCallValidationException {
    Optional<ArtifactHelperUtils.ConfigAwareModel<T>> caw = findConfigAwareModel(extensionModel, declaration);
    if (!caw.isPresent()) {
      throw new SessionCallValidationException(format("Model not found for name: '%s' and extension: '%s'",
                                                      declaration.getName(),
                                                      declaration.getDeclaringExtension()),
                                               "Model not found in extension model",
                                               INVALID_ELEMENT);
    }
    return caw.get();
  }
}
