/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.github.tomakehurst.wiremock.client.WireMock.aResponse;
import static com.github.tomakehurst.wiremock.client.WireMock.get;
import static com.github.tomakehurst.wiremock.client.WireMock.urlEqualTo;
import static com.github.tomakehurst.wiremock.core.WireMockConfiguration.wireMockConfig;
import static java.util.Collections.emptyList;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.sameInstance;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.ArgumentMatchers.isNull;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.mule.tooling.client.test.utils.ZipUtils.compress;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ClassLoaderModel;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.internal.application.ApplicationClassLoaderFactory;
import org.mule.tooling.client.internal.application.ArtifactResources;
import org.mule.tooling.client.internal.application.DefaultApplication;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.beust.jcommander.internal.Lists;
import com.github.tomakehurst.wiremock.junit.WireMockClassRule;
import com.google.common.collect.ImmutableMap;
import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.io.IOUtils;
import org.hamcrest.CoreMatchers;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.mockito.ArgumentMatcher;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;

@Feature("Application")
@Story("DefaultApplication provides the list of extension models, class loader, application model and remote application context")
@RunWith(MockitoJUnitRunner.class)
public class DefaultApplicationTestCase {

  private static final String TOOLING_APP_ID = "toolingAppId";
  private static final String APPLICATIONS_DATASENSE_STATIC = "applications/datasense-static";

  @Mock
  private RuntimeToolingService runtimeToolingService;

  @Mock
  private ToolingArtifactContext context;

  @Mock
  private ToolingArtifactClassLoader applicationClassLoader;

  @Mock
  private ExtensionModel runtimeExtensionModel;

  @Mock
  private ExtensionModel extensionModel;

  @Mock
  private ApplicationClassLoaderFactory applicationClassLoaderFactory;

  @Mock
  private MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;

  @Rule
  public WireMockClassRule wireMockRule = new WireMockClassRule(wireMockConfig().dynamicPort());

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Rule
  public ExpectedException expectedException = none();

  private File artifactFile;

  private final String REQUEST_PATH = "/app";

  private Map<String, String> properties = ImmutableMap.of("key", "value");

  @Before
  public void before() {
    artifactFile = toFile(this.getClass().getClassLoader().getResource(APPLICATIONS_DATASENSE_STATIC));

    when(context.getRuntimeToolingService()).thenReturn(runtimeToolingService);
  }

  private ApplicationDescriptor createApplicationDescriptor(File applicationLocation) throws IOException {
    return mock(ApplicationDescriptor.class);
  }

  @Test
  @Description("Validates that when application is created from a remote content it will call the deployment with the inputStream")
  public void deploymentRemoteApplicationUrlContentShouldBeByInputStream() throws Exception {
    File jarFile = temporaryFolder.newFile();
    compress(jarFile, artifactFile);

    wireMockRule.stubFor(
                         get(urlEqualTo("/app"))
                             .willReturn(aResponse()
                                 .withStatus(200)
                                 .withHeader("Content-Type", "application/octet-stream")
                                 .withBody(IOUtils.toByteArray(jarFile.toURI().toURL()))));
    int port = wireMockRule.port();

    URL applicationContentUrl = new URI("http://localhost:" + port + REQUEST_PATH).toURL();
    when(runtimeToolingService.deployApplication(eq(TOOLING_APP_ID), any(InputStream.class), eq(properties)))
        .thenReturn(TOOLING_APP_ID);

    DefaultApplication application =
        new DefaultApplication(TOOLING_APP_ID, new ArtifactResources(TOOLING_APP_ID, applicationContentUrl),
                               createApplicationDescriptor(artifactFile), null, context, properties, false);
    String applicationId =
        application.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(TOOLING_APP_ID));

    verify(runtimeToolingService).deployApplication(eq(TOOLING_APP_ID), any(InputStream.class), eq(properties));
  }

  @Test
  @Description("Validates that when application is created from a local content it will call the deployment with the path")
  public void deploymentLocalApplicationUrlContentShouldBeByPath() throws Exception {
    when(runtimeToolingService.deployApplication(eq(TOOLING_APP_ID), any(File.class), eq(properties))).thenReturn(TOOLING_APP_ID);

    DefaultApplication application =
        new DefaultApplication(TOOLING_APP_ID, new ArtifactResources(TOOLING_APP_ID, artifactFile.toURI().toURL()),
                               createApplicationDescriptor(artifactFile), null, context, properties, false);
    String applicationId =
        application.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(TOOLING_APP_ID));

    verify(runtimeToolingService).deployApplication(eq(TOOLING_APP_ID), any(File.class), eq(properties));
  }

  @Test
  @Description("Validates that if a remote application was deployed it will be disposed when application is disposed")
  public void disposeRemoteApplicationIfDeployed() throws Exception {
    when(runtimeToolingService.deployApplication(eq(TOOLING_APP_ID), any(File.class), eq(properties))).thenReturn(TOOLING_APP_ID);

    DefaultApplication application =
        new DefaultApplication(TOOLING_APP_ID, new ArtifactResources(TOOLING_APP_ID, artifactFile.toURI().toURL()),
                               createApplicationDescriptor(artifactFile), null, context, properties, false);
    String applicationId =
        application.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(TOOLING_APP_ID));

    application.dispose();

    verify(runtimeToolingService).deployApplication(eq(TOOLING_APP_ID), any(File.class), eq(properties));
    verify(runtimeToolingService).disposeApplication(TOOLING_APP_ID);
  }

  @Test
  @Description("Validates that if a remote application was not deployed when application is disposed the runtimeToolingService should not be called")
  public void doNotDisposeRemoteApplication() throws Exception {
    DefaultApplication application =
        new DefaultApplication(TOOLING_APP_ID, new ArtifactResources(TOOLING_APP_ID, artifactFile.toURI().toURL()),
                               createApplicationDescriptor(artifactFile), null, context, properties, false);
    verifyNoMoreInteractions(runtimeToolingService);
    application.dispose();
  }

  @Test
  @Description("Validates that class loader for application should be closed once the application is disposed")
  public void disposeInvalidatesClassLoader() throws Exception {
    when(context.getApplicationClassLoaderFactory()).thenReturn(applicationClassLoaderFactory);
    when(applicationClassLoaderFactory.createApplicationClassLoader(any(ApplicationDescriptor.class), any(File.class),
                                                                    isNull(ToolingArtifactClassLoader.class)))
                                                                        .thenReturn(applicationClassLoader);

    DefaultApplication application =
        new DefaultApplication(TOOLING_APP_ID, new ArtifactResources(TOOLING_APP_ID, artifactFile.toURI().toURL()),
                               createApplicationDescriptor(artifactFile), null, context, properties, false);

    assertThat(application.getArtifactClassLoader(), sameInstance(applicationClassLoader));
    application.dispose();

    verify(applicationClassLoaderFactory).createApplicationClassLoader(any(ApplicationDescriptor.class), any(File.class),
                                                                       isNull(ToolingArtifactClassLoader.class));
    verify(applicationClassLoader).dispose();
  }

  @Test
  @Description("Validates that when an application is fetched only the remote application is disposed if the URL has changed")
  public void disposeRemoteApplicationIfToolingUrlChanges() throws Exception {
    when(runtimeToolingService.deployApplication(eq(TOOLING_APP_ID), eq(artifactFile), eq(properties)))
        .thenReturn(TOOLING_APP_ID);

    DefaultApplication application =
        new DefaultApplication(TOOLING_APP_ID, new ArtifactResources(TOOLING_APP_ID, artifactFile.toURI().toURL()),
                               createApplicationDescriptor(artifactFile), null, context, properties, false);
    String applicationId =
        application.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(TOOLING_APP_ID));

    AgentConfiguration agentConfiguration = mock(AgentConfiguration.class);
    URL originalUrl = new URL("http://localhost:9991/mule");
    when(agentConfiguration.getToolingApiUrl()).thenReturn(originalUrl);
    Optional<AgentConfiguration> agentConfigurationOptional = of(agentConfiguration);
    when(context.getAgentConfiguration()).thenReturn(agentConfigurationOptional);

    AgentConfiguration newAgentConfiguration = mock(AgentConfiguration.class);
    URL newUrl = new URL("http://localhost:9992/mule");
    when(newAgentConfiguration.getToolingApiUrl()).thenReturn(newUrl);
    Optional<AgentConfiguration> newAgentConfigurationOptional = of(newAgentConfiguration);
    ToolingArtifactContext newContext = mock(ToolingArtifactContext.class);
    when(newContext.getAgentConfiguration()).thenReturn(newAgentConfigurationOptional);

    application.setContext(newContext);

    String otherAppId = "otherAppId";
    when(runtimeToolingService.deployApplication(eq(otherAppId), eq(artifactFile), eq(properties))).thenReturn(otherAppId);
    DefaultApplication otherApplication =
        new DefaultApplication(otherAppId, new ArtifactResources(otherAppId, artifactFile.toURI().toURL()),
                               createApplicationDescriptor(artifactFile), null, context, properties, false);
    applicationId =
        otherApplication.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(otherAppId));

    verify(runtimeToolingService).disposeApplication(TOOLING_APP_ID);
    verify(runtimeToolingService, times(1)).deployApplication(eq(TOOLING_APP_ID), eq(artifactFile), eq(properties));
    verify(runtimeToolingService, times(1)).deployApplication(eq(otherAppId), eq(artifactFile), eq(properties));
  }

  @Test
  @Description("Validates that extension models are loaded for an application when applicationUrlContent points to an exploded content")
  public void getDependenciesExplodedApplicationUrl() throws Exception {
    when(context.getApplicationClassLoaderFactory()).thenReturn(applicationClassLoaderFactory);

    when(context.getMuleRuntimeExtensionModelProvider()).thenReturn(muleRuntimeExtensionModelProvider);
    when(muleRuntimeExtensionModelProvider.getRuntimeExtensionModels()).thenReturn(emptyList());

    List<BundleDependency> pluginDependenciesResolved = Lists.newArrayList();
    pluginDependenciesResolved.add(new BundleDependency.Builder()
        .sedBundleDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.test")
            .setArtifactId("plugin-test")
            .setClassifier("mule-plugin")
            .setVersion("1.0")
            .setBaseVersion("1.0")
            .build())
        .build());
    pluginDependenciesResolved.add(new BundleDependency.Builder()
        .sedBundleDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.test")
            .setArtifactId("plugin-dependency")
            .setVersion("1.0")
            .setBaseVersion("1.0")
            .build())
        .build());

    List<ArtifactClassLoader> artifactPluginClassLoaders = new ArrayList<>();

    ArtifactClassLoader pluginTestClassLoader = mock(ArtifactClassLoader.class);
    ArtifactDescriptor pluginTestArtifactDescriptor = mock(ArtifactDescriptor.class);
    when(pluginTestArtifactDescriptor.getBundleDescriptor())
        .thenReturn(toBundleDescriptor(pluginDependenciesResolved.get(0).getDescriptor()));
    URI pluginBundleUri = temporaryFolder.newFolder().toURI();
    when(pluginTestClassLoader.getArtifactDescriptor()).thenReturn(pluginTestArtifactDescriptor);
    artifactPluginClassLoaders.add(pluginTestClassLoader);

    ToolingArtifactClassLoader toolingArtifactClassLoader = mock(ToolingArtifactClassLoader.class);
    when(toolingArtifactClassLoader.getArtifactPluginClassLoaders()).thenReturn(artifactPluginClassLoaders);
    when(applicationClassLoaderFactory.createApplicationClassLoader(any(ApplicationDescriptor.class), any(File.class),
                                                                    isNull(ToolingArtifactClassLoader.class)))
                                                                        .thenReturn(toolingArtifactClassLoader);
    when(context.getApplicationClassLoaderFactory()).thenReturn(applicationClassLoaderFactory);
    when(context.getMuleRuntimeExtensionModelProvider()).thenReturn(muleRuntimeExtensionModelProvider);
    when(muleRuntimeExtensionModelProvider.getRuntimeExtensionModels()).thenReturn(Lists.newArrayList(runtimeExtensionModel));
    when(muleRuntimeExtensionModelProvider.loadExtensionModels(any())).thenAnswer(i -> {
      List<ArtifactClassLoader> inputClassLoaders = i.getArgument(0);
      return inputClassLoaders
          .stream()
          .filter(acl -> acl.getArtifactDescriptor().getBundleDescriptor()
              .equals(pluginTestArtifactDescriptor.getBundleDescriptor()))
          .map(cl -> extensionModel)
          .collect(toSet());
    });

    DefaultApplication application =
        new DefaultApplication(TOOLING_APP_ID, new ArtifactResources(TOOLING_APP_ID, artifactFile.toURI().toURL()),
                               createApplicationDescriptor(artifactFile), null, context, properties, false);

    final List<ExtensionModel> extensionModels = application.getExtensionModels();

    assertThat(extensionModels, hasSize(2));
    assertThat(extensionModels.get(0), CoreMatchers.sameInstance(extensionModel));
    assertThat(extensionModels.get(1), CoreMatchers.sameInstance(runtimeExtensionModel));

    verify(applicationClassLoaderFactory).createApplicationClassLoader(any(ApplicationDescriptor.class), any(File.class),
                                                                       isNull(ToolingArtifactClassLoader.class));
    verify(muleRuntimeExtensionModelProvider).loadExtensionModels(
                                                                  argThat(list -> list.stream()
                                                                      .anyMatch(acl -> acl.getArtifactDescriptor()
                                                                          .getBundleDescriptor()
                                                                          .equals(pluginTestArtifactDescriptor
                                                                              .getBundleDescriptor()))));
  }

  private org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor toBundleDescriptor(BundleDescriptor descriptor) {
    return new org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor.Builder()
        .setGroupId(descriptor.getGroupId())
        .setArtifactId(descriptor.getArtifactId())
        .setVersion(descriptor.getVersion())
        .setBaseVersion(descriptor.getBaseVersion())
        .setType(descriptor.getType())
        .setClassifier(descriptor.getClassifier().orElse(null))
        .build();
  }

}
