/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.hamcrest;

import static java.util.Optional.empty;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.mule.tooling.client.internal.hamcrest.HamcrestUtils.validateThat;

import org.mule.runtime.extension.api.property.DefaultImplementingTypeModelProperty;
import org.mule.runtime.extension.api.property.InfrastructureParameterModelProperty;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;
import org.mule.runtime.extension.api.property.QNameModelProperty;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;

import java.util.Collection;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;
import org.hamcrest.core.IsEqual;

public class ParameterModelMatcher extends TypeSafeDiagnosingMatcher<ParameterModel> {

  public static Matcher<ParameterModel>[] sFrom(Collection<org.mule.runtime.api.meta.model.parameter.ParameterModel> runtimeParameterModels) {
    return runtimeParameterModels.stream().map(ParameterModelMatcher::from).toArray(ParameterModelMatcher[]::new);
  }

  public static Matcher<ParameterModel> from(org.mule.runtime.api.meta.model.parameter.ParameterModel runtimeParameterModel) {
    if (runtimeParameterModel == null) {
      return nullValue(ParameterModel.class);
    }
    return new ParameterModelMatcher(runtimeParameterModel);
  }

  private final org.mule.runtime.api.meta.model.parameter.ParameterModel runtimeParameterModel;

  private ParameterModelMatcher(org.mule.runtime.api.meta.model.parameter.ParameterModel runtimeParameterModel) {
    this.runtimeParameterModel = runtimeParameterModel;
  }

  @Override
  protected boolean matchesSafely(ParameterModel parameterModel, Description description) {
    return validateThat("isRequired", parameterModel.isRequired(), equalTo(runtimeParameterModel.isRequired()), description) &&
        validateThat("name", parameterModel.getName(), equalTo(runtimeParameterModel.getName()), description) &&
        validateThat("description", parameterModel.getDescription(), equalTo(runtimeParameterModel.getDescription()), description)
        &&
        runtimeParameterModel
            .getDisplayModel()
            .map(rdp -> parameterModel.getDisplayModel()
                .map(dm -> validateThat("displayModel", dm, DisplayModelMatcher.from(rdp), description))
                .orElse(false))
            .orElseGet(() -> validateThat("displayModel", parameterModel.getDisplayModel(), equalTo(empty()), description))
        &&
        validateThat("type", parameterModel.getType(), equalTo(runtimeParameterModel.getType()), description) &&
        validateThat("hasDynamicType", parameterModel.hasDynamicType(), equalTo(runtimeParameterModel.hasDynamicType()),
                     description)
        &&
        validateThat("isOverrideFromConfig", parameterModel.isOverrideFromConfig(),
                     equalTo(runtimeParameterModel.isOverrideFromConfig()), description)
        &&
        validateThat("expressionSupport", parameterModel.getExpressionSupport(),
                     ExpressionSupportMatcher.from(runtimeParameterModel.getExpressionSupport()), description)
        &&
        validateThat("defaultValue", parameterModel.getDefaultValue(), equalTo(runtimeParameterModel.getDefaultValue()),
                     description)
        &&
        validateThat("role", parameterModel.getRole(), ParameterRoleMatcher.from(runtimeParameterModel.getRole()), description) &&
        validateThat("dslConfiguration", parameterModel.getDslConfiguration(),
                     ParameterDslConfigurationMatcher.from(runtimeParameterModel.getDslConfiguration()), description)
        &&
        runtimeParameterModel
            .getLayoutModel()
            .map(rlm -> parameterModel.getLayoutModel()
                .map(lm -> validateThat("layoutModel", lm, LayoutModelMatcher.from(rlm), description))
                .orElse(false))
            .orElseGet(() -> validateThat("layoutModel", parameterModel.getLayoutModel(), equalTo(empty()), description))
        &&
        runtimeParameterModel
            .getValueProviderModel()
            .map(rvpm -> parameterModel.getValueProviderModel()
                .map(vpm -> validateThat("valueProviderModel", vpm, ValueProviderMatcher.from(rvpm), description))
                .orElse(false))
            .orElseGet(() -> validateThat("valueProviderModel", parameterModel.getValueProviderModel(), equalTo(empty()),
                                          description))
        &&

        (runtimeParameterModel.getAllowedStereotypes().isEmpty()
            ? validateThat("allowedStereotypes", parameterModel.getAllowedStereotypes(), hasSize(0), description)
            : validateThat("allowedStereotypes",
                           parameterModel.getAllowedStereotypes(),
                           contains(StereotypeMatcher.sFrom(runtimeParameterModel.getAllowedStereotypes())),
                           description))
        &&

        (!parameterModel.getDeprecationModel().isEnabled() ||
            runtimeParameterModel.getDeprecationModel()
                .map(rdm -> validateThat("deprecationModel", parameterModel.getDeprecationModel().get(),
                                         DeprecationModelMatcher.from(rdm),
                                         description))
                .orElseGet(() -> validateThat("deprecationModel", parameterModel.getDeprecationModel().get(), nullValue(),
                                              description)))
        &&

        (!parameterModel.isComponentId().isEnabled() ||
            validateThat("isComponentId", parameterModel.isComponentId().get(), equalTo(runtimeParameterModel.isComponentId()),
                         description))
        &&

        (runtimeParameterModel.getFieldValueProviderModels().isEmpty()
            ? validateThat("fieldValueProviderModels", parameterModel.getFieldValueProviderModels(), hasSize(0), description)
            : validateThat("fieldValueProviderModels",
                           parameterModel.getFieldValueProviderModels(),
                           contains(FieldValueProviderMatcher.sFrom(runtimeParameterModel.getFieldValueProviderModels())),
                           description))
        &&

        runtimeParameterModel
            .getModelProperty(MetadataKeyPartModelProperty.class)
            .map(mp -> parameterModel.getMetadataKeyPartModel()
                .map(m -> validateThat("metadataKeyPartModel", m, MetadataKeyPartMatcher.from(mp), description))
                .orElse(false))
            .orElseGet(() -> validateThat("metadataKeyPartModel", parameterModel.getMetadataKeyPartModel(), equalTo(empty()),
                                          description))
        &&

        runtimeParameterModel
            .getModelProperty(QNameModelProperty.class)
            .map(qmp -> parameterModel.getQNameModel()
                .map(q -> validateThat("qName", q.getValue(), equalTo(qmp.getValue()), description))
                .orElse(false))
            .orElseGet(() -> validateThat("qName", parameterModel.getQNameModel(), equalTo(empty()), description))
        &&

        runtimeParameterModel
            .getModelProperty(InfrastructureParameterModelProperty.class)
            .map(imp -> parameterModel.getInfrastructureParameterModel()
                .map(i -> validateThat("infrastructureParameterModelSequence", i.getSequence(), equalTo(imp.getSequence()),
                                       description))
                .orElse(false))
            .orElseGet(() -> validateThat("infrastructureParameterModelSequence",
                                          parameterModel.getInfrastructureParameterModel(), equalTo(empty()), description))
        &&

        runtimeParameterModel
            .getModelProperty(DefaultImplementingTypeModelProperty.class)
            .map(dmp -> parameterModel.getDefaultImplementingTypeModel()
                .map(d -> validateThat("defaultImplementingType", d.value(), equalTo(dmp.value()), description))
                .orElse(false))
            .orElseGet(() -> validateThat("defaultImplementingType", parameterModel.getDefaultImplementingTypeModel(),
                                          equalTo(empty()), description))
        &&

        validateThat("semanticTerms", parameterModel.getSemanticTerms(),
                     IsEqual.equalTo(runtimeParameterModel.getSemanticTerms()), description);
  }

  @Override
  public void describeTo(Description description) {
    description.appendText("ParameterModel: ").appendValue(runtimeParameterModel);
  }

}
