/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.util.Arrays.asList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newListValue;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newParameterGroup;
import static org.mule.runtime.config.api.dsl.ArtifactDeclarationUtils.toArtifactast;
import static org.mule.runtime.core.api.extension.MuleExtensionModelProvider.MULE_NAME;
import static org.mule.runtime.internal.dsl.DslConstants.FLOW_ELEMENT_IDENTIFIER;

import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.app.declaration.api.fluent.ElementDeclarer;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.config.dsl.model.AbstractDslModelTestCase;
import org.mule.runtime.core.internal.locator.ComponentLocator;
import org.mule.runtime.deployment.model.api.artifact.extension.ExtensionModelDiscoverer;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.Mockito;

public class ToolingMetadataCacheIdGeneratorTestCase extends AbstractDslModelTestCase {

  private final Set<ExtensionModel> runtimeExtensionModels = new ExtensionModelDiscoverer().discoverRuntimeExtensionModels();

  private static final String MY_CONFIG = "myConfig";
  private static final String MY_FLOW = "myFlow";
  private static final Location OPERATION_LOCATION =
      Location.builder().globalName(MY_FLOW).addProcessorsPart().addIndexPart(0).build();

  private ElementDeclarer declarer;
  private Set<ExtensionModel> extensions;
  private DslResolvingContext dslResolvingContext;

  private ExtensionModel mockExtension;

  @Before
  public void setUp() {
    extensions = new HashSet<>();
    mockExtension = mock(ExtensionModel.class);
    when(mockExtension.getName()).thenReturn(EXTENSION_NAME);
    when(mockExtension.getXmlDslModel()).thenReturn(XmlDslModel.builder()
        .setXsdFileName("mule-mockns.xsd")
        .setPrefix(NAMESPACE)
        .setNamespace(NAMESPACE_URI)
        .setSchemaLocation(SCHEMA_LOCATION)
        .setSchemaVersion("4.0")
        .build());
    when(mockExtension.getConfigurationModels()).thenReturn(asList(configuration));
    when(mockExtension.getOperationModels()).thenReturn(asList(operation, anotherOperation));
    when(mockExtension.getConnectionProviders()).thenReturn(asList(connectionProvider));
    when(mockExtension.getSourceModels()).thenReturn(asList(source));

    extensions.add(mockExtension);
    extensions.addAll(runtimeExtensionModels);

    dslResolvingContext = DslResolvingContext.getDefault(extensions);
    declarer = ElementDeclarer.forExtension(EXTENSION_NAME);
  }

  @Test
  public void differentMajorGivesDifferentIds() throws Exception {
    ArtifactDeclaration declaration = getBaseApp();
    final String version1 = "1.0.0";
    when(mockExtension.getVersion()).thenReturn(version1);
    String cacheId1 = getIdForOperation(declaration);
    final String version2 = "2.0.0";
    when(mockExtension.getVersion()).thenReturn(version2);
    String cacheId2 = getIdForOperation(declaration);
    assertThat(cacheId1, is(not(cacheId2)));
  }

  @Test
  public void sameMajorGivesSameId() throws Exception {
    ArtifactDeclaration declaration = getBaseApp();
    final String version1 = "1.0.0";
    when(mockExtension.getVersion()).thenReturn(version1);
    String cacheId1 = getIdForOperation(declaration);
    final String version2 = "1.5.0";
    when(mockExtension.getVersion()).thenReturn(version2);
    String cacheId2 = getIdForOperation(declaration);
    assertThat(cacheId1, is(cacheId2));
  }

  @Test
  public void malformedVersionGivesDelegateId() throws Exception {
    ArtifactDeclaration declaration = getBaseApp();
    final String version = "this is not a valid semver format";
    when(mockExtension.getVersion()).thenReturn(version);
    getIdForOperation(declaration); // does not fail
  }

  private String getIdForOperation(ArtifactDeclaration declaration) throws Exception {
    ArtifactAst app = toArtifactast(declaration, extensions);
    ComponentAst component = new Locator(app)
        .get(OPERATION_LOCATION)
        .get();
    return createGenerator(app).getIdForComponentMetadata(component).get();
  }

  private DefaultToolingCacheIdGenerator createGenerator(ArtifactAst app) {
    return new DefaultToolingCacheIdGenerator(dslResolvingContext, new Locator(app));
  }

  private ArtifactDeclaration getBaseApp() {
    return ElementDeclarer.newArtifact()
        .withGlobalElement(declarer.newConfiguration(CONFIGURATION_NAME)
            .withRefName(MY_CONFIG)
            .withParameterGroup(newParameterGroup()
                .withParameter(CONTENT_NAME, CONTENT_VALUE)
                .withParameter(BEHAVIOUR_NAME, BEHAVIOUR_VALUE)
                .withParameter(LIST_NAME, newListValue().withValue(ITEM_VALUE).build())
                .getDeclaration())
            .withConnection(declarer.newConnection(CONNECTION_PROVIDER_NAME)
                .withParameterGroup(newParameterGroup()
                    .withParameter(CONTENT_NAME, CONTENT_VALUE)
                    .withParameter(BEHAVIOUR_NAME, BEHAVIOUR_VALUE)
                    .withParameter(LIST_NAME,
                                   newListValue().withValue(ITEM_VALUE).build())
                    .getDeclaration())
                .getDeclaration())
            .getDeclaration())
        .withGlobalElement(ElementDeclarer.forExtension(MULE_NAME)
            .newConstruct(FLOW_ELEMENT_IDENTIFIER)
            .withRefName(MY_FLOW)
            .withComponent(
                           declarer.newOperation(OPERATION_NAME)
                               .withConfig(MY_CONFIG)
                               .withParameterGroup(g -> g.withParameter(CONTENT_NAME, "nonKey"))
                               .getDeclaration())
            .getDeclaration())
        .getDeclaration();
  }

  private static class Locator implements ComponentLocator<ComponentAst> {

    private final Map<Location, ComponentAst> components = new HashMap<>();

    Locator(ArtifactAst app) {
      app.topLevelComponentsStream().forEach(this::addComponent);
    }

    @Override
    public Optional<ComponentAst> get(Location location) {
      return Optional.ofNullable(components.get(location));
    }

    private Location getLocation(ComponentAst component) {
      return Location.builderFromStringRepresentation(component.getLocation().getLocation()).build();
    }

    private void addComponent(ComponentAst component) {
      components.put(getLocation(component), component);
      component.directChildrenStream().forEach(this::addComponent);
    }

  }
}
