/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.Optional.ofNullable;
import static java.util.function.UnaryOperator.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.Stream.concat;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.ComponentLocationFactory.toComponentIdentifierDTO;
import static org.mule.tooling.client.internal.ComponentLocationFactory.toComponentLocationDTO;
import static org.mule.tooling.client.internal.ComponentLocationFactory.toComponentTypeToDTO;
import static org.mule.tooling.client.internal.ComponentLocationFactory.toSourceCodeLocationDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toParameterGroupModelsDTO;
import static org.mule.tooling.client.internal.dsl.DefaultDslSyntaxResolverService.toDslDto;

import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.runtime.core.internal.locator.ComponentLocator;
import org.mule.runtime.extension.api.values.ValueProvider;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.ast.ComponentGenerationInformation;
import org.mule.tooling.client.api.artifact.resources.ResourceLoader;
import org.mule.tooling.client.api.component.location.ComponentLocationService;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingService;
import org.mule.tooling.client.api.datasense.DataSenseService;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.ParameterizedModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.metadata.MetadataService;
import org.mule.tooling.client.api.sampledata.SampleDataService;
import org.mule.tooling.client.api.value.provider.ValueProviderService;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.Artifact;
import org.mule.tooling.client.internal.application.Domain;
import org.mule.tooling.client.internal.artifact.DefaultResourceLoader;
import org.mule.tooling.client.internal.datasense.DataSenseArtifact;
import org.mule.tooling.client.internal.dataweave.ApplicationRemoteRunner;
import org.mule.tooling.client.internal.dataweave.DataWeaveRunner;
import org.mule.tooling.client.internal.dataweave.DefaultDataWeaveService;
import org.mule.tooling.client.internal.dataweave.DomainRemoteRunner;
import org.mule.tooling.client.internal.dataweave.ModulesAnalyzer;
import org.mule.tooling.client.internal.metadata.MetadataCache;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.service.ServiceRegistry;
import org.mule.tooling.client.internal.values.ValueProviderCache;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * Implementation that allows to do Tooling operations without deploying the application until it is needed.
 * <p/>
 * If an operation requires the application to be deployed it will deploy it first and then delegate the operation to the default
 * implementation of {@link ToolingArtifact}.
 *
 * @since 4.0
 */
public class DefaultToolingArtifact implements ToolingArtifact, Command {

  private final String id;
  private final Artifact artifact;
  private final Serializer serializer;
  private final Optional<MuleVersion> targetRuntimeVersion;

  private final ConnectivityTestingService connectivityTestingService;
  private final MetadataService metadataService;
  private final DataSenseService dataSenseService;
  private final DataWeaveService dataWeaveService;
  private final ValueProviderService valueProviderService;
  private final Feature<SampleDataService> sampleDataService;
  private final ComponentLocationService componentLocationService;

  private ToolingArtifact parentToolingArtifact;

  /**
   * Creates an instance of the {@link DefaultToolingArtifact} from a fetched applicationId or deploys the application to obtain
   * an identifier in case if null.
   *
   * @param id                  {@link String} identifier for this {@link ToolingArtifact}. Non null.
   * @param application         {@link Application} to handle state, local and remote (Mule Runtime) in order to resolve
   *                            operations. Non null.
   * @param serializer          {@link Serializer} to serialize parameters and results when called from Tooling API. Non null.
   * @param metadataCache       {@link MetadataCache} to be used by this {@link ToolingArtifact}. Non null.
   * @param parentMetadataCache {@link MetadataCache} to be used by the parent {@link ToolingArtifact}. Can be null.
   * @param valueProvidersCache {@link ValueProviderCache} to be used when resolving values with {@link ValueProvider}s
   */
  public DefaultToolingArtifact(String id,
                                Application application,
                                ToolingArtifact parentToolingArtifact,
                                Serializer serializer,
                                ServiceRegistry serviceRegistry,
                                LazyValue<MetadataCache> metadataCache,
                                LazyValue<MetadataCache> parentMetadataCache,
                                LazyValue<DslResolvingContext> dslResolvingContext,
                                LazyValue<ComponentLocator<ComponentAst>> componentLocator,
                                LazyValue<ValueProviderCache> valueProvidersCache,
                                Optional<MuleVersion> targetRuntimeVersion) {
    checkNotNull(id, "id cannot be null");
    checkNotNull(application, "application cannot be null");
    checkNotNull(serializer, "serializer cannot be null");
    checkNotNull(serviceRegistry, "serviceRegistry cannot be null");
    checkNotNull(metadataCache, "metadataCache cannot be null");
    checkNotNull(dslResolvingContext, "dslResolvingContext cannot be null");
    checkNotNull(componentLocator, "component locator cannot be null");

    this.id = id;
    this.artifact = application;
    this.parentToolingArtifact = parentToolingArtifact;
    this.serializer = serializer;
    this.targetRuntimeVersion = targetRuntimeVersion;

    // Services registration...
    MetadataProvider metadataProvider = new InternalApplicationMetadataProvider(application);

    DataWeaveRunner remoteRunner = new ApplicationRemoteRunner(application);
    this.connectivityTestingService = new ApplicationConnectivityTestingService(application, serializer);
    this.metadataService =
        new ToolingMetadataServiceAdapter(new LazyValue<>(this::getAst),
                                          metadataProvider,
                                          metadataCache,
                                          serializer,
                                          dslResolvingContext,
                                          componentLocator);

    DataSenseArtifact parentDataSenseArtifact = application.getDomain()
        .map(domain -> new DataSenseArtifact(domain, parentMetadataCache, new InternalDomainMetadataProvider(domain),
                                             domain.getProperties(), serviceRegistry.getExpressionLanguageMetadataService(),
                                             serviceRegistry.getApikitMetadataService(),
                                             null))
        .orElse(null);
    DataSenseArtifact dataSenseArtifact =
        new DataSenseArtifact(application,
                              metadataCache,
                              metadataProvider,
                              application.getProperties(),
                              serviceRegistry.getExpressionLanguageMetadataService(),
                              serviceRegistry.getApikitMetadataService(),
                              parentDataSenseArtifact);


    this.dataSenseService = new DefaultDataSenseService(dataSenseArtifact, serializer);

    this.valueProviderService =
        new ApplicationValueProviderService(application, () -> artifact.getExtensionModels(), serializer, valueProvidersCache,
                                            componentLocator);

    this.sampleDataService = enabled(new SampleDataProviderServiceAdapter(application, serializer));

    this.dataWeaveService =
        new DefaultDataWeaveService(new LazyValue<>(() -> artifact.getArtifactClassLoader().getClassLoader()),
                                    remoteRunner,
                                    serviceRegistry.getExpressionLanguageMetadataService(),
                                    new ModulesAnalyzer(serviceRegistry.getExpressionLanguageCapabilitiesService()), serializer);
    this.componentLocationService = new DefaultComponentLocationService(new LazyValue<>(this::getAst), serializer);
  }

  public DefaultToolingArtifact(String id,
                                Domain domain,
                                Serializer serializer,
                                ServiceRegistry serviceRegistry,
                                LazyValue<MetadataCache> metadataCache,
                                LazyValue<DslResolvingContext> dslResolvingContext,
                                LazyValue<ComponentLocator<ComponentAst>> componentLocator,
                                LazyValue<ValueProviderCache> valueProvidersCache,
                                Optional<MuleVersion> targetRuntimeVersion) {
    checkNotNull(id, "id cannot be null");
    checkNotNull(domain, "domain cannot be null");
    checkNotNull(serializer, "serializer cannot be null");
    checkNotNull(serviceRegistry, "serviceRegistry cannot be null");
    checkNotNull(metadataCache, "metadataCache cannot be null");
    checkNotNull(dslResolvingContext, "dslResolvingContext cannot be null");
    checkNotNull(componentLocator, "component locator cannot be null");
    checkNotNull(valueProvidersCache, "value providers cache cannot be null");

    this.id = id;
    this.artifact = domain;
    this.serializer = serializer;
    this.targetRuntimeVersion = targetRuntimeVersion;

    // Services registration...
    MetadataProvider metadataProvider = new InternalDomainMetadataProvider(domain);
    DataWeaveRunner remoteRunner = new DomainRemoteRunner(domain);
    this.connectivityTestingService = new DomainConnectivityTestingService(domain, serializer);
    this.dataSenseService = new UnavailableDataSenseService(ArtifactType.DOMAIN);
    this.metadataService =
        new ToolingMetadataServiceAdapter(new LazyValue<>(this::getAst),
                                          metadataProvider,
                                          metadataCache,
                                          serializer,
                                          dslResolvingContext,
                                          componentLocator);
    this.valueProviderService = new DomainValueProviderService(domain, () -> artifact.getExtensionModels(), serializer,
                                                               valueProvidersCache, componentLocator);
    this.sampleDataService = disabled(); // SampleData is disabled for domains.

    this.dataWeaveService =
        new DefaultDataWeaveService(new LazyValue<>(() -> artifact.getArtifactClassLoader().getClassLoader()),
                                    remoteRunner,
                                    serviceRegistry.getExpressionLanguageMetadataService(),
                                    new ModulesAnalyzer(serviceRegistry.getExpressionLanguageCapabilitiesService()), serializer);

    this.componentLocationService = new DefaultComponentLocationService(new LazyValue<>(this::getAst), serializer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getId() {
    return id;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ToolingArtifact> getParent() {
    return ofNullable(parentToolingArtifact);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, String> getProperties() {
    return artifact.getProperties();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivityTestingService connectivityTestingService() {
    return this.connectivityTestingService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataService metadataService() {
    return this.metadataService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataSenseService dataSenseService() {
    return this.dataSenseService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeaveService dataWeaveService() {
    return dataWeaveService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValueProviderService valueProviderService() {
    return valueProviderService;
  }

  @Override
  public Feature<SampleDataService> sampleDataService() {
    return sampleDataService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Feature<ResourceLoader> getResourceLoader() {
    return enabled(new DefaultResourceLoader(artifact, serializer));
  }

  @Override
  public ComponentLocationService componentLocationService() {
    return componentLocationService;
  }

  @Override
  public org.mule.tooling.client.api.artifact.ast.ArtifactAst getArtifactAst() {
    final ArtifactAst ast = getAst();
    try {
      return toArtifactAstDto(ast);
    } catch (Exception e) {
      throw new ToolingException(format("Error while creating a DTO from the ArtifactAst of artifact %s", id),
                                 e);
    }
  }

  private org.mule.tooling.client.api.artifact.ast.ArtifactAst toArtifactAstDto(ArtifactAst ast) {
    final Map<String, ExtensionModel> dependencies = ast.dependencies().stream()
        .map(em ->
        // this factory keeps state that may cause items from different extensions with the same names to conflict
        new ExtensionModelFactory(targetRuntimeVersion).createExtensionModel(em, artifact.getMinMuleVersion().toString()))
        .collect(toMap(em -> em.getName(), identity()));

    return new org.mule.tooling.client.api.artifact.ast.ArtifactAst(ast.getParent().map(this::toArtifactAstDto).orElse(null),
                                                                    new HashSet<>(dependencies.values()),
                                                                    toErrorTypeRepositoryDto(ast.getErrorTypeRepository()),
                                                                    ast.topLevelComponentsStream()
                                                                        .map(comp -> toComponentAstDto(comp, dependencies))
                                                                        .collect(toList()));
  }

  private org.mule.tooling.client.api.error.ErrorTypeRepository toErrorTypeRepositoryDto(ErrorTypeRepository errorTypesRepo) {
    final Set<org.mule.tooling.client.api.error.ErrorType> errorTypes = errorTypesRepo.getErrorTypes()
        .stream()
        .map(DefaultToolingArtifact::toErrorTypeDTO)
        .collect(toSet());
    final Set<org.mule.tooling.client.api.error.ErrorType> internalErrorTypes = errorTypesRepo.getInternalErrorTypes()
        .stream()
        .map(DefaultToolingArtifact::toErrorTypeDTO)
        .collect(toSet());

    return new org.mule.tooling.client.api.error.ErrorTypeRepository(errorTypes, internalErrorTypes,
                                                                     errorTypesRepo.getErrorNamespaces());
  }

  private static org.mule.tooling.client.api.error.ErrorType toErrorTypeDTO(ErrorType errorType) {
    if (errorType == null) {
      return null;
    }
    return new org.mule.tooling.client.api.error.ErrorType(errorType.getIdentifier(), errorType.getNamespace(),
                                                           toErrorTypeDTO(errorType.getParentErrorType()));
  }

  private org.mule.tooling.client.api.artifact.ast.ComponentAst toComponentAstDto(ComponentAst ast,
                                                                                  Map<String, ExtensionModel> dependencies) {
    final ExtensionModel extension = dependencies.get(ast.getExtensionModel().getName());
    final ParameterizedModel model = resolveModel(ast, extension);

    final List<org.mule.tooling.client.api.artifact.ast.ComponentParameterAst> params;
    if (model != null) {
      params = ast.getParameters()
          .stream()
          .map(p -> toComponentParameterAstDto(model, p, dependencies))
          .collect(toList());
    } else if (ast.getType() != null) {
      params = ast.getModel(org.mule.runtime.api.meta.model.parameter.ParameterizedModel.class).map(asPmzd -> {
        return ast.getParameters()
            .stream()
            .map(p -> toComponentParameterAstDto(new ParameterizedModel() {

              @Override
              public List<ParameterGroupModel> getParameterGroupModels() {
                return toParameterGroupModelsDTO(pm -> Optional.empty(), asPmzd.getParameterGroupModels());
              }

              @Override
              public Optional<ParameterGroupModel> getParameterGroupModel(String name) {
                return getParameterGroupModels()
                    .stream()
                    .filter(pgm -> pgm.getName().equals(name))
                    .findAny();
              }
            }, p, dependencies))
            .collect(toList());
      }).orElse(emptyList());
    } else {
      params = emptyList();
    }

    return new org.mule.tooling.client.api.artifact.ast.ComponentAst(ast.getComponentId().orElse(null),
                                                                     toComponentTypeToDTO(ast.getComponentType()),
                                                                     extension,
                                                                     new ComponentGenerationInformation(ast
                                                                         .getGenerationInformation().getSyntax()
                                                                         .map(stx -> toDslDto(stx))
                                                                         .orElse(null)),
                                                                     toComponentIdentifierDTO(ast.getIdentifier()),
                                                                     toComponentLocationDTO(ast.getLocation()),
                                                                     toSourceCodeLocationDTO(ast.getMetadata()),
                                                                     model,
                                                                     ast.getType(),
                                                                     params,
                                                                     ast.directChildrenStream()
                                                                         .map(comp -> toComponentAstDto(comp, dependencies))
                                                                         .collect(toList()));
  }

  protected ParameterizedModel resolveModel(ComponentAst ast, final ExtensionModel extension) {
    if (ast.getModel(ConfigurationModel.class).isPresent()) {
      return ast.getModel(ConfigurationModel.class)
          .flatMap(model -> extension.getConfigurationModel(model.getName())).get();
    }
    if (ast.getModel(ConnectionProviderModel.class).isPresent()) {
      return ast.getModel(ConnectionProviderModel.class)
          .flatMap(model -> concat(extension.getConnectionProviders().stream(),
                                   extension.getConfigurationModels()
                                       .stream()
                                       .flatMap(cfgModel -> cfgModel.getConnectionProviders().stream()))
                                           .filter(connModel -> connModel.getName().equals(model.getName()))
                                           .findAny())
          .get();
    }
    if (ast.getModel(ConstructModel.class).isPresent()) {
      return ast.getModel(ConstructModel.class)
          .flatMap(model -> extension.getConstructModel(model.getName())).get();
    }
    if (ast.getModel(OperationModel.class).isPresent()) {
      return ast.getModel(OperationModel.class)
          .flatMap(model -> concat(extension.getOperationModels().stream(),
                                   extension.getConfigurationModels()
                                       .stream()
                                       .flatMap(cfgModel -> cfgModel.getOperationModels().stream()))
                                           .filter(opModel -> opModel.getName().equals(model.getName()))
                                           .findAny())
          .get();
    }
    if (ast.getModel(SourceModel.class).isPresent()) {
      return ast.getModel(SourceModel.class)
          .flatMap(model -> concat(extension.getSourceModels().stream(),
                                   extension.getConfigurationModels()
                                       .stream()
                                       .flatMap(cfgModel -> cfgModel.getSourceModels().stream()))
                                           .filter(srcModel -> srcModel.getName().equals(model.getName()))
                                           .findAny())
          .get();
    }

    return null;
  }

  private org.mule.tooling.client.api.artifact.ast.ComponentParameterAst toComponentParameterAstDto(ParameterizedModel owner,
                                                                                                    ComponentParameterAst ast,
                                                                                                    Map<String, ExtensionModel> dependencies) {
    final Either<String, Object> valueDto = ast.getValue().mapRight(r -> {
      if (r instanceof ComponentAst) {
        return toComponentAstDto((ComponentAst) r, dependencies);
      } else {
        return r;
      }
    });

    // TODO MULE-19361 This will not work properly for source callbacks.
    final ParameterModel paramModel = owner.getAllParameterModels().stream()
        .filter(pm -> pm.getName().equals(ast.getModel().getName()))
        .findFirst()
        .orElse(null);

    // TODO MULE-19361 fetch the proper group/parameter models from the owner model and set them into the parameter being
    // converted below.
    return new org.mule.tooling.client.api.artifact.ast.ComponentParameterAst(paramModel,
                                                                              ofNullable(valueDto.getLeft()),
                                                                              ofNullable(valueDto.getRight()),
                                                                              ast.getRawValue(),
                                                                              ast.getResolvedRawValue(),
                                                                              ast.getMetadata()
                                                                                  .map(ComponentLocationFactory::toSourceCodeLocationDTO)
                                                                                  .orElse(null),
                                                                              new ComponentGenerationInformation(ast
                                                                                  .getGenerationInformation().getSyntax()
                                                                                  .map(stx -> toDslDto(stx))
                                                                                  .orElse(null)),
                                                                              ast.isDefaultValue());
  }

  protected ArtifactAst getAst() {
    return artifact.getApplicationModel().getMuleApplicationModel();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void dispose() {
    artifact.dispose();
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "id": {
        return serializer.serialize(this.getId());
      }
      case "getParent": {
        return this.getParent();
      }
      case "connectivityTestingService": {
        return this.connectivityTestingService();
      }
      case "metadataService": {
        return this.metadataService();
      }
      case "dataSenseService": {
        return this.dataSenseService();
      }
      case "dataWeaveService": {
        return this.dataWeaveService();
      }
      case "valueProviderService": {
        return this.valueProviderService();
      }
      case "sampleDataService": {
        return this.sampleDataService();
      }
      case "getResourceLoader": {
        return this.getResourceLoader();
      }
      case "componentLocationService": {
        return this.componentLocationService();
      }
      case "getArtifactAst": {
        return serializer.serialize(this.getArtifactAst());
      }
      case "dispose": {
        this.dispose();
        return null;
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
