/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session;

import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.mule.tooling.client.internal.utils.ExtensionModelUtils.resolveExtensionModels;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.tooling.client.api.descriptors.dependency.Dependency;
import org.mule.tooling.client.internal.MuleRuntimeExtensionModelProvider;

import com.google.common.collect.ImmutableSet;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

public class ExtensionModelProvider {

  private static final String MULE_PLUGIN = "mule-plugin";

  private final LazyValue<Map<String, ExtensionModel>> extensionModelsByName;
  private final LazyValue<Set<ExtensionModel>> extensionModelsSet;
  private final LazyValue<Set<String>> extensionModelsNameSet;


  public ExtensionModelProvider(MavenClient mavenClient,
                                MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                                List<Dependency> dependencyList) {
    extensionModelsByName = new LazyValue<>(() -> resolveExtensionModels(mavenClient,
                                                                         muleRuntimeExtensionModelProvider,
                                                                         dependencyList
                                                                             .stream()
                                                                             .filter(dependency -> MULE_PLUGIN
                                                                                 .equals(dependency.getClassifier()))
                                                                             .map(dependency -> new BundleDescriptor.Builder()
                                                                                 .setGroupId(dependency.getGroupId())
                                                                                 .setArtifactId(dependency.getArtifactId())
                                                                                 .setVersion(dependency.getVersion())
                                                                                 .setType(dependency.getType())
                                                                                 .setClassifier(dependency.getClassifier())
                                                                                 .build())
                                                                             .collect(toList())).stream()
                                                                                 .collect(toMap(ExtensionModel::getName,
                                                                                                model -> model)));

    extensionModelsSet = new LazyValue<>(() -> {
      ImmutableSet.Builder<ExtensionModel> setBuilder = ImmutableSet.builder();
      extensionModelsByName.get().values().forEach(setBuilder::add);
      return setBuilder.build();
    });

    extensionModelsNameSet = new LazyValue<>(() -> extensionModelsByName.get().keySet());

  }

  public Optional<ExtensionModel> get(String extensionName) {
    return ofNullable(extensionModelsByName.get().get(extensionName));
  }

  public Set<ExtensionModel> getAll() {
    return extensionModelsSet.get();
  }

  public Set<String> getAllNames() {
    return extensionModelsNameSet.get();
  }

}
