/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.mediator.resolver;

import static java.util.Collections.emptySet;
import static org.mule.runtime.module.extension.internal.loader.utils.FieldValueProviderNameUtils.getParameterName;
import static org.mule.tooling.client.internal.utils.ValueResolverUtils.toResolvedValues;
import static org.mule.tooling.client.internal.utils.ValueResolverUtils.toValueResolverResult;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.api.value.ValueResult;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.tooling.client.api.value.resolver.ValueResolverResult;
import org.mule.tooling.client.internal.session.filter.FieldValueFilter;
import org.mule.tooling.client.internal.session.filter.exception.InvalidLevelValueException;
import org.mule.tooling.client.internal.session.filter.exception.MissingLevelException;
import org.mule.tooling.client.internal.session.filter.exception.UnknownLevelValueException;

import java.util.Set;
import org.mule.tooling.client.internal.session.cache.DeclarationValueProviderCache;

import org.slf4j.Logger;

public class FieldValueProviderResolver extends SkeletalValueResolver<ValueResult> implements ValuesResolver {

  private static final Logger LOGGER = getLogger(FieldValueProviderResolver.class);

  private final ParameterizedElementDeclaration elementDeclaration;
  private final FieldValueProviderModel fieldValueProviderModel;
  private final DeclarationValueProviderCache valueProviderCache;
  private final FieldValueFilter fieldValueFilter;

  public FieldValueProviderResolver(SessionFunction<ValueResult> sessionFunction,
                                    boolean ignoreCache,
                                    ParameterizedElementDeclaration elementDeclaration,
                                    FieldValueProviderModel fieldValueProviderModel,
                                    DeclarationValueProviderCache valueProviderCache,
                                    FieldValueFilter fieldValueFilter) {
    super(sessionFunction, ignoreCache);
    this.elementDeclaration = elementDeclaration;
    this.fieldValueProviderModel = fieldValueProviderModel;
    this.valueProviderCache = valueProviderCache;
    this.fieldValueFilter = fieldValueFilter;
  }

  @Override
  public ValueResolverResult resolve(ParameterizedModel parameterizedModel,
                                     ParameterModel parameterModel,
                                     String parameterName) {
    final String targetSelector = fieldValueProviderModel.getTargetSelector();
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(
                   "Resolving values using FieldValueProvider mechanism for parameter: '{}' with targetSelector: '{}' using providerName: '{}' on element: '{}' for declared for extension: '{}'",
                   parameterName, targetSelector, parameterName, elementDeclaration.getName(),
                   elementDeclaration.getDeclaringExtension());
    }

    org.mule.runtime.api.value.ValueResult values =
        valueProviderCache.getValues(
                                     elementDeclaration,
                                     parameterName,
                                     targetSelector,
                                     () -> sessionFunction.apply(
                                                                 (runtimeToolingService, id) -> runtimeToolingService
                                                                     .getFieldValues(id, elementDeclaration, parameterName,
                                                                                     targetSelector)),
                                     ignoreCache);

    if (!values.isSuccess()) {
      return toValueResolverResult(values);
    }

    Set<Value> requestedLevelValues = emptySet();

    try {
      requestedLevelValues = fieldValueFilter.filter(fieldValueProviderModel,
                                                     parameterModel,
                                                     elementDeclaration,
                                                     values.getValues());
    } catch (MissingLevelException e) {
      return missingActingParametersFailure(e.getMissingParameterName());
    } catch (InvalidLevelValueException e) {
      return invalidActingParameterFailure(e.getTargetSelector(), e.getMessage());
    } catch (UnknownLevelValueException e) {
      if (!fieldValueProviderModel.isOpen() && fieldValueProviderModel.getPartOrder() > 1) {
        return unknownActingParametersFailure(e.getParameterName(), e.getParameterValue());
      }
    }
    return ValueResolverResult.success(toResolvedValues(requestedLevelValues));
  }
}
