/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.values;

import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.api.value.ResolvingFailure.Builder.newFailure;
import static org.mule.runtime.api.value.ValueResult.resultFrom;
import static org.mule.tooling.client.internal.utils.FunctionalUtils.executeHandling;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.value.ValueResult;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.tooling.client.internal.metadata.ToolingCacheIdGenerator;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

public class DefaultValueProviderCache implements ValueProviderCache {

  private final Map<String, ValueResult> storage;
  private final ToolingCacheIdGenerator<ComponentAst> cacheIdGenerator;

  public DefaultValueProviderCache(Map<String, ValueResult> storage,
                                   ToolingCacheIdGenerator<ComponentAst> cacheIdGenerator) {
    this.storage = storage;
    this.cacheIdGenerator = cacheIdGenerator;
  }

  @Override
  public ValueResult getValues(ComponentAst componentAst, String parameterName, Callable<ValueResult> callable) {
    return cacheIdGenerator.getIdForResolvedValues(componentAst, parameterName)
        .map(id -> storage.computeIfAbsent(id, k -> executeHandling(callable, e -> resultFrom(newFailure(e).build()))))
        .orElseThrow(() -> new IllegalArgumentException(format("Couldn't create a value provider cache id for component: %s",
                                                               componentAst.getLocation())));
  }

  @Override
  public void dispose(ComponentAst componentAst, List<String> parameterNames) {
    List<String> allParameters = parameterNames.isEmpty() ? collectAllProvidedParameterNames(componentAst) : parameterNames;
    allParameters.forEach(
                          p -> cacheIdGenerator.getIdForResolvedValues(componentAst, p).ifPresent(
                                                                                                  id -> storage
                                                                                                      .keySet()
                                                                                                      .remove(id)));
  }

  private List<String> collectAllProvidedParameterNames(ComponentAst componentAst) {
    return componentAst
        .getModel(ParameterizedModel.class)
        .map(m -> m
            .getAllParameterModels()
            .stream()
            .filter(p -> p.getValueProviderModel().isPresent())
            .map(p -> p.getValueProviderModel().get().getProviderName())
            .collect(toList()))
        .orElse(emptyList());
  }
}
