/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.util.Optional.of;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.core.IsNot.not;
import static org.mockito.Mockito.when;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.app.declaration.api.ElementDeclaration;
import org.mule.runtime.config.api.dsl.model.metadata.DeclarationBasedMetadataCacheIdGenerator;
import org.mule.runtime.config.dsl.model.AbstractMockedValueProviderExtensionTestCase;
import org.mule.runtime.core.internal.locator.ComponentLocator;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.junit.Test;

public class DeclarationToolingCacheIdGeneratorTestCase extends AbstractMockedValueProviderExtensionTestCase {

  private static final String CONFIG_ID = "config";
  private static final String CONNECTION_ID = "connection";
  private static final String OPERATION_ID = "operation";


  @Test
  public void configLessConnectionLessOperationDoesNotContainParts() {
    Map<String, String> idMap = computeAllIds();
    String operationId = idMap.get(OPERATION_ID);
    String configId = idMap.get(CONFIG_ID);
    String connectionId = idMap.get(CONNECTION_ID);

    assertThat(operationId, not(containsString(configId)));
    assertThat(operationId, not(containsString(connectionId)));
  }

  @Test
  public void configAwareOperationContainsConfigPart() {
    when(valueProviderModel.requiresConfiguration()).thenReturn(true);

    Map<String, String> idMap = computeAllIds();
    String operationId = idMap.get(OPERATION_ID);
    String configId = idMap.get(CONFIG_ID);
    String connectionId = idMap.get(CONNECTION_ID);

    assertThat(operationId, containsString(configId));
    assertThat(operationId, not(containsString(connectionId)));
  }

  @Test
  public void connectionAwareOperationContainsConnectionPart() {
    when(valueProviderModel.requiresConnection()).thenReturn(true);

    Map<String, String> idMap = computeAllIds();
    String operationId = idMap.get(OPERATION_ID);
    String configId = idMap.get(CONFIG_ID);
    String connectionId = idMap.get(CONNECTION_ID);

    assertThat(operationId, not(containsString(configId)));
    assertThat(operationId, containsString(connectionId));
  }

  @Test
  public void configAndConnectionAwareOperationContainsBoth() {
    when(valueProviderModel.requiresConfiguration()).thenReturn(true);
    when(valueProviderModel.requiresConnection()).thenReturn(true);

    Map<String, String> idMap = computeAllIds();
    String operationId = idMap.get(OPERATION_ID);
    String configId = idMap.get(CONFIG_ID);
    String connectionId = idMap.get(CONNECTION_ID);

    assertThat(operationId, containsString(configId));
    assertThat(operationId, containsString(connectionId));
  }

  private Map<String, String> computeAllIds() {
    ArtifactDeclaration artifactDeclaration = getBaseApp();
    ElementDeclaration configurationElementDeclaration = getDeclaration(artifactDeclaration, MY_CONFIG).get();
    ElementDeclaration connectionElementDeclaration = getDeclaration(artifactDeclaration, MY_CONNECTION).get();
    ElementDeclaration operationDeclaration = getDeclaration(artifactDeclaration, OPERATION_LOCATION).get();

    ComponentLocator<ElementDeclaration> locator = l -> of(configurationElementDeclaration);

    ToolingCacheIdGenerator<ElementDeclaration> cacheIdGenerator = new DeclarationToolingMetadataCacheIdGenerator(
                                                                                                                  new DeclarationBasedMetadataCacheIdGenerator(dslContext,
                                                                                                                                                               locator),
                                                                                                                  new DeclarationToolingMetadataCacheIdGenerator.DeclarationBasedValueProviderCacheIdGeneratorAdapter(dslContext,
                                                                                                                                                                                                                      locator));

    Map<String, String> map = new HashMap<>();
    Optional<String> operationId = cacheIdGenerator.getIdForResolvedValues(operationDeclaration, PROVIDED_PARAMETER_NAME);
    Optional<String> configId = cacheIdGenerator.getIdForResolvedValuesDependency(configurationElementDeclaration);
    Optional<String> connectionId = cacheIdGenerator.getIdForResolvedValuesDependency(connectionElementDeclaration);

    map.put(CONFIG_ID, configId.get());
    map.put(CONNECTION_ID, connectionId.get());
    map.put(OPERATION_ID, operationId.get());

    return map;
  }

}
