/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.util.Preconditions.checkState;

import org.mule.runtime.api.meta.MuleVersion;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.internal.serialization.Serializer;

import java.io.File;
import java.util.List;
import java.util.Optional;

/**
 * Implementation of {@link ExtensionModelService} that deleages to a {@link MuleRuntimeExtensionModelProvider} and builds the
 * Tooling view of a Mule Runtime {@link org.mule.runtime.api.meta.model.ExtensionModel} to
 * {@link org.mule.tooling.client.api.extension.model.ExtensionModel}.
 *
 * @since 4.0
 */
public class ToolingExtensionModelAdapter implements ExtensionModelService, Command {

  private MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
  private Serializer serializer;
  private Optional<MuleVersion> targetRuntimeVersion;

  public ToolingExtensionModelAdapter(MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                                      Serializer serializer,
                                      Optional<MuleVersion> targetRuntimeVersion) {
    this.muleRuntimeExtensionModelProvider = muleRuntimeExtensionModelProvider;
    this.serializer = serializer;
    this.targetRuntimeVersion = targetRuntimeVersion;
  }

  @Override
  public Optional<ExtensionModel> loadExtensionModel(ArtifactDescriptor pluginDescriptor) {
    requireNonNull(pluginDescriptor, "pluginDescriptor cannot be null");
    return muleRuntimeExtensionModelProvider.getExtensionModel(pluginDescriptor)
        .map(runtimeExtensionModel -> toDTO(runtimeExtensionModel,
                                            muleRuntimeExtensionModelProvider.getMinMuleVersion(pluginDescriptor).get()));
  }

  @Override
  public Optional<ExtensionModel> loadExtensionModel(File pluginFile) {
    requireNonNull(pluginFile, "pluginFile cannot be null");
    return muleRuntimeExtensionModelProvider.getExtensionModel(pluginFile)
        .map(loadedExtensionInformation -> toDTO(loadedExtensionInformation.getExtensionModel(),
                                                 loadedExtensionInformation.getMinMuleVersion()));
  }

  @Override
  public Optional<String> loadExtensionSchema(ArtifactDescriptor pluginDescriptor) {
    requireNonNull(pluginDescriptor, "pluginDescriptor cannot be null");
    return muleRuntimeExtensionModelProvider.getExtensionSchema(pluginDescriptor);
  }

  @Override
  public Optional<String> loadExtensionSchema(File pluginFile) {
    requireNonNull(pluginFile, "pluginDescriptor cannot be null");
    return muleRuntimeExtensionModelProvider.getExtensionSchema(pluginFile);
  }

  @Override
  public List<ExtensionModel> loadMuleExtensionModels() {
    return muleRuntimeExtensionModelProvider.getRuntimeExtensionModels().stream()
        .map(runtimeExtensionModel -> toDTO(runtimeExtensionModel, runtimeExtensionModel.getVersion()))
        .collect(toList());
  }

  private ExtensionModel toDTO(org.mule.runtime.api.meta.model.ExtensionModel runtimeExtensionModel, String minMuleVersion) {
    return new ExtensionModelFactory(targetRuntimeVersion).createExtensionModel(runtimeExtensionModel, minMuleVersion);
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "loadExtensionModel": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        final String argClazz = classes[0];
        checkState(classes.length == 1
            && (argClazz.equals(ArtifactDescriptor.class.getName()) || argClazz.equals(File.class.getName())),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));

        if (argClazz.equals(ArtifactDescriptor.class.getName())) {
          return serializer.serialize(loadExtensionModel((ArtifactDescriptor) serializer.deserialize(arguments[0])));
        } else if (argClazz.equals(File.class.getName())) {
          return serializer.serialize(loadExtensionModel((File) serializer.deserialize(arguments[0])));
        }
      }
      case "loadExtensionSchema": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        final String argClazz = classes[0];
        checkState(classes.length == 1
            && (argClazz.equals(ArtifactDescriptor.class.getName()) || argClazz.equals(File.class.getName())),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));

        if (argClazz.equals(ArtifactDescriptor.class.getName())) {
          return serializer.serialize(loadExtensionSchema((ArtifactDescriptor) serializer.deserialize(arguments[0])));
        } else if (argClazz.equals(File.class.getName())) {
          return serializer.serialize(loadExtensionSchema((File) serializer.deserialize(arguments[0])));
        }
      }
      case "loadMuleExtensionModels": {
        return serializer.serialize(loadMuleExtensionModels());
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
