/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static org.mule.runtime.config.api.properties.ConfigurationPropertiesResolverProvider.builder;
import static org.mule.runtime.core.api.util.ClassUtils.memoize;

import static java.lang.String.format;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;

import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.config.api.properties.ConfigurationPropertiesResolverProvider;
import org.mule.runtime.deployment.model.api.plugin.resolver.PluginDependenciesResolver;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.ApplicationDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.DefaultApplicationModelFactory;
import org.mule.tooling.client.internal.ToolingArtifactContext;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.locks.Lock;
import java.util.function.Function;

import com.google.common.collect.ImmutableList;

/**
 * Represents a Mule application and allows to be deployed into Mule Runtime.
 *
 * @since 4.0
 */
public class DefaultApplication extends AbstractArtifact<ApplicationDescriptor> implements Application {

  private final Optional<Domain> domain;
  private boolean shouldDisposeDomain = false;

  private final List<ArtifactPluginDescriptor> effectivePlugins;

  public DefaultApplication(String id, ArtifactResources artifactResources, ApplicationDescriptor applicationDescriptor,
                            Domain domain, ToolingArtifactContext context, Map<String, String> properties,
                            boolean shouldDisposeDomain) {
    super(id, artifactResources, applicationDescriptor, context, properties);
    if (shouldDisposeDomain && domain == null) {
      throw new IllegalArgumentException("Application cannot be set to dispose the domain without setting a domain reference");
    }
    this.domain = ofNullable(domain);
    this.shouldDisposeDomain = shouldDisposeDomain;

    this.effectivePlugins = effectivePlugins(context.getPluginDependenciesResolver());
  }

  @Override
  public List<ArtifactPluginDescriptor> getEffectivePlugins() {
    return effectivePlugins;
  }

  private List<ArtifactPluginDescriptor> effectivePlugins(PluginDependenciesResolver applicationPluginDescriptorsResolver) {
    if (domain.isPresent()) {
      try {
        return applicationPluginDescriptorsResolver
            .resolve(domain.get().getDescriptor().getPlugins(), new LinkedList<>(this.getDescriptor().getPlugins()), true);
      } catch (Exception e) {
        throw new ToolingException("Error while creating application", e);
      }
    }
    return ImmutableList.copyOf(this.getDescriptor().getPlugins());
  }

  @Override
  public boolean shouldDisposeDomain() {
    return shouldDisposeDomain;
  }

  @Override
  public Optional<Domain> getDomain() {
    return domain;
  }

  @Override
  public ApplicationDescriptor getDescriptor() {
    return artifactDescriptor;
  }

  @Override
  protected List<ArtifactClassLoader> getArtifactPluginClassLoaders() {
    if (!domain.isPresent()) {
      return super.getArtifactPluginClassLoaders();
    }
    return ImmutableList.<ArtifactClassLoader>builder()
        .addAll(domain.get().getArtifactClassLoader().getArtifactPluginClassLoaders())
        .addAll(getArtifactClassLoader().getArtifactPluginClassLoaders())
        .build();
  }

  @Override
  public <R> R evaluateWithRemoteApplication(ApplicationRemoteFunction<R> function) {
    checkState();

    String domainId = null;
    if (domain.isPresent()) {
      domainId = domain.get().evaluateWithRemoteDomain(
                                                       (deployedDomainId, runtimeToolingService) -> deployedDomainId);
    }

    if (!deployed) {
      Lock writeLock = deploymentLock.writeLock();
      if (writeLock.tryLock()) {
        try {
          if (!deployed) {
            deployed = true;
            remoteArtifactId = artifactDeployer.deploy(domainId);
            return function.apply(remoteArtifactId, runtimeToolingService.get());
          }
        } finally {
          writeLock.unlock();
        }
      }
    }

    Lock readLock = deploymentLock.readLock();
    readLock.lock();
    try {
      return function.apply(remoteArtifactId, runtimeToolingService.get());
    } finally {
      readLock.unlock();
    }
  }

  @Override
  protected LazyValue<ToolingArtifactClassLoader> newToolingArtifactClassLoaderLazyValue() {
    return new LazyValue<>(() -> context.getApplicationClassLoaderFactory().createApplicationClassLoader(artifactDescriptor,
                                                                                                         artifactResources
                                                                                                             .getWorkingDirectory(),
                                                                                                         domain
                                                                                                             .map(parent -> parent
                                                                                                                 .getArtifactClassLoader())
                                                                                                             .orElse(null)));
  }

  @Override
  protected LazyValue<ToolingApplicationModel> newToolingApplicationModelLazyValue() {
    return new LazyValue<>(() -> {
      // Cache the provider generated for the ast during its parse to be able to store it in the ToolingApplicationModel created
      // below
      Function<ArtifactAst, ConfigurationPropertiesResolverProvider> configurationPropertiesResolver =
          memoize(originalAst -> builder().from(originalAst)
              .withParentProperties(of(domain
                  .map(parent -> parent
                      .getApplicationModel()
                      .getConfigurationProperties())
                  .orElse(noFailureConfigurationProperties)))
              .withDeploymentProperties(getProperties())
              .loadingResourcesWith(getArtifactClassLoader().getClassLoader())
              .build(),
                  new HashMap<>());

      final ArtifactAst ast = new DefaultApplicationModelFactory()
          .createApplicationModel(artifactDescriptor,
                                  getAstXmlParser(),
                                  getArtifactClassLoader()
                                      .getClassLoader(),
                                  configurationPropertiesResolver
                                      .andThen(ConfigurationPropertiesResolverProvider::getConfigurationPropertiesResolver))
          .orElseThrow(() -> new ToolingException(format("Couldn't create ApplicationModel from %s",
                                                         this)));

      return new ToolingApplicationModel(domain.map(parent -> parent.getApplicationModel()).orElse(null),
                                         ast,
                                         configurationPropertiesResolver.apply(ast),
                                         getArtifactClassLoader().getClassLoader());
    });
  }

}
