/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.artifact;

import static java.lang.String.format;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.utils.ExtensionModelUtils.resolveExtensionModels;
import static org.mule.tooling.client.internal.utils.ExtensionModelUtils.toBundleDescriptors;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.api.app.declaration.serialization.ArtifactDeclarationJsonSerializer;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.config.api.dsl.ArtifactDeclarationXmlSerializer;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationException;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.declaration.request.AbstractXmlArtifactRequest;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactSerializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactSerializationRequest;
import org.mule.tooling.client.internal.Command;
import org.mule.tooling.client.internal.MuleRuntimeExtensionModelProvider;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.utils.ExtensionModelUtils;

import java.io.InputStream;
import java.util.List;

/**
 * Default implementation of {@link ArtifactSerializationService}.
 *
 * @since 4.0
 */
public class DefaultArtifactSerializationService implements ArtifactSerializationService, Command {

  private final List<ExtensionModel> muleModels;
  private final MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
  private final MavenClient mavenClient;
  private final Serializer serializer;

  public DefaultArtifactSerializationService(List<ExtensionModel> muleModels,
                                             MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                                             MavenClient mavenClient,
                                             Serializer serializer) {
    this.muleModels = muleModels;
    this.muleRuntimeExtensionModelProvider = muleRuntimeExtensionModelProvider;
    this.mavenClient = mavenClient;
    this.serializer = serializer;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toXml(XmlArtifactSerializationRequest request) {
    checkArgument(request != null, "Serialization request cannot be null");
    checkArgument(request.getArtifactDeclaration() != null,
                  "Missing ArtifactDeclaration to serialize. None was found in the given request");
    checkArgument(request.getPluginArtifactDescriptors() != null,
                  "pluginArtifactDescriptors cannot be null");

    try {
      return ArtifactDeclarationXmlSerializer.getDefault(createDslResolvingContext(request))
          .serialize(request.getArtifactDeclaration());
    } catch (Exception e) {
      throw new ArtifactSerializationException("An error occurred while serializing the ArtifactDeclaration", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ArtifactDeclaration fromXml(XmlArtifactDeserializationRequest request) {
    checkArgument(request != null, "Deserialization request cannot be null");
    checkArgument(request.getPluginArtifactDescriptors() != null,
                  "pluginArtifactDescriptors cannot be null");
    checkArgument(request.getArtifactSource() != null,
                  "Missing ArtifactSource to deserialize. None was found in the given request");

    try (InputStream inputStream = request.getArtifactSource()) {
      return ArtifactDeclarationXmlSerializer.getDefault(createDslResolvingContext(request))
          .deserialize(inputStream);
    } catch (Exception e) {
      throw new ArtifactSerializationException("An error occurred while deserializing the ArtifactDeclaration", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toJson(JsonArtifactSerializationRequest request) {
    checkArgument(request != null, "Serialization request cannot be null");
    checkArgument(request.getArtifactDeclaration() != null,
                  "Missing ArtifactDeclaration to serialize. None was found in the given request");

    try {
      return ArtifactDeclarationJsonSerializer.getDefault(request.isPrettyPrint()).serialize(request.getArtifactDeclaration());
    } catch (Exception e) {
      throw new ArtifactSerializationException("An error occurred while serializing the ArtifactDeclaration", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ArtifactDeclaration fromJson(JsonArtifactDeserializationRequest request) {
    checkArgument(request != null, "Serialization request cannot be null");
    checkArgument(request.getArtifactSource() != null && !request.getArtifactSource().trim().isEmpty(),
                  "Missing ArtifactDeclaration source to deserialize. None was found in the given request");

    try {
      return ArtifactDeclarationJsonSerializer.getDefault(false).deserialize(request.getArtifactSource());
    } catch (Exception e) {
      throw new ArtifactSerializationException("An error occurred while deserializing the ArtifactDeclaration", e);
    }
  }

  private DslResolvingContext createDslResolvingContext(AbstractXmlArtifactRequest request) {
    List<ExtensionModel> plugins = resolveExtensionModels(mavenClient,
                                                          muleRuntimeExtensionModelProvider,
                                                          toBundleDescriptors(request.getPluginArtifactDescriptors()));

    return ExtensionModelUtils.createDslResolvingContext(muleModels, plugins);
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "toXml": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(XmlArtifactSerializationRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(toXml(serializer.deserialize(arguments[0])));
      }
      case "fromXml": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(XmlArtifactDeserializationRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(fromXml(serializer.deserialize(arguments[0])));
      }
      case "toJson": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(JsonArtifactSerializationRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(toJson(serializer.deserialize(arguments[0])));
      }
      case "fromJson": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(JsonArtifactDeserializationRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(fromJson(serializer.deserialize(arguments[0])));
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
