/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dataweave;

import org.mule.runtime.api.el.DataFormat;
import org.mule.runtime.api.el.DataFormatConfigOption;
import org.mule.runtime.api.el.ExpressionLanguageCapabilitiesService;
import org.mule.tooling.client.api.dataweave.DataWeaveModuleOption;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

public class ModulesAnalyzer {

  private ExpressionLanguageCapabilitiesService expressionLanguageCapabilitiesService;

  public ModulesAnalyzer(ExpressionLanguageCapabilitiesService expressionLanguageCapabilitiesService) {
    this.expressionLanguageCapabilitiesService = expressionLanguageCapabilitiesService;
  }

  public List<DataWeaveModuleOption> getOptions(String contentType,
                                                Function<DataFormat, Set<DataFormatConfigOption>> getter) {
    expressionLanguageCapabilitiesService.getSupportedDataFormats();

    Optional<DataFormat> moduleOption = byContentType(contentType);
    if (moduleOption.isPresent()) {
      List<DataWeaveModuleOption> response = new ArrayList<>();
      Set<DataFormatConfigOption> formatConfigOptions = getter.apply(moduleOption.get());
      formatConfigOptions.stream().forEach(option -> response.add(toRestModel(option)));
      return response;
    }
    throw new IllegalArgumentException("Module for " + contentType + " not found");
  }

  private Optional<DataFormat> byContentType(String contentType) {
    return expressionLanguageCapabilitiesService.getSupportedDataFormats().stream()
        .filter(dataFormat -> dataFormat.getSupportedMimeTypes().contains(contentType)).findFirst();
  }

  private DataWeaveModuleOption toRestModel(DataFormatConfigOption option) {
    DataWeaveModuleOption model = new DataWeaveModuleOption();
    model.setName(option.getName());
    model.setType(option.getType().toString());
    model.setDefaultValue(option.getDefaultValue().map(object -> object.toString()).orElse(""));
    model.setDescription(option.getDescription());
    model
        .setPossibleValues(option.getPossibleValues().stream().map(object -> object.toString()).collect(Collectors.joining(",")));
    return model;
  }

}
