/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.lang.Long.parseLong;
import static java.util.Base64.getDecoder;
import static java.util.Base64.getEncoder;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.internal.dsl.DslConstants.CONFIG_ATTRIBUTE_NAME;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataFailuresDTO;
import static org.mule.tooling.client.internal.serialization.KryoFactory.defaultKryo;

import org.mule.datasense.enrichment.model.IdComponentModelSelector;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.tooling.client.api.datasense.ImmutableMetadataCacheKeyInfo;
import org.mule.tooling.client.api.datasense.ImmutableMetadataResult;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.metadata.MetadataFailure;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;

import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.pool.KryoPool;

public class LegacyToolingMetadataCache implements MetadataCache {

  public static final String TIMESTAMP = "timestamp";
  private final LazyValue<ArtifactAst> artifactAstLazyValue;
  private final Map<String, String> toolingArtifactProperties;
  private final org.mule.tooling.client.api.datasense.MetadataCache delegate;
  private final KryoPool kryoPool;

  public LegacyToolingMetadataCache(LazyValue<ArtifactAst> artifactAstLazyValue, Map<String, String> toolingArtifactProperties,
                                    org.mule.tooling.client.api.datasense.MetadataCache delegate) {
    this.artifactAstLazyValue = artifactAstLazyValue;
    this.toolingArtifactProperties = toolingArtifactProperties;
    this.delegate = delegate;
    this.kryoPool = new KryoPool.Builder(() -> defaultKryo()).build();
  }

  @Override
  public MetadataResult<OperationModel> getOperationMetadata(ComponentAst componentAst,
                                                             Callable<MetadataResult<OperationModel>> resolver) {
    return deserialize((String) delegate.getOperationMetadata(
                                                              new ImmutableMetadataCacheKeyInfo(getComponentId(componentAst),
                                                                                                componentAst.getLocation()
                                                                                                    .getLocation(),
                                                                                                resolveEffectiveTimestamp(componentAst),
                                                                                                toolingArtifactProperties),
                                                              () -> {
                                                                MetadataResult<OperationModel> result = resolver.call();
                                                                List<MetadataFailure> failures = emptyList();
                                                                if (!result.isSuccess()) {
                                                                  failures = toMetadataFailuresDTO(result.getFailures());
                                                                }
                                                                return new ImmutableMetadataResult(result.isSuccess(),
                                                                                                   serialize(result), failures);
                                                              }));
  }

  @Override
  public MetadataResult<SourceModel> getSourceMetadata(ComponentAst componentAst,
                                                       Callable<MetadataResult<SourceModel>> resolver) {
    return deserialize((String) delegate.getSourceMetadata(
                                                           new ImmutableMetadataCacheKeyInfo(getComponentId(componentAst),
                                                                                             componentAst.getLocation()
                                                                                                 .getLocation(),
                                                                                             resolveEffectiveTimestamp(componentAst),
                                                                                             toolingArtifactProperties),
                                                           () -> {
                                                             MetadataResult<SourceModel> result = resolver.call();
                                                             List<MetadataFailure> failures = emptyList();
                                                             if (!result.isSuccess()) {
                                                               failures = toMetadataFailuresDTO(result.getFailures());
                                                             }
                                                             return new ImmutableMetadataResult(result.isSuccess(),
                                                                                                serialize(result), failures);
                                                           }));
  }

  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(ComponentAst componentAst,
                                                               Callable<MetadataResult<MetadataKeysContainer>> resolver) {
    return deserialize((String) delegate.getMetadataKeys(
                                                         new ImmutableMetadataCacheKeyInfo(getComponentId(componentAst),
                                                                                           componentAst.getLocation()
                                                                                               .getLocation(),
                                                                                           resolveEffectiveTimestamp(componentAst),
                                                                                           toolingArtifactProperties),
                                                         () -> {
                                                           MetadataResult<MetadataKeysContainer> result = resolver.call();
                                                           List<MetadataFailure> failures = emptyList();
                                                           if (!result.isSuccess()) {
                                                             failures = toMetadataFailuresDTO(result.getFailures());
                                                           }
                                                           return new ImmutableMetadataResult(result.isSuccess(),
                                                                                              serialize(result), failures);
                                                         }));
  }

  @Override
  public void dispose(ComponentAst componentAst) {
    // Not supported
  }

  @Override
  public void invalidateMetadataKeysFor(ComponentAst componentAst) {
    // Not supported
  }

  private String serialize(Object object) {
    try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
      try (Output output = new Output(byteArrayOutputStream)) {
        kryoPool.run(kryo -> {
          kryo.writeClassAndObject(output, object);
          return null;
        });
      }
      return getEncoder().encodeToString(byteArrayOutputStream.toByteArray());
    } catch (IOException e) {
      throw new ToolingException("Error while creating object from serialization", e);
    }
  }

  private <T> T deserialize(String content) {
    try (Input input = new Input(new ByteArrayInputStream(getDecoder().decode(content)))) {
      return (T) kryoPool.run(kryo -> kryo.readClassAndObject(input));
    }
  }

  private Optional<Long> resolveTimestamp(ComponentAst componentAst) {
    if (componentAst == null) {
      return empty();
    }
    try {
      return ofNullable(parseLong(componentAst.getMetadata().getDocAttributes().get(TIMESTAMP)));
    } catch (NumberFormatException e) {
      return empty();
    }
  }

  private Optional<String> getConfigurationRef(ComponentAst componentAst) {
    ComponentParameterAst componentAstParameter = componentAst.getParameter(DEFAULT_GROUP_NAME, CONFIG_ATTRIBUTE_NAME);
    if (componentAstParameter == null) {
      return empty();
    }
    return componentAstParameter.getValue().getValue();
  }

  private String getComponentId(ComponentAst componentAst) {
    return IdComponentModelSelector.getComponentId(componentAst);
  }

  private Long resolveEffectiveTimestamp(ComponentAst componentAst) {
    final Long componentTimestamp = resolveTimestamp(componentAst).orElse(null);
    final Optional<ComponentAst> relatedConfigurationOptional =
        getConfigurationRef(componentAst).flatMap(globalElementName -> artifactAstLazyValue.get().topLevelComponentsStream()
            .filter(topLevelComponentAst -> globalElementName.equals(topLevelComponentAst.getComponentId().orElse(null)))
            .findAny());
    if (relatedConfigurationOptional.isPresent()) {
      final Long configurationTimestamp = resolveTimestamp(relatedConfigurationOptional.get()).orElse(null);
      return componentTimestamp != null && configurationTimestamp != null ? Long.max(componentTimestamp, configurationTimestamp)
          : null;
    } else {
      return componentTimestamp;
    }
  }


}
