/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session;

import static java.lang.String.format;
import static java.util.Collections.emptyMap;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.exception.ExceptionUtils.getStackTrace;
import static org.mule.runtime.api.metadata.resolving.FailureCode.INVALID_CONFIGURATION;
import static org.mule.runtime.api.metadata.resolving.FailureCode.INVALID_METADATA_KEY;
import static org.mule.runtime.api.metadata.resolving.FailureCode.UNKNOWN;
import static org.mule.sdk.api.values.ValueResolvingException.MISSING_REQUIRED_PARAMETERS;
import static org.mule.tooling.client.api.metadata.FailureCode.componentNotFoundFailureCode;
import static org.mule.tooling.client.api.metadata.FailureCode.invalidConfigurationFailureCode;
import static org.mule.tooling.client.api.metadata.FailureCode.invalidMetadataKeyFailureCode;
import static org.mule.tooling.client.api.metadata.FailureCode.noneFailureCode;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toComponentMetadataTypesDescriptorDTO;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataResultDTO;
import static org.mule.tooling.client.internal.session.cache.NoOpDeclarationCache.noOpDeclarationCache;
import static org.mule.tooling.client.internal.session.validation.SessionCallValidator.INVALID_ELEMENT;
import static org.mule.tooling.client.internal.utils.MetadataTypesUtils.requiresConfiguration;
import static org.mule.tooling.client.internal.utils.MetadataTypesUtils.requiresConnection;
import static org.mule.tooling.client.internal.utils.SampleDataUtils.toComponentSampleDataResult;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.api.MavenClient;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.meta.model.HasOutputModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataFailure;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.api.value.ValueResult;
import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.app.declaration.api.ConfigurationElementDeclaration;
import org.mule.runtime.app.declaration.api.ElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.runtime.app.declaration.api.TopLevelParameterDeclaration;
import org.mule.runtime.config.api.dsl.model.metadata.DeclarationBasedMetadataCacheIdGenerator;
import org.mule.runtime.metadata.api.locator.ComponentLocator;
import org.mule.runtime.extension.api.property.TypeResolversInformationModelProperty;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.api.sampledata.SampleDataMessageModelFailure;
import org.mule.tooling.api.sampledata.SampleDataMessageModelResult;
import org.mule.tooling.client.api.cache.CacheStorage;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;
import org.mule.tooling.client.api.declaration.session.DeclarationSession;
import org.mule.tooling.client.api.declaration.session.DeclarationSessionCacheService;
import org.mule.tooling.client.api.descriptors.dependency.Dependency;
import org.mule.tooling.client.api.metadata.ComponentMetadataTypesDescriptor;
import org.mule.tooling.client.api.metadata.FailureCode;
import org.mule.tooling.client.api.metadata.MetadataResult;
import org.mule.tooling.client.api.sampledata.ComponentSampleDataFailure;
import org.mule.tooling.client.api.sampledata.ComponentSampleDataResult;
import org.mule.tooling.client.api.value.resolver.ValueResolverResult;
import org.mule.tooling.client.internal.Command;
import org.mule.tooling.client.internal.MuleRuntimeExtensionModelProvider;
import org.mule.tooling.client.internal.cache.DefaultCacheStorageSerializer;
import org.mule.tooling.client.internal.metadata.DeclarationToolingMetadataCacheIdGenerator;
import org.mule.tooling.client.internal.metadata.ToolingCacheIdGenerator;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.session.cache.DeclarationMetadataCache;
import org.mule.tooling.client.internal.session.cache.DeclarationValueProviderCache;
import org.mule.tooling.client.internal.session.cache.DefaultDeclarationMetadataCache;
import org.mule.tooling.client.internal.session.cache.DefaultDeclarationValueProviderCache;
import org.mule.tooling.client.internal.session.cache.MDCMapDecorator;
import org.mule.tooling.client.internal.session.factory.RemoteDeclarationSessionProvider;
import org.mule.tooling.client.internal.session.filter.FieldValueFilter;
import org.mule.tooling.client.internal.session.filter.MetadataKeyFilter;
import org.mule.tooling.client.internal.session.filter.ValueFilter;
import org.mule.tooling.client.internal.session.mediator.FieldValuesResolverModelMediator;
import org.mule.tooling.client.internal.session.mediator.ValuesResolverModelMediator;
import org.mule.tooling.client.internal.session.mediator.resolver.ValuesResolverFactory;
import org.mule.tooling.client.internal.session.validation.ComponentMetadataKeyValidator;
import org.mule.tooling.client.internal.session.validation.ConnectionAndConfigurationValidator;
import org.mule.tooling.client.internal.session.validation.SampleDataActingParametersValidator;
import org.mule.tooling.client.internal.session.validation.SessionCallValidationException;
import org.mule.tooling.client.internal.session.validation.SessionCallValidator;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;

public class DefaultDeclarationSession implements DeclarationSession, Command {

  private static final Map<String, FailureCode> FAILURE_CODES = ImmutableMap.of(
                                                                                INVALID_CONFIGURATION.getName(),
                                                                                invalidConfigurationFailureCode(INVALID_CONFIGURATION
                                                                                    .getName()),
                                                                                INVALID_ELEMENT,
                                                                                componentNotFoundFailureCode(INVALID_ELEMENT),
                                                                                INVALID_METADATA_KEY.getName(),
                                                                                invalidMetadataKeyFailureCode(INVALID_METADATA_KEY
                                                                                    .getName()));

  private final Logger logger = getLogger(this.getClass());

  private final RemoteDeclarationSessionProvider remoteDeclarationSessionProvider;
  private final RemoteSessionExecutor remoteSessionExecutor;
  private final Serializer serializer;
  private final RuntimeToolingService runtimeToolingService;
  private final DeclarationSessionCacheService cacheService;
  private final ConfigurationDeclarationProvider configDeclarationProvider;
  private final DeclarationValueProviderCache valueProviderCache;
  private final DeclarationMetadataCache metadataCache;

  private final SessionCallValidator callValidator;

  private final ValuesResolverFactory valuesResolverFactory;

  private DefaultDeclarationSession(RemoteDeclarationSessionProvider remoteDeclarationSessionProvider,
                                    Serializer serializer,
                                    ExtensionModelProvider extensionModelProvider,
                                    ConfigurationDeclarationProvider configurationDeclarationProvider,
                                    RuntimeToolingService runtimeToolingService,
                                    DeclarationValueProviderCache valueProviderCache,
                                    DeclarationMetadataCache metadataCache,
                                    DeclarationSessionCacheService cacheService,
                                    ExpressionLanguage expressionLanguage) {
    requireNonNull(remoteDeclarationSessionProvider, "remoteDeclarationSessionProvider info cannot be null");
    requireNonNull(serializer, "serializer cannot be null");
    requireNonNull(extensionModelProvider, "extensionModelProvider cannot be null");
    requireNonNull(configurationDeclarationProvider, "configurationElementDeclaration cannot be null");
    requireNonNull(runtimeToolingService, "runtimeToolingService cannot be null");
    requireNonNull(valueProviderCache, "valueProviderCache cannot be null");
    requireNonNull(metadataCache, "metadataCache cannot be null");
    requireNonNull(cacheService, "cacheService cannot be null");
    requireNonNull(expressionLanguage, "expressionLanguage cannot be null");

    this.remoteDeclarationSessionProvider = remoteDeclarationSessionProvider;
    this.remoteSessionExecutor = new RemoteSessionExecutor(remoteDeclarationSessionProvider);
    this.serializer = serializer;
    this.configDeclarationProvider = configurationDeclarationProvider;
    this.runtimeToolingService = runtimeToolingService;
    this.valueProviderCache = valueProviderCache;
    this.metadataCache = metadataCache;
    this.cacheService = cacheService;

    this.callValidator = new SessionCallValidator(extensionModelProvider, configurationDeclarationProvider);
    this.valuesResolverFactory = new ValuesResolverFactory(remoteSessionExecutor, runtimeToolingService, valueProviderCache,
                                                           metadataCache, new ValueFilter(), new MetadataKeyFilter(),
                                                           new FieldValueFilter(expressionLanguage));
  }

  @Override
  public ConnectionValidationResult testConnection(String configName) {
    try {
      callValidator.validateConfig(configName,
                                   c -> {
                                     if (c.getConfigurationModel().getConnectionProviders().isEmpty()
                                         && c.getExtensionModel().getConnectionProviders().isEmpty()) {
                                       throw new SessionCallValidationException(format("Referenced config: '%s' can't have a connection provider",
                                                                                       c.getConfigDeclaration().getRefName()),
                                                                                "No connection provider in config",
                                                                                INVALID_ELEMENT);
                                     }
                                     if (!c.getConfigDeclaration().getConnection().isPresent()) {
                                       throw new SessionCallValidationException(format("The referenced config: '%s' does not have a connection defined",
                                                                                       c.getConfigDeclaration().getRefName()),
                                                                                "No connection in config",
                                                                                MISSING_REQUIRED_PARAMETERS);
                                     }
                                   });
    } catch (SessionCallValidationException e) {
      ConnectionValidationResult connectionValidationResult =
          new ConnectionValidationResult();
      connectionValidationResult.setValid(false);
      connectionValidationResult.setMessage(e.getMessage());
      connectionValidationResult.setException(getStackTrace(e));
      return connectionValidationResult;
    }
    return remoteSessionExecutor.withRemoteDeclarationSession(id -> runtimeToolingService.testConnection(id, configName),
                                                              e -> {
                                                                ConnectionValidationResult connectionValidationResult =
                                                                    new ConnectionValidationResult();
                                                                connectionValidationResult.setValid(false);
                                                                connectionValidationResult.setMessage(e.getMessage());
                                                                connectionValidationResult.setException(getStackTrace(e));
                                                                return connectionValidationResult;
                                                              });
  }

  @Override
  public ValueResolverResult getValues(ParameterizedElementDeclaration elementDeclaration,
                                       String parameterName,
                                       boolean ignoreCache) {
    logger.info("Resolving values for parameter: '{}' on element: '{}' for declared for extension: '{}'", parameterName,
                elementDeclaration.getName(), elementDeclaration.getDeclaringExtension());
    return new ValuesResolverModelMediator(configDeclarationProvider, elementDeclaration, callValidator, parameterName)
        .resolve(valuesResolverFactory, ignoreCache);
  }

  @Override
  public ValueResolverResult getFieldValues(ParameterizedElementDeclaration elementDeclaration, String parameterName,
                                            String targetSelector, boolean ignoreCache) {
    logger.info("Resolving values for parameter: '{}' for targetSelector '{}' on element: '{}' for declared for extension: '{}'",
                parameterName,
                targetSelector,
                elementDeclaration.getName(),
                elementDeclaration.getDeclaringExtension());
    return new FieldValuesResolverModelMediator(configDeclarationProvider, elementDeclaration, callValidator, parameterName,
                                                targetSelector).resolve(valuesResolverFactory, ignoreCache);
  }

  @Override
  public MetadataResult<ComponentMetadataTypesDescriptor> resolveComponentMetadata(ComponentElementDeclaration componentElementDeclaration,
                                                                                   boolean ignoreCache) {
    try {
      Reference<Boolean> requiresConnection = new Reference<>(false);
      Reference<Boolean> requiresConfiguration = new Reference<>(false);
      callValidator.validateComponent(componentElementDeclaration,
                                      c -> c.getEnrichableModel()
                                          .getModelProperty(TypeResolversInformationModelProperty.class)
                                          .ifPresent(mp -> {
                                            requiresConnection.set(requiresConnection(c.getParameterizedModel(), mp));
                                            requiresConfiguration.set(requiresConfiguration(c.getParameterizedModel(), mp));
                                          }),
                                      new ConnectionAndConfigurationValidator(componentElementDeclaration,
                                                                              configDeclarationProvider,
                                                                              requiresConnection::get,
                                                                              requiresConfiguration::get,
                                                                              () -> format("Type resolver on element: '%s' for extension: '%s'",
                                                                                           componentElementDeclaration.getName(),
                                                                                           componentElementDeclaration
                                                                                               .getDeclaringExtension())),
                                      new ComponentMetadataKeyValidator(componentElementDeclaration));
    } catch (SessionCallValidationException e) {
      return MetadataResult.failure(
                                    org.mule.tooling.client.api.metadata.MetadataFailure.Builder.newFailure()
                                        .withMessage(e.getMessage())
                                        .withReason(e.getReason())
                                        .withFailureCode(FAILURE_CODES.getOrDefault(e.getFailureCode(),
                                                                                    noneFailureCode(UNKNOWN.getName())))
                                        .onComponent());
    }

    org.mule.runtime.api.metadata.resolving.MetadataResult<org.mule.runtime.api.metadata.descriptor.ComponentMetadataTypesDescriptor> componentMetadataTypesMetadataResult =
        metadataCache.getComponentMetadata(componentElementDeclaration,
                                           () -> remoteSessionExecutor
                                               .withRemoteDeclarationSession(id -> runtimeToolingService.getComponentMetadata(id,
                                                                                                                              componentElementDeclaration,
                                                                                                                              ignoreCache),
                                                                             e -> org.mule.runtime.api.metadata.resolving.MetadataResult
                                                                                 .failure(MetadataFailure.Builder.newFailure(e)
                                                                                     .onComponent())),
                                           ignoreCache);
    return

    toMetadataResultDTO(componentMetadataTypesMetadataResult,
                        () ->

                        toComponentMetadataTypesDescriptorDTO(componentMetadataTypesMetadataResult.get()));
  }

  @Override
  public ComponentSampleDataResult getSampleData(ComponentElementDeclaration componentElementDeclaration) {
    if (logger.isDebugEnabled()) {
      logger.debug(
                   "Resolving sample data on element: '{}' declared for extension: '{}'",
                   componentElementDeclaration.getName(), componentElementDeclaration.getDeclaringExtension());
    }
    Reference<Boolean> requiresConfig = new Reference<>(false);
    Reference<Boolean> requiresConnection = new Reference<>(false);
    try {
      callValidator.validateComponent(componentElementDeclaration,
                                      c -> c.getComponentModel()
                                          .filter(cm -> cm instanceof HasOutputModel)
                                          .map(cm -> (HasOutputModel) cm)
                                          .flatMap(HasOutputModel::getSampleDataProviderModel)
                                          .ifPresent(sampleDataProviderModel -> {
                                            requiresConfig.set(sampleDataProviderModel.requiresConfiguration());
                                            requiresConnection.set(sampleDataProviderModel.requiresConnection());
                                          }),
                                      new SampleDataActingParametersValidator(),
                                      new ConnectionAndConfigurationValidator(componentElementDeclaration,
                                                                              configDeclarationProvider,
                                                                              requiresConnection::get,
                                                                              requiresConfig::get,
                                                                              () -> format("SampleData on element: '%s' for extension: '%s'",
                                                                                           componentElementDeclaration.getName(),
                                                                                           componentElementDeclaration
                                                                                               .getDeclaringExtension())));
    } catch (SessionCallValidationException e) {
      return new ComponentSampleDataResult(new ComponentSampleDataFailure(e.getMessage(), e.getReason(), e.getFailureCode()));
    }
    SampleDataMessageModelResult sampleDataMessageModelResult =
        remoteSessionExecutor
            .withRemoteDeclarationSession(id -> runtimeToolingService.getSampleData(id, componentElementDeclaration),
                                          e -> new SampleDataMessageModelResult(new SampleDataMessageModelFailure(e)));
    return toComponentSampleDataResult(sampleDataMessageModelResult);
  }

  @Override
  public DeclarationSessionCacheService getCacheService() {
    return this.cacheService;
  }


  @Override
  public void dispose() {
    if (this.remoteDeclarationSessionProvider.isDeployed()) {
      remoteSessionExecutor.withRemoteDeclarationSession(id -> {
        runtimeToolingService.disposeDeclarationSession(id);
        return null;
      }, e -> null);
    }
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "getFieldValues": {
        validateInput(arguments, classes, ParameterizedElementDeclaration.class, String.class, String.class, Boolean.class);
        final ParameterizedElementDeclaration elementDeclaration = serializer.deserialize(arguments[0]);
        final String parameterName = serializer.deserialize(arguments[1]);
        final String targetSelector = serializer.deserialize(arguments[2]);
        final boolean ignoreCache = serializer.deserialize(arguments[3]);
        return serializer.serialize(getFieldValues(elementDeclaration, parameterName, targetSelector, ignoreCache));
      }
      case "getValues": {
        validateInput(arguments, classes, ParameterizedElementDeclaration.class, String.class, Boolean.class);
        final ParameterizedElementDeclaration elementDeclaration = serializer.deserialize(arguments[0]);
        final String parameterName = serializer.deserialize(arguments[1]);
        final boolean ignoreCache = serializer.deserialize(arguments[2]);
        return serializer.serialize(getValues(elementDeclaration, parameterName, ignoreCache));
      }
      case "resolveComponentMetadata": {
        validateInput(arguments, classes, ComponentElementDeclaration.class, Boolean.class);
        final ComponentElementDeclaration<?> componentElementDeclaration = serializer.deserialize(arguments[0]);
        final boolean ignoreCache = serializer.deserialize(arguments[1]);
        return serializer.serialize(resolveComponentMetadata(componentElementDeclaration, ignoreCache));
      }
      case "getSampleData": {
        validateInput(arguments, classes, ComponentElementDeclaration.class);
        final ComponentElementDeclaration<?> componentElementDeclaration = serializer.deserialize(arguments[0]);
        return serializer.serialize(getSampleData(componentElementDeclaration));
      }
      case "testConnection": {
        validateInput(arguments, classes, String.class);
        final String configName = serializer.deserialize(arguments[0]);
        return serializer.serialize(testConnection(configName));
      }
      case "getCacheService": {
        return getCacheService();
      }
      case "dispose": {
        dispose();
        return null;
      }
    }
    throw Command.methodNotFound(this.getClass(), methodName);
  }

  public static class DefaultDeclarationSessionBuilder implements Builder, Command {

    private Serializer serializer;
    private RuntimeToolingService runtimeToolingService;

    private List<ConfigurationElementDeclaration> configurationElementDeclarations = new ArrayList<>();
    private List<TopLevelParameterDeclaration> globalParameterDeclarations = new ArrayList<>();
    private List<Dependency> dependencies = new ArrayList<>();

    private Map<String, String> sessionProperties = emptyMap();

    private MavenClient mavenClient;
    private MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
    private Map<String, ?> cacheStorage;
    private ExpressionLanguage expressionLanguage;

    public DefaultDeclarationSessionBuilder(Serializer serializer, RuntimeToolingService runtimeToolingService,
                                            MavenClient mavenClient,
                                            MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                                            ExpressionLanguage expressionLanguage) {
      this.serializer = serializer;
      this.runtimeToolingService = runtimeToolingService;
      this.mavenClient = mavenClient;
      this.muleRuntimeExtensionModelProvider = muleRuntimeExtensionModelProvider;
      this.expressionLanguage = expressionLanguage;
    }

    @Override
    public Builder withConfigurationElementDeclarations(List<ConfigurationElementDeclaration> configurationElementDeclarations) {
      requireNonNull(configurationElementDeclarations, "configurationElementDeclarations cannot be null");
      this.configurationElementDeclarations.addAll(configurationElementDeclarations);
      return this;
    }

    @Override
    public Builder withGlobalParameters(List<TopLevelParameterDeclaration> globalParameterDeclarations) {
      requireNonNull(globalParameterDeclarations, "globalParameterDeclarations cannot be null");
      this.globalParameterDeclarations.addAll(globalParameterDeclarations);
      return this;
    }

    @Override
    public Builder withSessionProperties(Map<String, String> sessionProperties) {
      this.sessionProperties = sessionProperties;
      return this;
    }

    @Override
    public Builder withDependency(Dependency dependency) {
      requireNonNull(dependency, "dependency cannot be null");
      this.dependencies.add(dependency);
      return this;
    }

    @Override
    public Builder withCacheStorage(CacheStorage cacheStorage) {
      // This method is not supposed to be called here because we need the storage wrapped as a map.
      throw new UnsupportedOperationException("Calling withCacheStorage on tooling client is not allowed. It must be called through the API.");
    }

    public Builder internalWithCacheStorage(Map<String, ?> cacheStorage) {
      requireNonNull(cacheStorage, "Cache storage cannot be null");
      this.cacheStorage = new MDCMapDecorator(cacheStorage);
      return this;
    }

    @Override
    public DeclarationSession build() {
      DeclarationSessionConfig declarationSessionConfig =
          new DeclarationSessionConfig(configurationElementDeclarations, globalParameterDeclarations, dependencies,
                                       sessionProperties);
      ExtensionModelProvider extensionModelProvider = new ExtensionModelProvider(mavenClient,
                                                                                 muleRuntimeExtensionModelProvider,
                                                                                 dependencies);

      ConfigurationDeclarationProvider configurationDeclarationProvider =
          new ConfigurationDeclarationProvider(configurationElementDeclarations);

      ToolingCacheIdGenerator<ElementDeclaration> cacheIdGenerator = null;
      ComponentLocator<ElementDeclaration> sessionLocator = null;
      DeclarationValueProviderCache valueProviderCache = noOpDeclarationCache();
      DeclarationMetadataCache metadataCache = noOpDeclarationCache();
      if (cacheStorage != null) {
        DslResolvingContext dslResolvingContext = DslResolvingContext.getDefault(extensionModelProvider.getAll());
        sessionLocator = new DeclarationSessionComponentLocator(configurationElementDeclarations,
                                                                globalParameterDeclarations);
        cacheIdGenerator = new DeclarationToolingMetadataCacheIdGenerator(
                                                                          new DeclarationBasedMetadataCacheIdGenerator(dslResolvingContext,
                                                                                                                       sessionLocator),
                                                                          new DeclarationToolingMetadataCacheIdGenerator.DeclarationBasedValueProviderCacheIdGeneratorAdapter(dslResolvingContext,
                                                                                                                                                                              sessionLocator));

        valueProviderCache = new DefaultDeclarationValueProviderCache(cacheIdGenerator,
                                                                      (Map<String, ValueResult>) cacheStorage);
        metadataCache = new DefaultDeclarationMetadataCache(cacheIdGenerator,
                                                            (Map<String, MetadataType>) cacheStorage,
                                                            (Map<String, org.mule.runtime.api.metadata.resolving.MetadataResult<MetadataKeysContainer>>) cacheStorage,
                                                            extensionModelProvider);
      }

      return new DefaultDeclarationSession(new RemoteDeclarationSessionProvider(extensionModelProvider,
                                                                                runtimeToolingService,
                                                                                declarationSessionConfig),
                                           serializer,
                                           extensionModelProvider,
                                           configurationDeclarationProvider,
                                           runtimeToolingService,
                                           valueProviderCache,
                                           metadataCache,
                                           new DefaultDeclarationSessionCacheService(
                                                                                     new DefaultCacheStorageSerializer(),
                                                                                     serializer,
                                                                                     cacheIdGenerator,
                                                                                     cacheStorage,
                                                                                     sessionLocator),
                                           expressionLanguage);
    }

    @Override
    public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
      switch (methodName) {
        case "withConfigurationElementDeclarations": {
          validateInput(arguments, classes, List.class);
          List<ConfigurationElementDeclaration> configurationElementDeclarations = serializer.deserialize(arguments[0]);
          return withConfigurationElementDeclarations(configurationElementDeclarations);
        }
        case "withGlobalParameters": {
          validateInput(arguments, classes, List.class);
          List<TopLevelParameterDeclaration> globalParameterDeclarations = serializer.deserialize(arguments[0]);
          return withGlobalParameters(globalParameterDeclarations);
        }
        case "withSessionProperties": {
          validateInput(arguments, classes, Map.class);
          Map<String, String> sessionProperties = serializer.deserialize(arguments[0]);
          return withSessionProperties(sessionProperties);
        }
        case "withDependency": {
          validateInput(arguments, classes, Dependency.class);
          final Dependency dependency = serializer.deserialize(arguments[0]);
          return withDependency(dependency);
        }
        case "build": {
          return build();
        }
      }
      throw Command.methodNotFound(this.getClass(), methodName);
    }
  }

}
