/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session.mediator;

import static java.lang.String.format;
import static org.mule.runtime.module.extension.internal.loader.utils.FieldValueProviderNameUtils.getParameterName;
import static org.mule.tooling.client.internal.session.validation.SessionCallValidator.INVALID_PARAMETER;

import org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.runtime.module.extension.internal.loader.utils.FieldValueProviderNameUtils;
import org.mule.tooling.client.api.value.resolver.ValueResolverFailure;
import org.mule.tooling.client.api.value.resolver.ValueResolverResult;
import org.mule.tooling.client.internal.session.ConfigurationDeclarationProvider;
import org.mule.tooling.client.internal.session.mediator.resolver.ValuesResolverFactory;
import org.mule.tooling.client.internal.session.validation.ConnectionAndConfigurationValidator;
import org.mule.tooling.client.internal.session.validation.ParameterExistsValidator;
import org.mule.tooling.client.internal.session.validation.SessionCallValidationException;
import org.mule.tooling.client.internal.session.validation.SessionCallValidator;
import org.mule.tooling.client.internal.session.validation.ValueProviderActingParametersValidator;

public class FieldValuesResolverModelMediator {

  private final ConfigurationDeclarationProvider configurationDeclarationProvider;
  private final SessionCallValidator sessionCallValidator;
  private final ParameterizedElementDeclaration parameterizedElementDeclaration;
  private final String parameterName;

  private final String targetSelector;
  private final ValueResolverResult failure;

  private Reference<ParameterizedModel> parameterizedModel = new Reference<>();
  private Reference<ParameterModel> parameterModel = new Reference<>();
  private FieldValueProviderModel fieldValueProviderModel;

  public FieldValuesResolverModelMediator(ConfigurationDeclarationProvider configurationDeclarationProvider,
                                          ParameterizedElementDeclaration elementDeclaration,
                                          SessionCallValidator sessionCallValidator,
                                          String parameterName,
                                          String targetSelector) {
    this.configurationDeclarationProvider = configurationDeclarationProvider;
    this.parameterizedElementDeclaration = elementDeclaration;
    this.parameterName = parameterName;
    this.sessionCallValidator = sessionCallValidator;
    this.targetSelector = targetSelector;

    this.failure = validateModel();
  }

  public ValueResolverResult resolve(ValuesResolverFactory valuesResolverFactory, boolean ignoreCache) {
    if (failure != null) {
      return failure;
    }
    return valuesResolverFactory.createFieldValueProviderResolver(
                                                                  parameterizedElementDeclaration, fieldValueProviderModel,
                                                                  ignoreCache)
        .resolve(parameterizedModel.get(), parameterModel.get(), parameterName);
  }

  private ValueResolverResult validateModel() {
    try {
      final Reference<Boolean> requiresConfig = new Reference<>(false);
      final Reference<Boolean> requiresConnection = new Reference<>(false);

      sessionCallValidator.validateComponent(parameterizedElementDeclaration,
                                             c -> {
                                               this.parameterizedModel.set(c.getParameterizedModel());

                                               ParameterExistsValidator parameterExistsValidator =
                                                   new ParameterExistsValidator(parameterName);
                                               parameterExistsValidator.validate(c);
                                               this.parameterModel.set(parameterExistsValidator.getParameterModel());

                                               if (!parameterModel.get().getFieldValueProviderModels().isEmpty()) {
                                                 this.fieldValueProviderModel =
                                                     parameterModel
                                                         .get()
                                                         .getFieldValueProviderModels()
                                                         .stream()
                                                         .filter(m -> getParameterName(m).equals(parameterName))
                                                         .filter(m -> m.getTargetSelector().equals(targetSelector))
                                                         .findAny()
                                                         .orElseThrow(() -> new SessionCallValidationException(
                                                                                                               format("Parameter: '%s' doesn't have a field value provider model defined for targetSelector: '%s' in Model: '%s' for extension: '%s'",
                                                                                                                      parameterName,
                                                                                                                      targetSelector,
                                                                                                                      parameterizedElementDeclaration
                                                                                                                          .getName(),
                                                                                                                      parameterizedElementDeclaration
                                                                                                                          .getDeclaringExtension()),
                                                                                                               "ParameterModel not found",
                                                                                                               INVALID_PARAMETER));
                                                 requiresConfig.set(fieldValueProviderModel.requiresConfiguration());
                                                 requiresConnection.set(fieldValueProviderModel.requiresConnection());
                                               }
                                             },
                                             new ValueProviderActingParametersValidator(parameterName),
                                             new ConnectionAndConfigurationValidator(parameterizedElementDeclaration,
                                                                                     configurationDeclarationProvider,
                                                                                     requiresConnection::get,
                                                                                     requiresConfig::get,
                                                                                     () -> format("Parameter: '%s' with targetSelector: '%s' on element : '%s' for extension: '%s'",
                                                                                                  parameterName,
                                                                                                  targetSelector,
                                                                                                  parameterizedElementDeclaration
                                                                                                      .getName(),
                                                                                                  parameterizedElementDeclaration
                                                                                                      .getDeclaringExtension())));
    } catch (SessionCallValidationException e) {
      return ValueResolverResult.failure(new ValueResolverFailure(e.getMessage(), e.getReason(), e.getFailureCode()));
    }
    return null;
  }

}
